<?php
session_start();

// Validamos que exista una session y ademas que el cargo que exista sea igual a 2 (Profesional)
if(!isset($_SESSION['cargo']) || $_SESSION['cargo'] != 2){
    header('location: ../../index.php');
}

// Definir constante para permitir acceso al modal de derivación
define('USER_ACCESS', true);

// Modificar la consulta inicial (alrededor de la línea 20)
$refphysician = $_SESSION['nombre'];
$usuarioId = $_SESSION['id'];

// Credenciales
$username = "iset";
$password = "iset";

// Variables de ordenamiento (definir por defecto)
$sortColumn = isset($_GET['sort']) ? $_GET['sort'] : 'StudyDate';
$sortOrder = isset($_GET['order']) ? $_GET['order'] : 'desc';

// Datos de la solicitud
$url = "https://iset.portalpaciente.com.ar/api/find"; //se debe activar pm2 con orthanc-proxy-api /Descargas/nodejs/
$data = json_encode([
    "Level" => "Study",
    "Query" => [
        "ReferringPhysicianName" => $_SESSION['nombre']
    ],
    "Expand" => true
]);

// Inicializar cURL
$ch = curl_init();

// Configurar opciones de cURL
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, 1);
curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    "Content-Type: application/json"
]);

// Deshabilitar verificación SSL (NO RECOMENDADO PARA PRODUCCIÓN)
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);

// Ejecutar la solicitud y obtener la respuesta
$response = curl_exec($ch);

if (curl_errno($ch)) {
    // Log error instead of echoing
    error_log('Error en la solicitud cURL: ' . curl_error($ch));
    $jsonData = null;
} else {
    // Decodificar la respuesta JSON
    $jsonData = json_decode($response, true);

    if ($jsonData === null) {
        // Log error instead of echoing
        error_log("Error al decodificar el JSON");
        $jsonData = [];
    } else {
        // Verificar si hay al menos un estudio en los resultados
        if (count($jsonData) > 0) {
            // Las variables de ordenamiento ya están definidas arriba

            // Crear un array para ordenamiento
            $sortArray = array();
            foreach ($jsonData as $key => $study) {
                switch ($sortColumn) {
                    case 'PatientName':
                        $sortArray[$key] = $study['PatientMainDicomTags']['PatientName'];
                        break;
                    case 'PatientID':
                        $sortArray[$key] = $study['PatientMainDicomTags']['PatientID'];
                        break;
                    case 'StudyDescription':
                        $sortArray[$key] = $study['MainDicomTags']['StudyDescription'];
                        break;
                    case 'StudyDate':
                        $sortArray[$key] = $study['MainDicomTags']['StudyDate'];
                        break;                    
                    case 'Modality':
                        $sortArray[$key] = $study['MainDicomTags']['Modality'];
                        break;                    
                    case 'ID':
                        $sortArray[$key] = $study['ID'];
                        break;
                    default:
                        $sortArray[$key] = $study['MainDicomTags']['StudyDate'];
                }
            }

            // Ordenar el array
            if ($sortOrder === 'asc') {
                asort($sortArray);
            } else {
                arsort($sortArray);
            }

            // Reordenar $jsonData basado en $sortArray
            $sortedData = array();
            foreach ($sortArray as $key => $value) {
                $sortedData[] = $jsonData[$key];
            }
            $jsonData = $sortedData;
        }
    }
}

// Cerrar cURL
curl_close($ch);

// Después de obtener $jsonData, agregar estudios derivados
require_once('../../model/conexion.php');
$conexionDerivados = new Conexion();

try {
    $conexionDerivados->conectar();
    
    // Obtener derivaciones recibidas con ID y fecha para ordenamiento
    $consultaDerivacionesRecibidas = "SELECT d.id as derivacion_id, d.study_instance_uid, d.fecha_derivacion
                                      FROM derivaciones d 
                                      WHERE d.usuario_destino_id = $usuarioId AND d.activo = 1 
                                      ORDER BY d.fecha_derivacion DESC";
    $resultadoDerivacionesRecibidas = $conexionDerivados->query($consultaDerivacionesRecibidas);
    
    if (!$resultadoDerivacionesRecibidas) {
        error_log('Error en consulta de derivaciones recibidas para lista: ' . $conexionDerivados->error);
        $resultadoDerivacionesRecibidas = null;
    }
    
    $jsonDataDerivados = [];
    if ($resultadoDerivacionesRecibidas) {
        while ($row = $resultadoDerivacionesRecibidas->fetch_assoc()) {
            $studyInstanceUID = $row['study_instance_uid'];
            $derivacionId = $row['derivacion_id'];
            $fechaDerivacion = $row['fecha_derivacion'];
            
            // Usar la función obtenerInfoEstudio() que ya funciona bien
            $infoEstudio = obtenerInfoEstudio($studyInstanceUID);
            
            if ($infoEstudio) {
                // Crear estructura similar a la de Orthanc pero con información procesada
                $studyData = [
                    'MainDicomTags' => [
                        'StudyInstanceUID' => $studyInstanceUID,
                        'StudyDate' => $infoEstudio['studyDate'],
                        'StudyDescription' => $infoEstudio['studyDescription'],
                        'StudyID' => $infoEstudio['studyID']
                    ],
                    'PatientMainDicomTags' => [
                        'PatientName' => $infoEstudio['patientName'],
                        'PatientID' => $infoEstudio['patientID'] // Agregar PatientID
                    ],
                    'es_derivado' => true, // Marcar como derivado
                    'derivacion_id' => $derivacionId, // ID de la derivación
                    'fecha_derivacion' => $fechaDerivacion // Fecha de derivación para ordenamiento
                ];
                
                $jsonDataDerivados[] = $studyData;
            }
        }
    }
    
    // Combinar estudios derivados con estudios propios
    if (count($jsonDataDerivados) > 0) {
        // Si no hay estudios propios, inicializar array
        if ($jsonData === null || count($jsonData) == 0) {
            $jsonData = $jsonDataDerivados;
        } else {
            // Combinar con estudios propios
            $jsonData = array_merge($jsonData, $jsonDataDerivados);
        }
    }
    
    // Ordenar la lista combinada por fecha de estudio (más reciente primero)
    if ($jsonData !== null && count($jsonData) > 0) {
        usort($jsonData, function($a, $b) {
            // Para todos los estudios (derivados y propios), usar StudyDate
            $fechaA = $a['MainDicomTags']['StudyDate'];
            $fechaB = $b['MainDicomTags']['StudyDate'];
            
            // Ordenar por fecha descendente (más reciente primero)
            return strtotime($fechaB) - strtotime($fechaA);
        });
    }
    
    // Obtener información de derivaciones para mostrar en el panel
    $consultaDerivacionesEnviadas = "SELECT d.id, d.study_instance_uid, d.usuario_origen_id, d.usuario_destino_id, 
                                    d.fecha_derivacion, d.activo, d.comentario, 
                                    COALESCE(d.vista, 0) as vista, COALESCE(d.descargada, 0) as descargada,
                                    d.fecha_vista, d.fecha_descarga,
                                    u.nombre as usuario_destino_nombre, u.email as usuario_destino_email 
                                    FROM derivaciones d 
                                    JOIN usuarios u ON d.usuario_destino_id = u.id 
                                    WHERE d.usuario_origen_id = $usuarioId AND d.activo = 1 
                                    ORDER BY d.fecha_derivacion DESC";
    $derivacionesEnviadas = $conexionDerivados->query($consultaDerivacionesEnviadas);
    
    if (!$derivacionesEnviadas) {
        error_log('Error en consulta de derivaciones enviadas: ' . $conexionDerivados->error);
        $derivacionesEnviadas = null;
    }
    
    $consultaDerivacionesRecibidas = "SELECT d.id, d.study_instance_uid, d.usuario_origen_id, d.usuario_destino_id, 
                                     d.fecha_derivacion, d.activo, d.comentario, 
                                     COALESCE(d.vista, 0) as vista, COALESCE(d.descargada, 0) as descargada,
                                     d.fecha_vista, d.fecha_descarga,
                                     u.nombre as usuario_origen_nombre, u.email as usuario_origen_email 
                                     FROM derivaciones d 
                                     JOIN usuarios u ON d.usuario_origen_id = u.id 
                                     WHERE d.usuario_destino_id = $usuarioId AND d.activo = 1 
                                     AND (d.oculta_por_usuario IS NULL OR JSON_CONTAINS(d.oculta_por_usuario, CAST($usuarioId AS JSON)) = 0)
                                     ORDER BY d.fecha_derivacion DESC";
    $derivacionesRecibidas = $conexionDerivados->query($consultaDerivacionesRecibidas);
    
    if (!$derivacionesRecibidas) {
        error_log('Error en consulta de derivaciones recibidas: ' . $conexionDerivados->error);
        $derivacionesRecibidas = null;
    }
    
    // Obtener derivaciones recibidas OCULTAS (para vista alternada)
    $consultaDerivacionesOcultas = "SELECT d.id, d.study_instance_uid, d.usuario_origen_id, d.usuario_destino_id, 
                                   d.fecha_derivacion, d.activo, d.comentario, 
                                   COALESCE(d.vista, 0) as vista, COALESCE(d.descargada, 0) as descargada,
                                   d.fecha_vista, d.fecha_descarga,
                                   u.nombre as usuario_origen_nombre, u.email as usuario_origen_email 
                                   FROM derivaciones d 
                                   JOIN usuarios u ON d.usuario_origen_id = u.id 
                                   WHERE d.usuario_destino_id = $usuarioId AND d.activo = 1 
                                   AND d.oculta_por_usuario IS NOT NULL AND JSON_CONTAINS(d.oculta_por_usuario, CAST($usuarioId AS JSON)) = 1
                                   ORDER BY d.fecha_derivacion DESC";
    $derivacionesOcultas = $conexionDerivados->query($consultaDerivacionesOcultas);
    
    if (!$derivacionesOcultas) {
        error_log('Error en consulta de derivaciones ocultas: ' . $conexionDerivados->error);
        $derivacionesOcultas = null;
    }
    
} catch (Exception $e) {
    error_log('Error en conexión a base de datos: ' . $e->getMessage());
    $derivacionesEnviadas = null;
    $derivacionesRecibidas = null;
}

// Función para obtener información del estudio
function obtenerInfoEstudio($studyInstanceUID) {
    global $conexionDerivados;
    
    $url = "https://iset.portalpaciente.com.ar/api/find";
    $data = json_encode([
        "Level" => "Study",
        "Query" => [
            "StudyInstanceUID" => $studyInstanceUID
        ],
        "Expand" => true
    ]);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: application/json"]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        error_log('Error cURL en obtenerInfoEstudio: ' . curl_error($ch));
        curl_close($ch);
        return null;
    }
    
    curl_close($ch);
    
    $jsonData = json_decode($response, true);
    
    if ($jsonData && count($jsonData) > 0) {
        $study = $jsonData[0];
        return [
            'patientName' => str_replace("^", " ", $study['PatientMainDicomTags']['PatientName'] ?? ''),
            'patientID' => $study['PatientMainDicomTags']['PatientID'] ?? '', // Agregar PatientID
            'studyDate' => $study['MainDicomTags']['StudyDate'] ?? '',
            'studyDescription' => str_replace("^", " ", $study['MainDicomTags']['StudyDescription'] ?? ''),
            'modality' => $study['MainDicomTags']['Modality'] ?? 'UNKNOWN',
            'studyID' => $study['ID'] ?? '' // Agregar el ID del estudio
        ];
    }
    
    return null;
}

// Función auxiliar para construir el nombre del archivo con formato: nombrepaciente_fechaestudio_modalidad.zip
function construirNombreArchivo($patientName, $studyDate, $modality = 'UNKNOWN') {
    // Limpiar nombre del paciente
    $safePatientName = preg_replace('/[^a-zA-Z0-9_-]/', '_', $patientName);
    
    // Formatear fecha: convertir YYYYMMDD a YYYYMMDD (mantener formato DICOM)
    $fechaFormateada = $studyDate;
    if (strlen($studyDate) == 8) {
        // Ya está en formato YYYYMMDD
        $fechaFormateada = $studyDate;
    } elseif (strtotime($studyDate)) {
        // Convertir a YYYYMMDD si viene en otro formato
        $fechaFormateada = date('Ymd', strtotime($studyDate));
    }
    
    // Limpiar modalidad
    $safeModality = preg_replace('/[^a-zA-Z0-9_-]/', '_', $modality);
    
    // Construir nombre: nombrepaciente_fechaestudio_modalidad.zip
    return "{$safePatientName}_{$fechaFormateada}_{$safeModality}.zip";
}

?>

<!DOCTYPE html>
             <html lang="es">
             <head>
                 <meta charset="utf-8">
                 <meta name="viewport" content="width=device-width, initial-scale=1">
                 <title>ISET-PROFESIONALES</title>
                 <link rel="stylesheet" href="../../css/bootstrap.min.css">
                 <link rel="stylesheet" href="../../css/font-awesome.min.css">
                 <link rel="stylesheet" href="../../css/sweetalert.css">
                 <link rel="stylesheet" href="../../css/style.css">
                 <link rel="stylesheet" href="../../css/user-styles.css">
                 
                 <!-- jQuery -->
                 <script src="../../js/jquery.js"></script>
                 
                 <!-- Bootstrap JS -->
                 <script src="../../js/bootstrap.min.js"></script>
                 
                 <!-- SweetAlert2 -->
                 <script src="../../js/sweetalert.min.js"></script>
                 <style>
                     body {
                         font-family: Arial, sans-serif;
                         line-height: 1.6;
                         color: #333;
                         max-width: 1200px;
                         margin: 0 auto;
                         padding: 20px;
                         background-color: #f4f4f4;
                     }
                     h1, h3 {
                         color: #2c3e50;
                     }
                     .welcome-message {
                         background-color: #3498db;
                         color: white;
                         padding: 10px;
                         border-radius: 5px;
                         margin-bottom: 20px;
                     }
                     .logout-button {
                         background-color: #e74c3c;
                         color: white;
                         border: none;
                         padding: 10px 15px;
                         border-radius: 5px;
                         cursor: pointer;
                         transition: background-color 0.3s;
                     }
                     .logout-button:hover {
                         background-color: #c0392b;
                     }
                     .header-buttons {
                         display: flex;
                         justify-content: space-between;
                         align-items: center;
                         margin-bottom: 20px;
                     }
                     .download-app-button {
                         background-color: #27ae60;
                         color: white;
                         border: none;
                         padding: 10px 15px;
                         border-radius: 5px;
                         cursor: pointer;
                         text-decoration: none;
                         display: inline-block;
                         margin-right: 10px;
                         transition: background-color 0.3s ease;
                     }
                     .download-app-button:hover {
                         background-color: #229954;
                     }
                     
                     .download-bluesky-button {
                         background-color: #8e44ad;
                         color: white;
                         border: none;
                         padding: 10px 15px;
                         border-radius: 5px;
                         cursor: pointer;
                         text-decoration: none;
                         display: inline-block;
                         transition: background-color 0.3s ease;
                     }
                     .download-bluesky-button:hover {
                         background-color: #7d3c98;
                     }
                     
                     .download-buttons-container {
                         display: flex;
                         gap: 10px;
                         flex-wrap: wrap;
                     }
                     
                     /* Estilos para paneles plegables */
                     .derivation-panel {
                         transition: all 0.3s ease;
                     }
                     
                     .derivation-title {
                         transition: background-color 0.3s ease;
                     }
                     
                     .derivation-title:hover {
                         background-color: #e9ecef;
                         border-radius: 4px;
                         padding: 4px;
                         margin: -4px;
                     }
                     
                     /* Animación de pulso para notificaciones */
                     @keyframes pulse {
                         0% { transform: scale(1); }
                         50% { transform: scale(1.2); }
                         100% { transform: scale(1); }
                     }
                     
                     @keyframes bounce {
                         0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
                         40% { transform: translateY(-5px); }
                         60% { transform: translateY(-3px); }
                     }
                     
                     /* Estilos para notificaciones */
                     .notification-badge {
                         background: #ff4444;
                         color: white;
                         border-radius: 50%;
                         padding: 2px 6px;
                         font-size: 10px;
                         animation: bounce 2s infinite;
                     }
                     
                     /* Sistema de notificaciones personalizado */
                     .custom-notification {
                         position: fixed;
                         top: 20px;
                         right: 20px;
                         padding: 15px 20px;
                         border-radius: 8px;
                         color: white;
                         font-weight: 500;
                         z-index: 10000;
                         max-width: 400px;
                         box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                         transform: translateX(100%);
                         transition: transform 0.3s ease;
                         display: flex;
                         align-items: center;
                         gap: 10px;
                     }
                     
                     .custom-notification.show {
                         transform: translateX(0);
                     }
                     
                     .custom-notification.success {
                         background: linear-gradient(135deg, #28a745, #20c997);
                         border-left: 4px solid #1e7e34;
                     }
                     
                     .custom-notification.error {
                         background: linear-gradient(135deg, #dc3545, #e74c3c);
                         border-left: 4px solid #c82333;
                     }
                     
                     .custom-notification.warning {
                         background: linear-gradient(135deg, #ffc107, #fd7e14);
                         border-left: 4px solid #e0a800;
                         color: #212529;
                     }
                     
                     .custom-notification.info {
                         background: linear-gradient(135deg, #17a2b8, #6f42c1);
                         border-left: 4px solid #138496;
                     }
                     
                     .custom-notification .icon {
                         font-size: 18px;
                         flex-shrink: 0;
                     }
                     
                     .custom-notification .content {
                         flex-grow: 1;
                     }
                     
                     .custom-notification .title {
                         font-weight: bold;
                         margin-bottom: 2px;
                         font-size: 14px;
                     }
                     
                     .custom-notification .message {
                         font-size: 13px;
                         opacity: 0.9;
                     }
                     
                     .custom-notification .close-btn {
                         background: none;
                         border: none;
                         color: inherit;
                         font-size: 16px;
                         cursor: pointer;
                         opacity: 0.7;
                         transition: opacity 0.2s;
                         flex-shrink: 0;
                     }
                     
                     .custom-notification .close-btn:hover {
                         opacity: 1;
                     }
                     
                     /* Estilos para iconos de estado */
                     .panel-icon {
                         transition: transform 0.3s ease;
                     }
                     
                     .panel-icon:hover {
                         transform: scale(1.1);
                     }
                     
                     /* Estilos responsivos */
                     @media (max-width: 768px) {
                         .derivations-container {
                             flex-direction: column;
                         }
                         
                         .derivation-panel {
                             min-width: 100%;
                         }
                         
                         .download-buttons-container {
                             flex-direction: column;
                             gap: 8px;
                         }
                         
                         .download-app-button,
                         .download-bluesky-button {
                             margin-right: 0;
                             text-align: center;
                             font-size: 14px;
                             padding: 12px 15px;
                         }
                         
                         .header-buttons {
                             flex-direction: column;
                             gap: 15px;
                         }
                     }
                     
                     /* Botón flotante para volver al inicio */
                     .scroll-to-top {
                         position: fixed;
                         bottom: 30px;
                         right: 30px;
                         width: 50px;
                         height: 50px;
                         background: linear-gradient(135deg, #3498db, #2980b9);
                         color: white;
                         border: none;
                         border-radius: 50%;
                         cursor: pointer;
                         box-shadow: 0 4px 12px rgba(52, 152, 219, 0.3);
                         z-index: 1000;
                         opacity: 0;
                         visibility: hidden;
                         transform: translateY(20px);
                         transition: all 0.3s ease;
                         display: flex;
                         align-items: center;
                         justify-content: center;
                         font-size: 18px;
                     }
                     
                     .scroll-to-top.show {
                         opacity: 1;
                         visibility: visible;
                         transform: translateY(0);
                     }
                     
                     .scroll-to-top:hover {
                         background: linear-gradient(135deg, #2980b9, #1f5f8b);
                         transform: translateY(-2px);
                         box-shadow: 0 6px 16px rgba(52, 152, 219, 0.4);
                     }
                     
                     .scroll-to-top:active {
                         transform: translateY(0);
                     }
                     
                     /* Animación de pulso suave */
                     @keyframes gentle-pulse {
                         0%, 100% { transform: scale(1); }
                         50% { transform: scale(1.05); }
                     }
                     
                     .scroll-to-top.show {
                         animation: gentle-pulse 2s ease-in-out infinite;
                     }
                     
                     /* Responsive para el botón flotante */
                     @media (max-width: 768px) {
                         .scroll-to-top {
                             bottom: 20px;
                             right: 20px;
                             width: 45px;
                             height: 45px;
                             font-size: 16px;
                         }
                     }
                 </style>
                 <script>
                 function downloadAndRename(url, patientName, buttonElement, derivacionId) {
                     buttonElement.disabled = true;
        buttonElement.innerHTML = "<span class='spinner'></span>Descargando...";
                     buttonElement.classList.remove("error", "success");
 
                     fetch(url)
                         .then(response => {
                             if (!response.ok) {
                                 throw new Error("HTTP error " + response.status);
                             }
                             return response.blob();
                         })
                         .then(blob => {
                             const url = window.URL.createObjectURL(blob);
                             const a = document.createElement("a");
                             a.style.display = "none";
                             a.href = url;
                             a.download = patientName + ".zip";
                             document.body.appendChild(a);
                             a.click();
                             window.URL.revokeObjectURL(url);
                             
                             buttonElement.classList.add("success");
                             buttonElement.innerHTML = "Descarga exitosa";
                             setTimeout(() => {
                                 buttonElement.classList.remove("success");
                                 buttonElement.innerHTML = "Descargar";
                             }, 3000);

                             // Marcar como descargada y vista en el backend
                             Promise.all([
                                 // Marcar como descargada
                                 fetch('../../controller/derivacionController.php', {
                                     method: 'POST',
                                     headers: {
                                         'Content-Type': 'application/x-www-form-urlencoded',
                                     },
                                     body: `action=marcar_descargada&derivacion_id=${derivacionId}`
                                 }),
                                 // Marcar como vista
                                 fetch('../../controller/derivacionController.php', {
                                     method: 'POST',
                                     headers: {
                                         'Content-Type': 'application/x-www-form-urlencoded',
                                     },
                                     body: `action=marcar_vista&derivacion_id=${derivacionId}`
                                 })
                             ])
                             .then(responses => Promise.all(responses.map(r => r.json())))
                             .then(results => {
                                 const [descargadaResult, vistaResult] = results;
                                 
                                 if (descargadaResult.success) {
                                     console.log('Derivación marcada como descargada:', derivacionId);
                                 } else {
                                     console.error('Error al marcar como descargada:', descargadaResult.message);
                                 }
                                 
                                 if (vistaResult.success) {
                                     console.log('Derivación marcada como vista:', derivacionId);
                                 } else {
                                     console.error('Error al marcar como vista:', vistaResult.message);
                                 }
                                 
                                 // Refrescar el panel de derivaciones para actualizar el estado visual
                                 setTimeout(() => {
                                     refrescarDerivaciones();
                                 }, 1000);
                             })
                             .catch(error => {
                                 console.error('Error al marcar derivación:', error);
                             });
                         })
                         .catch(error => {
                             console.error("Error en la descarga:", error);
                             
                             if (typeof Swal !== 'undefined') {
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Error en la descarga',
                                    text: 'Hubo un error en la descarga. Por favor, inténtelo de nuevo.',
                                    confirmButtonText: 'Aceptar',
                                    confirmButtonColor: '#dc3545'
                                });
                            } else {
                                showError('Error en la descarga', 'Hubo un error en la descarga. Por favor, inténtelo de nuevo.');
                            }
                            
                            buttonElement.classList.add("error");
                            buttonElement.innerHTML = "Error en descarga";
                            setTimeout(() => {
                                buttonElement.classList.remove("error");
                                buttonElement.innerHTML = "Descargar";
                            }, 3000);
                        })
                         .finally(() => {
                             buttonElement.disabled = false;
                         });
                 }
 
                 function searchTable() {
                    var input, filter, table, tr, td, i, txtValue;
                    input = document.getElementById("searchInput");
                    filter = input.value.toUpperCase();
                    table = document.querySelector("table");
                    tr = table.getElementsByTagName("tr");
                    var visibleCount = 0;
                    for (i = 0; i < tr.length; i++) {
                        td = tr[i].getElementsByTagName("td");
                        if (td.length > 0) { // Asegurarse de que no es una fila de encabezado
                            var rowVisible = false;
                            for (var j = 0; j < td.length; j++) {
                                if (td[j]) {
                                    txtValue = td[j].textContent || td[j].innerText;
                                    if (txtValue.toUpperCase().indexOf(filter) > -1) {
                                        rowVisible = true;
                                        break;
                                    }
                                }
                            }
                            if (rowVisible) {
                                tr[i].style.display = "";
                                visibleCount++;
                            } else {
                                tr[i].style.display = "none";
                            }
                        }
                    }
                    // Actualizar el contador de estudios
                    document.getElementById("studyCount").textContent = visibleCount;
                }
                 </script>
             </head>
            <body>
                <div class="logo-container">
                    <img src="isettacselogotransparent.png" alt="Logo ISET" class="logo">
                </div>

                <div class="welcome-message">
        <h1>Hola estimado colega <?php echo strtoupper($_SESSION['nombre']); ?></h1>
                    <p>Estos son los estudios de tus pacientes, solicitados a ISET:</p>
                </div>
                
                <div class="header-buttons">
                    <a href="../../controller/cerrarSesion.php">
                        <button class="logout-button" type="button">Cerrar sesión</button>
                    </a>
                    <div class="download-buttons-container">
                        <a href="../../controller/downloadApp.php" class="download-app-button">
                            📱 Descargar OnDemand 3D Communicator
                        </a>
                        <a href="../../controller/downloadBlueSky.php" class="download-bluesky-button">
                            🧠 Descargar BlueSkyPlan
                        </a>
                    </div>
                </div>

                <!-- Panel de Derivaciones -->
            <div class="derivations-container" style="display: flex; gap: 15px; margin-bottom: 20px; flex-wrap: wrap;">
                <!-- Derivaciones Enviadas -->
                <div class="derivation-panel" style="flex: 1; min-width: 300px; background: #f8f9fa; border-radius: 8px; padding: 15px; border: 1px solid #dee2e6;">
                    <div class="derivation-title" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; font-weight: bold; color: #495057; cursor: pointer;" onclick="togglePanel('enviadas')">
                        <span>📤 Derivaciones Enviadas <span id="contador-enviadas" style="background: #007bff; color: white; padding: 2px 8px; border-radius: 12px; font-size: 11px; font-weight: normal;"><?php echo $derivacionesEnviadas ? $derivacionesEnviadas->num_rows : 0; ?></span></span>
                        <div style="display: flex; align-items: center; gap: 8px;">
                            <span id="icono-enviadas" style="font-size: 16px;">📤</span>
                            <button onclick="event.stopPropagation(); refrescarManual()" style="background: none; border: none; color: #6c757d; cursor: pointer; font-size: 14px;" title="Refrescar">🔄</button>
                        </div>
                    </div>
                    <div id="contenido-derivaciones-enviadas" style="max-height: 0px; overflow: hidden; transition: all 0.3s ease;">
                        <?php if ($derivacionesEnviadas && $derivacionesEnviadas->num_rows > 0): ?>
                            <?php while ($derivacion = $derivacionesEnviadas->fetch_assoc()): ?>
                                <div class="derivation-item" style="background: white; padding: 8px; margin-bottom: 6px; border-radius: 4px; border: 1px solid #dee2e6; font-size: 11px;">
                                    <?php
                                    $infoEstudio = obtenerInfoEstudio($derivacion['study_instance_uid']);
                                    $patientName = $infoEstudio['patientName'] ?? 'No disponible';
                                    $studyDate = $infoEstudio['studyDate'] ?? 'No disponible';
                                    $studyDescription = $infoEstudio['studyDescription'] ?? 'No disponible';
                                    ?>
                                    <div style="font-weight: bold; color: #495057; margin-bottom: 2px;">👤 <?php echo htmlspecialchars($patientName); ?></div>
                                    <div style="color: #6c757d; margin-bottom: 2px;">📅 <?php echo $studyDate ? date('d/m/Y', strtotime($studyDate)) : 'No disponible'; ?> | <?php echo htmlspecialchars($studyDescription); ?></div>
                                    <div style="color: #6c757d; margin-bottom: 2px;">👤 Para: <?php echo htmlspecialchars($derivacion['usuario_destino_nombre']); ?></div>
                                    <div style="color: #6c757d; margin-bottom: 3px;">📅 Derivado: <?php echo date('d/m/Y H:i', strtotime($derivacion['fecha_derivacion'])); ?></div>
                                    <?php if (!empty($derivacion['comentario'])): ?>
                                        <div style="color: #28a745; font-style: italic; margin-bottom: 3px;">💬 <?php echo htmlspecialchars($derivacion['comentario']); ?></div>
                                    <?php endif; ?>
                                    <button onclick="desactivarDerivacion(<?php echo $derivacion['id']; ?>)" 
                                            style="background: #dc3545; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 10px; cursor: pointer;">
                                        ❌ Cancelar
                                    </button>
                                </div>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <p style="color: #6c757d; font-style: italic; font-size: 12px;">No has enviado derivaciones</p>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Derivaciones Recibidas -->
                <div class="derivation-panel" style="flex: 1; min-width: 300px; background: #f8f9fa; border-radius: 8px; padding: 15px; border: 1px solid #dee2e6;">
                    <div class="derivation-title" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; font-weight: bold; color: #495057; cursor: pointer;" onclick="togglePanel('recibidas')">
                        <span>📥 Derivaciones Recibidas <span id="contador-recibidas" style="background: #28a745; color: white; padding: 2px 8px; border-radius: 12px; font-size: 11px; font-weight: normal;"><?php echo $derivacionesRecibidas ? $derivacionesRecibidas->num_rows : 0; ?></span></span>
                        <div style="display: flex; align-items: center; gap: 8px;">
                            <span id="icono-recibidas" style="font-size: 16px;">📥</span>
                            <div id="notificacion-nuevas" style="display: none; background: #ff4444; color: white; padding: 2px 6px; border-radius: 10px; font-size: 10px; animation: pulse 1s infinite;">🆕</div>
                            <button onclick="event.stopPropagation(); refrescarManual()" style="background: none; border: none; color: #6c757d; cursor: pointer; font-size: 14px;" title="Refrescar">🔄</button>
                            <button id="btn-ver-ocultas" onclick="event.stopPropagation(); alternarVistaDerivaciones()" style="background: #6c757d; color: white; border: none; padding: 4px 8px; border-radius: 4px; font-size: 10px; cursor: pointer;" title="Ver derivaciones ocultas">👁️ Ver ocultas</button>
                            <button id="btn-ver-normales" onclick="event.stopPropagation(); alternarVistaDerivaciones()" style="background: #28a745; color: white; border: none; padding: 4px 8px; border-radius: 4px; font-size: 10px; cursor: pointer; display: none;" title="Ver derivaciones normales">📋 Ver normales</button>
                        </div>
                    </div>
                    <div id="contenido-derivaciones-recibidas" style="max-height: 0px; overflow: hidden; transition: all 0.3s ease;">
                        <?php if ($derivacionesRecibidas && $derivacionesRecibidas->num_rows > 0): ?>
                            <?php while ($derivacion = $derivacionesRecibidas->fetch_assoc()): ?>
                                <?php
                                $infoEstudio = obtenerInfoEstudio($derivacion['study_instance_uid']);
                                $patientName = $infoEstudio['patientName'] ?? 'No disponible';
                                $studyDate = $infoEstudio['studyDate'] ?? 'No disponible';
                                $studyDescription = $infoEstudio['studyDescription'] ?? 'No disponible';
                                
                                // Preparar URLs para descarga y visualización
                                $studyInstanceUID = $derivacion['study_instance_uid'];
                                $studyID = $infoEstudio['studyID'] ?? '';
                                $modality = $infoEstudio['modality'] ?? 'UNKNOWN';
                                $filename = construirNombreArchivo($patientName, $studyDate, $modality);
                                $downloadUrl = "https://iset.portalpaciente.com.ar/api/isetarc/{$studyInstanceUID}?accept=application/zip&filename=" . urlencode($filename);
                                
                                // Inicializar viewUrl con valor por defecto
                                $viewUrl = "view.php?study=" . urlencode($studyInstanceUID);
                                
                                if (stripos($studyDescription, 'Cone') !== false) {
                                    // Obtener el ID del estudio para volview
                                    $studyInfo = obtenerInfoEstudio($studyInstanceUID);
                                    $studyID = $studyInfo['studyID'] ?? '';
                                    if (!empty($studyID)) {
                                        $viewUrl = "https://iset.portalpaciente.com.ar/volview/index.html?names=[archive.zip]&urls=[../studies/{$studyID}/archive]";
                                    }
                                }
                                
                                // Usar los datos ya obtenidos en lugar de hacer consulta adicional
                                $haSidoVista = isset($derivacion['vista']) ? $derivacion['vista'] == 1 : false;
                                $haSidoDescargada = isset($derivacion['descargada']) ? $derivacion['descargada'] == 1 : false;
                                
                                // Determinar el estado y color
                                if ($haSidoVista || $haSidoDescargada) {
                                    $colorBorde = '#28a745'; // Verde para vista/descargada
                                    $colorFondo = '#d4edda';
                                    $indicadorEstado = '✅';
                                    $textoEstado = $haSidoVista ? 'Vista' : 'Descargada';
                                } else {
                                    $colorBorde = '#ffc107'; // Amarillo para nueva
                                    $colorFondo = '#fff3cd';
                                    $indicadorEstado = '🆕';
                                    $textoEstado = 'Nueva';
                                }
                                ?>
                                <div class="derivation-item" style="background: <?php echo $colorFondo; ?>; padding: 8px; margin-bottom: 6px; border-radius: 4px; border: 2px solid <?php echo $colorBorde; ?>; font-size: 11px; position: relative;">
                                    <!-- Indicador de estado -->
                                    <div style="position: absolute; top: 4px; right: 4px; background: <?php echo $colorBorde; ?>; color: white; padding: 1px 4px; border-radius: 3px; font-size: 9px; font-weight: bold;">
                                        <?php echo $indicadorEstado; ?> <?php echo $textoEstado; ?>
                                    </div>
                                    
                                    <div style="font-weight: bold; color: #495057; margin-bottom: 2px;">👤 <?php echo htmlspecialchars($patientName); ?></div>
                                    <div style="color: #6c757d; margin-bottom: 2px;">📅 <?php echo $studyDate ? date('d/m/Y', strtotime($studyDate)) : 'No disponible'; ?> | <?php echo htmlspecialchars($studyDescription); ?></div>
                                    <div style="color: #6c757d; margin-bottom: 2px;">👤 De: <?php echo htmlspecialchars($derivacion['usuario_origen_nombre']); ?></div>
                                    <div style="color: #6c757d; margin-bottom: 3px;">📅 Recibido: <?php echo date('d/m/Y H:i', strtotime($derivacion['fecha_derivacion'])); ?></div>
                                    <?php if (!empty($derivacion['comentario'])): ?>
                                        <div style="color: #28a745; font-style: italic; margin-bottom: 3px;">💬 <?php echo htmlspecialchars($derivacion['comentario']); ?></div>
                                    <?php endif; ?>
                                    
                                    <!-- Botones de acción para derivaciones recibidas -->
                                    <div style="display: flex; gap: 4px; margin-top: 4px;">
                                        <button onclick='downloadAndRename("<?php echo htmlspecialchars($downloadUrl); ?>", "<?php echo htmlspecialchars($safePatientName); ?>", this, <?php echo $derivacion['id']; ?>)' 
                                                style="background: #3498db; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 9px; cursor: pointer;">
                                            📥 Descargar
                                        </button>
                                        <a href="<?php echo htmlspecialchars($viewUrl); ?>" target='_blank' rel='noopener noreferrer' 
                                           onclick="marcarComoVista(<?php echo $derivacion['id']; ?>)"
                                           style="background: #3498db; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 9px; cursor: pointer; text-decoration: none; display: inline-block;">
                                            👁️ Ver
                                        </a>
                                        <button onclick="ocultarDerivacionRecibida(<?php echo $derivacion['id']; ?>)" 
                                                style="background: #28a745; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 9px; cursor: pointer;">
                                            ✅ Marcar como vista
                                        </button>
                                    </div>
                                </div>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <p style="color: #6c757d; font-style: italic; font-size: 12px;">No has recibido derivaciones</p>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Sección de Derivaciones Ocultas (inicialmente oculta) -->
                    <div id="contenido-derivaciones-ocultas" style="max-height: 0px; overflow-y: auto; transition: all 0.3s ease; display: none;">
                        <?php if ($derivacionesOcultas && $derivacionesOcultas->num_rows > 0): ?>
                            <?php while ($derivacion = $derivacionesOcultas->fetch_assoc()): ?>
                                <?php
                                $infoEstudio = obtenerInfoEstudio($derivacion['study_instance_uid']);
                                $patientName = $infoEstudio['patientName'] ?? 'No disponible';
                                $studyDate = $infoEstudio['studyDate'] ?? 'No disponible';
                                $studyDescription = $infoEstudio['studyDescription'] ?? 'No disponible';
                                $studyInstanceUID = $derivacion['study_instance_uid'];
                                
                                // Determinar estado y colores
                                $vista = $derivacion['vista'] ?? 0;
                                $descargada = $derivacion['descargada'] ?? 0;
                                
                                if ($vista && $descargada) {
                                    $colorFondo = '#d4edda';
                                    $colorBorde = '#28a745';
                                    $indicadorEstado = '✅';
                                    $textoEstado = 'Completa';
                                } elseif ($vista) {
                                    $colorFondo = '#fff3cd';
                                    $colorBorde = '#ffc107';
                                    $indicadorEstado = '👁️';
                                    $textoEstado = 'Vista';
                                } elseif ($descargada) {
                                    $colorFondo = '#cce5ff';
                                    $colorBorde = '#007bff';
                                    $indicadorEstado = '📥';
                                    $textoEstado = 'Descargada';
                                } else {
                                    $colorFondo = '#f8d7da';
                                    $colorBorde = '#dc3545';
                                    $indicadorEstado = '🆕';
                                    $textoEstado = 'Nueva';
                                }
                                
                                // URLs para descarga y visualización
                                $studyID = $infoEstudio['studyID'] ?? '';
                                $modality = $infoEstudio['modality'] ?? 'UNKNOWN';
                                $filename = construirNombreArchivo($patientName, $studyDate, $modality);
                                $downloadUrl = "https://iset.portalpaciente.com.ar/api/isetarc/{$studyInstanceUID}?accept=application/zip&filename=" . urlencode($filename);
                                $viewUrl = "https://iset.portalpaciente.com.ar/viewer/" . urlencode($derivacion['study_instance_uid']);
                                ?>
                                <div class="derivation-item" style="background: <?php echo $colorFondo; ?>; padding: 8px; margin-bottom: 6px; border-radius: 4px; border: 2px solid <?php echo $colorBorde; ?>; font-size: 11px; position: relative; opacity: 0.7;">
                                    <!-- Indicador de estado -->
                                    <div style="position: absolute; top: 4px; right: 4px; background: <?php echo $colorBorde; ?>; color: white; padding: 1px 4px; border-radius: 3px; font-size: 9px; font-weight: bold;">
                                        <?php echo $indicadorEstado; ?> <?php echo $textoEstado; ?>
                                    </div>
                                    
                                    <!-- Indicador de que está oculta -->
                                    <div style="position: absolute; bottom: 4px; right: 4px; background: #6c757d; color: white; padding: 1px 4px; border-radius: 3px; font-size: 9px; font-weight: bold;">
                                        👁️ OCULTA
                                    </div>
                                    
                                    <div style="font-weight: bold; color: #495057; margin-bottom: 2px;">👤 <?php echo htmlspecialchars($patientName); ?></div>
                                    <div style="color: #6c757d; margin-bottom: 2px;">📅 <?php echo $studyDate ? date('d/m/Y', strtotime($studyDate)) : 'No disponible'; ?> | <?php echo htmlspecialchars($studyDescription); ?></div>
                                    <div style="color: #6c757d; margin-bottom: 2px;">👤 De: <?php echo htmlspecialchars($derivacion['usuario_origen_nombre']); ?></div>
                                    <div style="color: #6c757d; margin-bottom: 3px;">📅 Recibido: <?php echo date('d/m/Y H:i', strtotime($derivacion['fecha_derivacion'])); ?></div>
                                    <?php if (!empty($derivacion['comentario'])): ?>
                                        <div style="color: #28a745; font-style: italic; margin-bottom: 3px;">💬 <?php echo htmlspecialchars($derivacion['comentario']); ?></div>
                                    <?php endif; ?>
                                    
                                    <!-- Botones de acción para derivaciones ocultas -->
                                    <div style="display: flex; gap: 4px; margin-top: 4px;">
                                        <button onclick='downloadAndRename("<?php echo htmlspecialchars($downloadUrl); ?>", "<?php echo htmlspecialchars($safePatientName); ?>", this, <?php echo $derivacion['id']; ?>)' 
                                                style="background: #3498db; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 9px; cursor: pointer;">
                                            📥 Descargar
                                        </button>
                                        <a href="<?php echo htmlspecialchars($viewUrl); ?>" target='_blank' rel='noopener noreferrer' 
                                           onclick="marcarComoVista(<?php echo $derivacion['id']; ?>)"
                                           style="background: #3498db; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 9px; cursor: pointer; text-decoration: none; display: inline-block;">
                                            👁️ Ver
                                        </a>
                                        <button onclick="mostrarDerivacionOculta(<?php echo $derivacion['id']; ?>)" 
                                                style="background: #17a2b8; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 9px; cursor: pointer;">
                                            🔄 Mostrar
                                        </button>
                                    </div>
                                </div>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <p style="color: #6c757d; font-style: italic; font-size: 12px;">No tienes derivaciones ocultas</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
                
    <h3>Estudios: <span id="studyCount"><?php echo $jsonData !== null ? count($jsonData) : 0; ?></span> encontrados 
        <button onclick="actualizarListaPacientes()" style="background: #007bff; color: white; border: none; padding: 4px 8px; border-radius: 4px; font-size: 12px; cursor: pointer; margin-left: 10px;" title="Actualizar lista de pacientes">🔄 Actualizar Lista</button>
    </h3>
    
    <?php if ($jsonData !== null && count($jsonData) > 0): ?>
                <input type="text" id="searchInput" onkeyup="searchTable()" placeholder="Buscar en la lista...">
                <table>
                    <thead>
                        <tr>
                    <th><a href="?sort=PatientName&order=<?php echo ($sortColumn == 'PatientName' && $sortOrder == 'asc' ? 'desc' : 'asc'); ?>">NOMBRE</a></th>
                    <th><a href="?sort=PatientID&order=<?php echo ($sortColumn == 'PatientID' && $sortOrder == 'asc' ? 'desc' : 'asc'); ?>">ID</a></th>
                    <th><a href="?sort=StudyDescription&order=<?php echo ($sortColumn == 'StudyDescription' && $sortOrder == 'asc' ? 'desc' : 'asc'); ?>">DESCRIPCIÓN</a></th>                            
                    <th><a href="?sort=StudyDate&order=<?php echo ($sortColumn == 'StudyDate' && $sortOrder == 'asc' ? 'desc' : 'asc'); ?>">FECHA</a></th>                            
                            <th>ACCIONES</th>
                        </tr>
                    </thead>
            <tbody>
                <?php foreach ($jsonData as $study): ?>
                    <?php 
                $patientName = str_replace("^", " ", $study['PatientMainDicomTags']['PatientName'] ?? '');
                $patientid = str_replace("^", " ", $study['PatientMainDicomTags']['PatientID'] ?? '');
                $studydescription = str_replace("^", " ", $study['MainDicomTags']['StudyDescription'] ?? '');
                $studyInstanceUID = $study['MainDicomTags']['StudyInstanceUID'];                
                $studyID = $study['ID'] ?? '';  // Usar Orthanc Study ID directamente, igual que en admin
                $studyDate = $study['MainDicomTags']['StudyDate'] ?? '';
                $modality = $study['MainDicomTags']['Modality'] ?? 'UNKNOWN';
                $filename = construirNombreArchivo($patientName, $studyDate, $modality);
                $downloadUrl = "https://iset.portalpaciente.com.ar/api/isetarc/{$studyInstanceUID}?accept=application/zip&filename=" . urlencode($filename);
                
                if (stripos($studydescription, 'Cone') !== false) {
                    $viewUrl = "https://iset.portalpaciente.com.ar/volview/index.html?names=[archive.zip]&urls=[../studies/{$studyID}/archive]";
                } else {
                    $viewUrl = "view.php?study=" . urlencode($study['MainDicomTags']['StudyInstanceUID']);
                }
                    ?>
                    <tr data-study-uid="<?php echo htmlspecialchars($studyInstanceUID); ?>">
                        <td><?php echo htmlspecialchars($patientName); ?>
                            <?php if (isset($study['es_derivado'])): ?>
                                <span class="derived-badge">DERIVADO</span>
                                            <?php endif; ?>
                        </td>
                        <td><?php echo isset($study['es_derivado']) ? $study['PatientMainDicomTags']['PatientID'] : htmlspecialchars($patientid); ?></td>
                        <td><?php echo htmlspecialchars($studydescription ?: 'No disponible'); ?></td>
                        <td><?php echo date("d/m/Y", strtotime($study['MainDicomTags']['StudyDate'])); ?></td>
                        <td class='actions'>
                            <button onclick='downloadAndRename("<?php echo htmlspecialchars($downloadUrl); ?>", "<?php echo htmlspecialchars($safePatientName); ?>", this)' class='action-button download'>Descargar</button>
                            
                            <?php if (!isset($study['es_derivado'])): ?>
                                <button onclick='mostrarModalDerivarMejorado("<?php echo htmlspecialchars($studyInstanceUID); ?>", "<?php echo htmlspecialchars($patientName); ?>", "<?php echo htmlspecialchars($studydescription); ?>", "<?php echo $study['MainDicomTags']['StudyDate']; ?>")' class='action-button derive'>Derivar</button>
                                            <?php endif; ?>
                            
                            <a href="<?php echo htmlspecialchars($viewUrl); ?>" target='_blank' rel='noopener noreferrer' class='action-button view'>Visualizar</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
                </table>
                            <?php else: ?>
        <h3>No se encontraron estudios para el paciente.</h3>
                            <?php endif; ?>
                
                <!-- Incluir el modal mejorado de derivación -->
                <?php include 'derivacion_modal_user.php'; ?>
                
                <script>
                // Función mejorada para mostrar el modal de derivación con información del estudio
                function mostrarModalDerivarMejorado(studyUID, patientName, studyDescription, studyDate) {
                    const estudioInfo = {
                        patientName: patientName,
                        studyDescription: studyDescription,
                        studyDate: studyDate ? new Date(studyDate).toLocaleDateString('es-ES') : 'No disponible'
                    };
                    
                    // Llamar a la función del modal mejorado
                    mostrarModalDerivar(studyUID, estudioInfo);
                }
                </script>
                
                <!-- Botón flotante para volver al inicio -->
                <button id="scrollToTopBtn" class="scroll-to-top" title="Volver al inicio" onclick="scrollToTop()">
                    ↑
                </button>

                <script>
                // Variables globales para la funcionalidad de derivación
                var usuariosDisponibles = [];
                
                // Funcionalidad del botón flotante para volver al inicio
                function scrollToTop() {
                    window.scrollTo({
                        top: 0,
                        behavior: 'smooth'
                    });
                }
                
                // Controlar visibilidad del botón flotante
                function toggleScrollButton() {
                    const scrollBtn = document.getElementById('scrollToTopBtn');
                    if (window.pageYOffset > 300) { // Mostrar después de 300px de scroll
                        scrollBtn.classList.add('show');
                    } else {
                        scrollBtn.classList.remove('show');
                    }
                }
                
                // Event listeners para el botón flotante
                window.addEventListener('scroll', toggleScrollButton);
                window.addEventListener('load', toggleScrollButton);
                

                                
                // Función para desactivar derivaciones
                function desactivarDerivacion(derivacionId) {
                    // Verificar si SweetAlert2 está disponible
                    if (typeof Swal !== 'undefined') {
                        Swal.fire({
                            title: '¿Cancelar derivación?',
                            text: '¿Estás seguro de que deseas cancelar esta derivación? Esta acción no se puede deshacer.',
                            icon: 'warning',
                            showCancelButton: true,
                            confirmButtonColor: '#dc3545',
                            cancelButtonColor: '#6c757d',
                            confirmButtonText: 'Sí, cancelar',
                            cancelButtonText: 'No, mantener',
                            reverseButtons: true
                        }).then((result) => {
                            if (result.isConfirmed) {
                                procesarCancelacion(derivacionId);
                            }
                        });
                    } else {
                        // Usar sistema de notificaciones personalizado
                        showCustomConfirmDialog(
                            '¿Cancelar derivación?',
                            '¿Estás seguro de que deseas cancelar esta derivación? Esta acción no se puede deshacer.',
                            () => procesarCancelacion(derivacionId),
                            'Sí, cancelar'
                        );
                    }
                }
                
                // Función para mostrar diálogo de confirmación personalizado
                function showCustomConfirmDialog(title, message, onConfirm, confirmText = 'Sí, confirmar') {
                    // Crear overlay
                    const overlay = document.createElement('div');
                    overlay.style.cssText = `
                        position: fixed;
                        top: 0;
                        left: 0;
                        width: 100%;
                        height: 100%;
                        background: rgba(0,0,0,0.5);
                        z-index: 10000;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                    `;
                    
                    // Crear modal
                    const modal = document.createElement('div');
                    modal.style.cssText = `
                        background: white;
                        border-radius: 10px;
                        padding: 25px;
                        max-width: 400px;
                        width: 90%;
                        box-shadow: 0 10px 30px rgba(0,0,0,0.3);
                        text-align: center;
                    `;
                    
                    modal.innerHTML = `
                        <div style="font-size: 48px; margin-bottom: 15px;">⚠️</div>
                        <h3 style="margin: 0 0 15px 0; color: #495057; font-size: 18px;">${title}</h3>
                        <p style="margin: 0 0 25px 0; color: #6c757d; line-height: 1.5;">${message}</p>
                        <div style="display: flex; gap: 10px; justify-content: center;">
                            <button id="cancelBtn" style="
                                padding: 10px 20px;
                                background: #6c757d;
                                color: white;
                                border: none;
                                border-radius: 5px;
                                cursor: pointer;
                                font-size: 14px;
                                transition: background-color 0.2s;
                            ">No, mantener</button>
                            <button id="confirmBtn" style="
                                padding: 10px 20px;
                                background: #dc3545;
                                color: white;
                                border: none;
                                border-radius: 5px;
                                cursor: pointer;
                                font-size: 14px;
                                transition: background-color 0.2s;
                            ">${confirmText}</button>
                        </div>
                    `;
                    
                    // Agregar al DOM
                    overlay.appendChild(modal);
                    document.body.appendChild(overlay);
                    
                    // Event listeners
                    const cancelBtn = modal.querySelector('#cancelBtn');
                    const confirmBtn = modal.querySelector('#confirmBtn');
                    
                    cancelBtn.addEventListener('click', () => {
                        document.body.removeChild(overlay);
                    });
                    
                    confirmBtn.addEventListener('click', () => {
                        document.body.removeChild(overlay);
                        onConfirm();
                    });
                    
                    // Cerrar al hacer clic en el overlay
                    overlay.addEventListener('click', (e) => {
                        if (e.target === overlay) {
                            document.body.removeChild(overlay);
                        }
                    });
                    
                    // Cerrar con Escape
                    const handleEscape = (e) => {
                        if (e.key === 'Escape') {
                            document.body.removeChild(overlay);
                            document.removeEventListener('keydown', handleEscape);
                        }
                    };
                    document.addEventListener('keydown', handleEscape);
                    
                    // Hover effects
                    cancelBtn.addEventListener('mouseenter', () => {
                        cancelBtn.style.backgroundColor = '#5a6268';
                    });
                    cancelBtn.addEventListener('mouseleave', () => {
                        cancelBtn.style.backgroundColor = '#6c757d';
                    });
                    
                    confirmBtn.addEventListener('mouseenter', () => {
                        confirmBtn.style.backgroundColor = '#c82333';
                    });
                    confirmBtn.addEventListener('mouseleave', () => {
                        confirmBtn.style.backgroundColor = '#dc3545';
                    });
                }
                
                function procesarCancelacion(derivacionId) {
                    fetch('../../controller/derivacionController.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: `action=desactivar&derivacion_id=${derivacionId}`
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            if (typeof Swal !== 'undefined') {
                                Swal.fire({
                                    icon: 'success',
                                    title: 'Derivación cancelada',
                                    text: 'La derivación ha sido cancelada exitosamente',
                                    confirmButtonText: 'Aceptar',
                                    confirmButtonColor: '#28a745',
                                    timer: 2500,
                                    timerProgressBar: true
                                });
                            } else {
                                showSuccess('Derivación cancelada', 'La derivación ha sido cancelada exitosamente');
                            }
                            // Refrescar los paneles después de cancelar una derivación
                            console.log('Derivación cancelada, refrescando paneles...');
                            refrescarDerivaciones();
                        } else {
                            if (typeof Swal !== 'undefined') {
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Error al cancelar',
                                    text: 'Error: ' + data.message,
                                    confirmButtonText: 'Aceptar',
                                    confirmButtonColor: '#dc3545'
                                });
                            } else {
                                showError('Error al cancelar', 'Error: ' + data.message);
                            }
                        }
                    })
                    .catch(error => {
                        if (typeof Swal !== 'undefined') {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error al cancelar',
                                text: 'Error al cancelar la derivación. Por favor, inténtelo de nuevo.',
                                confirmButtonText: 'Aceptar',
                                confirmButtonColor: '#dc3545'
                            });
                        } else {
                            showError('Error al cancelar', 'Error al cancelar la derivación. Por favor, inténtelo de nuevo.');
                        }
                    });
                }
                
                // Función para ocultar derivación recibida (marcar como vista permanentemente)
                function ocultarDerivacionRecibida(derivacionId) {
                    // Verificar si SweetAlert2 está disponible
                    if (typeof Swal !== 'undefined') {
                        Swal.fire({
                            title: '¿Marcar como vista?',
                            text: '¿Estás seguro de que deseas marcar esta derivación como vista? Se ocultará de tu lista de derivaciones recibidas.',
                            icon: 'question',
                            showCancelButton: true,
                            confirmButtonColor: '#28a745',
                            cancelButtonColor: '#6c757d',
                            confirmButtonText: 'Sí, marcar como vista',
                            cancelButtonText: 'No, mantener',
                            reverseButtons: true
                        }).then((result) => {
                            if (result.isConfirmed) {
                                procesarOcultacion(derivacionId);
                            }
                        });
                    } else {
                        // Usar sistema de notificaciones personalizado
                        showCustomConfirmDialog(
                            '¿Marcar como vista?',
                            '¿Estás seguro de que deseas marcar esta derivación como vista? Se ocultará de tu lista de derivaciones recibidas.',
                            () => procesarOcultacion(derivacionId),
                            'Sí, marcar como vista'
                        );
                    }
                }
                
                // Función para procesar la ocultación de derivación recibida
                function procesarOcultacion(derivacionId) {
                    console.log('🔍 Procesando ocultación de derivación:', derivacionId);
                    
                    fetch('../../controller/derivacionController.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: `action=ocultar_derivacion&derivacion_id=${derivacionId}`
                    })
                    .then(response => {
                        console.log('📡 Respuesta del servidor:', response);
                        return response.json();
                    })
                    .then(data => {
                        console.log('📊 Datos recibidos:', data);
                        if (data.success) {
                            if (typeof Swal !== 'undefined') {
                                Swal.fire({
                                    icon: 'success',
                                    title: 'Derivación marcada como vista',
                                    text: 'La derivación ha sido oculta de tu lista de derivaciones recibidas',
                                    confirmButtonText: 'Aceptar',
                                    confirmButtonColor: '#28a745',
                                    timer: 2500,
                                    timerProgressBar: true
                                });
                            } else {
                                showSuccess('Derivación marcada como vista', 'La derivación ha sido oculta de tu lista de derivaciones recibidas');
                            }
                            // Refrescar los paneles después de ocultar una derivación
                            console.log('Derivación oculta, refrescando paneles...');
                            // Forzar refresco completo sin timestamp para mostrar cambios
                            ultimaActualizacion = null;
                            refrescarDerivaciones();
                            
                            // Si estamos en vista de ocultas, actualizar también esa vista
                            if (vistaDerivacionesActual === 'ocultas') {
                                setTimeout(() => {
                                    actualizarVistaOcultas();
                                }, 1000);
                            }
                        } else {
                            if (typeof Swal !== 'undefined') {
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Error al ocultar',
                                    text: 'Error: ' + data.message,
                                    confirmButtonText: 'Aceptar',
                                    confirmButtonColor: '#dc3545'
                                });
                            } else {
                                showError('Error al ocultar', 'Error: ' + data.message);
                            }
                        }
                    })
                    .catch(error => {
                        if (typeof Swal !== 'undefined') {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error al ocultar',
                                text: 'Error al ocultar la derivación. Por favor, inténtelo de nuevo.',
                                confirmButtonText: 'Aceptar',
                                confirmButtonColor: '#dc3545'
                            });
                        } else {
                            showError('Error al ocultar', 'Error al ocultar la derivación. Por favor, inténtelo de nuevo.');
                        }
                    });
                }
                
                // Variable para controlar la vista actual de derivaciones
                let vistaDerivacionesActual = 'normal'; // 'normal' o 'ocultas'
                
                // Función para limpiar timestamps cuando sea necesario
                function limpiarTimestamps() {
                    console.log('🧹 Limpiando timestamps de actualización...');
                    localStorage.removeItem('ultimaActualizacionDerivaciones');
                    localStorage.removeItem('ultimaActualizacionListaPacientes');
                    ultimaActualizacion = null;
                }
                
                // Función para detectar y eliminar duplicados en la lista de pacientes
                function detectarDuplicadosListaPacientes() {
                    console.log('🔍 Verificando duplicados en lista de pacientes...');
                    
                    const filasPacientes = document.querySelectorAll('tbody tr');
                    const studyInstanceUIDs = new Set();
                    const duplicados = [];
                    
                    filasPacientes.forEach((fila, index) => {
                        // Usar el atributo data-study-uid en lugar de la columna 2
                        const studyInstanceUID = fila.getAttribute('data-study-uid');
                        if (studyInstanceUID) {
                            if (studyInstanceUIDs.has(studyInstanceUID)) {
                                duplicados.push({ index, studyInstanceUID });
                                console.log(`⚠️ Duplicado encontrado en fila ${index + 1}: ${studyInstanceUID}`);
                            } else {
                                studyInstanceUIDs.add(studyInstanceUID);
                            }
                        }
                    });
                    
                    if (duplicados.length > 0) {
                        console.log(`🔄 Eliminando ${duplicados.length} duplicados...`);
                        // Eliminar duplicados (mantener solo el primero)
                        duplicados.reverse().forEach(duplicado => {
                            const fila = filasPacientes[duplicado.index];
                            if (fila && fila.parentNode) {
                                fila.parentNode.removeChild(fila);
                                console.log(`✅ Duplicado eliminado: ${duplicado.studyInstanceUID}`);
                            }
                        });
                        
                        // Actualizar contador
                        const contador = document.getElementById('studyCount');
                        if (contador) {
                            contador.textContent = studyInstanceUIDs.size;
                        }
                    } else {
                        console.log('✅ No se encontraron duplicados');
                    }
                    
                    return duplicados.length;
                }
                
                // Función para actualizar la vista de derivaciones ocultas
                function actualizarVistaOcultas() {
                    console.log('🔄 Actualizando vista de derivaciones ocultas...');
                    
                    fetch('../../controller/derivacionController.php', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                        body: 'action=obtener_derivaciones_ocultas'
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success && data.derivacionesOcultas) {
                            console.log('📊 Derivaciones ocultas obtenidas:', data.derivacionesOcultas.length);
                            actualizarPanelDerivaciones('ocultas', data.derivacionesOcultas);
                        }
                    })
                    .catch(error => {
                        console.error('❌ Error al actualizar vista ocultas:', error);
                    });
                }
                
                // Función para alternar entre vista normal y ocultas
                function alternarVistaDerivaciones() {
                    const contenidoNormal = document.getElementById('contenido-derivaciones-recibidas');
                    const contenidoOcultas = document.getElementById('contenido-derivaciones-ocultas');
                    const btnVerOcultas = document.getElementById('btn-ver-ocultas');
                    const btnVerNormales = document.getElementById('btn-ver-normales');
                    const contadorRecibidas = document.getElementById('contador-recibidas');
                    
                    if (vistaDerivacionesActual === 'normal') {
                        // Cambiar a vista de ocultas
                        contenidoNormal.style.display = 'none';
                        contenidoOcultas.style.display = 'block';
                        contenidoOcultas.style.maxHeight = '500px';
                        contenidoOcultas.style.overflowY = 'auto';
                        btnVerOcultas.style.display = 'none';
                        btnVerNormales.style.display = 'inline-block';
                        contadorRecibidas.textContent = '<?php echo $derivacionesOcultas ? $derivacionesOcultas->num_rows : 0; ?>';
                        contadorRecibidas.style.background = '#6c757d';
                        vistaDerivacionesActual = 'ocultas';
                        console.log('Cambiado a vista de derivaciones ocultas');
                    } else {
                        // Cambiar a vista normal
                        contenidoOcultas.style.display = 'none';
                        contenidoOcultas.style.maxHeight = '0px';
                        contenidoOcultas.style.overflowY = 'hidden';
                        contenidoNormal.style.display = 'block';
                        btnVerOcultas.style.display = 'inline-block';
                        btnVerNormales.style.display = 'none';
                        contadorRecibidas.textContent = '<?php echo $derivacionesRecibidas ? $derivacionesRecibidas->num_rows : 0; ?>';
                        contadorRecibidas.style.background = '#28a745';
                        vistaDerivacionesActual = 'normal';
                        console.log('Cambiado a vista normal de derivaciones');
                    }
                }
                
                // Función para mostrar una derivación oculta (quitar de ocultas)
                function mostrarDerivacionOculta(derivacionId) {
                    if (typeof Swal !== 'undefined') {
                        Swal.fire({
                            title: '¿Mostrar derivación?',
                            text: '¿Estás seguro de que deseas mostrar esta derivación nuevamente en tu lista normal?',
                            icon: 'question',
                            showCancelButton: true,
                            confirmButtonColor: '#17a2b8',
                            cancelButtonColor: '#6c757d',
                            confirmButtonText: 'Sí, mostrar',
                            cancelButtonText: 'No, mantener oculta',
                            reverseButtons: true
                        }).then((result) => {
                            if (result.isConfirmed) {
                                procesarMostrarDerivacion(derivacionId);
                            }
                        });
                    } else {
                        showCustomConfirmDialog(
                            '¿Mostrar derivación?',
                            '¿Estás seguro de que deseas mostrar esta derivación nuevamente en tu lista normal?',
                            () => procesarMostrarDerivacion(derivacionId),
                            'Sí, mostrar'
                        );
                    }
                }
                
                // Función para procesar la acción de mostrar derivación oculta
                function procesarMostrarDerivacion(derivacionId) {
                    console.log('🔍 Procesando mostrar derivación:', derivacionId);
                    
                    fetch('../../controller/derivacionController.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: `action=mostrar_derivacion&derivacion_id=${derivacionId}`
                    })
                    .then(response => {
                        console.log('📡 Respuesta del servidor:', response);
                        return response.json();
                    })
                    .then(data => {
                        console.log('📊 Datos recibidos:', data);
                        if (data.success) {
                            if (typeof Swal !== 'undefined') {
                                Swal.fire({
                                    icon: 'success',
                                    title: 'Derivación mostrada',
                                    text: 'La derivación ha sido mostrada nuevamente en tu lista de derivaciones recibidas',
                                    confirmButtonText: 'Aceptar',
                                    confirmButtonColor: '#17a2b8',
                                    timer: 2500,
                                    timerProgressBar: true
                                });
                            } else {
                                showSuccess('Derivación mostrada', 'La derivación ha sido mostrada nuevamente en tu lista de derivaciones recibidas');
                            }
                            // Refrescar los paneles después de mostrar una derivación
                            console.log('Derivación mostrada, refrescando paneles...');
                            ultimaActualizacion = null;
                            refrescarDerivaciones();
                            
                            // Actualizar también la vista de ocultas si está activa
                            if (vistaDerivacionesActual === 'ocultas') {
                                setTimeout(() => {
                                    actualizarVistaOcultas();
                                }, 1000);
                            }
                        } else {
                            if (typeof Swal !== 'undefined') {
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Error al mostrar',
                                    text: 'Error: ' + data.message,
                                    confirmButtonText: 'Aceptar',
                                    confirmButtonColor: '#dc3545'
                                });
                            } else {
                                showError('Error al mostrar', 'Error: ' + data.message);
                            }
                        }
                    })
                    .catch(error => {
                        if (typeof Swal !== 'undefined') {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error al mostrar',
                                text: 'Error al mostrar la derivación. Por favor, inténtelo de nuevo.',
                                confirmButtonText: 'Aceptar',
                                confirmButtonColor: '#dc3545'
                            });
                        } else {
                            showError('Error al mostrar', 'Error al mostrar la derivación. Por favor, inténtelo de nuevo.');
                        }
                    });
                }
                
                // Función para marcar como vista
                function marcarComoVista(derivacionId) {
                    console.log('Marcando derivación como vista:', derivacionId);
                    
                    fetch('../../controller/derivacionController.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: `action=marcar_vista&derivacion_id=${derivacionId}`
                    })
                    .then(response => {
                        console.log('Respuesta del servidor:', response);
                        return response.json();
                    })
                    .then(data => {
                        console.log('Datos recibidos:', data);
                        if (data.success) {
                            console.log('Derivación marcada como vista exitosamente');
                            // Refrescar el panel de derivaciones recibidas
                            refrescarDerivaciones();
                        } else {
                            console.error('Error al marcar como vista:', data.message);
                            if (typeof Swal !== 'undefined') {
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Error al marcar como vista',
                                    text: 'No se pudo marcar la derivación como vista. Por favor, inténtelo de nuevo.',
                                    confirmButtonText: 'Aceptar',
                                    confirmButtonColor: '#dc3545'
                                });
                            } else {
                                showError('Error al marcar como vista', 'No se pudo marcar la derivación como vista. Por favor, inténtelo de nuevo.');
                            }
                        }
                    })
                    .catch(error => {
                        console.error('Error en la petición:', error);
                        if (typeof Swal !== 'undefined') {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error de conexión',
                                text: 'No se pudo conectar con el servidor. Por favor, verifique su conexión e inténtelo de nuevo.',
                                confirmButtonText: 'Aceptar',
                                confirmButtonColor: '#dc3545'
                            });
                        } else {
                            showError('Error de conexión', 'No se pudo conectar con el servidor. Por favor, verifique su conexión e inténtelo de nuevo.');
                        }
                    });
                }

                // Función para mostrar notificación de nuevas derivaciones
                function mostrarNotificacionNuevasDerivaciones(nuevasDerivaciones) {
                    console.log('🎯 Función mostrarNotificacionNuevasDerivaciones llamada');
                    console.log('📋 Datos de nuevas derivaciones:', nuevasDerivaciones);
                    
                    if (!nuevasDerivaciones || !Array.isArray(nuevasDerivaciones) || nuevasDerivaciones.length === 0) {
                        console.log('⚠️ No hay nuevas derivaciones para mostrar');
                        return;
                    }
                    
                    // Crear mensaje con información de las nuevas derivaciones
                    let titulo = '';
                    let mensaje = '';
                    
                    if (nuevasDerivaciones.length === 1) {
                        const derivacion = nuevasDerivaciones[0];
                        titulo = '🆕 Nueva derivación recibida';
                        mensaje = `Has recibido una nueva derivación:\n\n👤 Paciente: ${derivacion.patientName}\n👨‍⚕️ De: ${derivacion.usuario_origen_nombre}\n📋 Estudio: ${derivacion.studyDescription || 'No especificado'}`;
                        console.log('📝 Notificación para 1 derivación:', { titulo, mensaje });
                    } else {
                        titulo = `🆕 ${nuevasDerivaciones.length} nuevas derivaciones recibidas`;
                        mensaje = `Has recibido ${nuevasDerivaciones.length} nuevas derivaciones:\n\n`;
                        nuevasDerivaciones.forEach((derivacion, index) => {
                            if (index < 3) { // Mostrar solo las primeras 3
                                mensaje += `• ${derivacion.patientName} - ${derivacion.usuario_origen_nombre}\n`;
                            }
                        });
                        if (nuevasDerivaciones.length > 3) {
                            mensaje += `\n... y ${nuevasDerivaciones.length - 3} derivación(es) más`;
                        }
                        console.log('📝 Notificación para múltiples derivaciones:', { titulo, mensaje });
                    }
                    
                    // Mostrar notificación personalizada con tipo 'success' para nuevas derivaciones
                    console.log('🎨 Llamando a showCustomNotification...');
                    try {
                        const notification = showCustomNotification('success', titulo, mensaje, 8000);
                        console.log('✅ Notificación mostrada exitosamente:', notification);
                    } catch (error) {
                        console.error('❌ Error al mostrar notificación:', error);
                        // Fallback: usar alert si falla la notificación personalizada
                        alert(`${titulo}\n\n${mensaje}`);
                    }
                    
                    // Verificar preferencia del usuario antes de expandir automáticamente
                    const userPreference = localStorage.getItem('panel-recibidas-expanded');
                    const panelRecibidas = document.getElementById('contenido-derivaciones-recibidas');
                    
                    if (panelRecibidas) {
                        // Solo expandir si el usuario no lo ha contraído manualmente
                        if (userPreference !== 'false') {
                            expandPanel('recibidas');
                            console.log('📤 Panel de recibidas expandido automáticamente por nuevas derivaciones');
                        } else {
                            console.log('📤 Panel de recibidas mantenido contraído por preferencia del usuario');
                        }
                    }
                    
                    // Mostrar notificación en el panel (siempre)
                    const notificacionPanel = document.getElementById('notificacion-nuevas');
                    if (notificacionPanel) {
                        notificacionPanel.style.display = 'block';
                        notificacionPanel.style.animation = 'pulse 1s infinite';
                        console.log('🔔 Badge de notificación activado');
                    } else {
                        console.log('⚠️ No se encontró el badge de notificación');
                    }
                    
                    // NO actualizar automáticamente la lista de pacientes aquí para evitar bucles
                    // La lista se actualizará en el próximo ciclo de refresco automático
                    console.log('ℹ️ Lista de pacientes se actualizará en el próximo ciclo automático');
                }
                

                
                // Función para forzar detección de nuevas derivaciones en la primera carga
                function detectarNuevasDerivacionesIniciales() {
                    console.log('🔍 Detectando nuevas derivaciones iniciales...');
                    
                    // Hacer una primera llamada sin timestamp para obtener todas las derivaciones
                    fetch('../../controller/derivacionController.php', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                        body: 'action=obtener_derivaciones'
                    })
                    .then(response => response.json())
                    .then(data => {
                        console.log('📡 Respuesta inicial del servidor:', data);
                        if (data.success && data.nuevasDerivaciones && data.nuevasDerivaciones.length > 0) {
                            console.log('🆕 Nuevas derivaciones encontradas en carga inicial:', data.nuevasDerivaciones);
                            console.log('📊 Cantidad de nuevas derivaciones iniciales:', data.nuevasDerivaciones.length);
                            mostrarNotificacionNuevasDerivaciones(data.nuevasDerivaciones);
                        } else {
                            console.log('ℹ️ No hay nuevas derivaciones en la carga inicial');
                        }
                        
                        // Establecer el timestamp inicial
                        ultimaActualizacion = new Date().toISOString();
                        console.log('⏰ Timestamp inicial establecido:', ultimaActualizacion);
                    })
                    .catch(error => {
                        console.error('❌ Error en detección inicial:', error);
                    });
                }
                
                // Función para refrescar manualmente
                function refrescarManual() {
                    console.log('Refresco manual iniciado...');
                    refrescarDerivaciones();
                    // También actualizar la lista de pacientes
                    setTimeout(() => {
                        actualizarListaPacientes();
                    }, 2000); // Esperar 2 segundos para que termine el refresco de derivaciones
                }
                
                // Variable para almacenar la última actualización
                let ultimaActualizacion = localStorage.getItem('ultimaActualizacionDerivaciones') || null;
                
                // Función para limpiar timestamps cuando sea necesario
                function limpiarTimestamps() {
                    console.log('🧹 Limpiando timestamps de actualización...');
                    localStorage.removeItem('ultimaActualizacionDerivaciones');
                    localStorage.removeItem('ultimaActualizacionListaPacientes');
                    ultimaActualizacion = null;
                }
                
                // Función para establecer timestamp inicial si no existe
                function establecerTimestampInicial() {
                    if (!ultimaActualizacion) {
                        ultimaActualizacion = new Date().toISOString();
                        localStorage.setItem('ultimaActualizacionDerivaciones', ultimaActualizacion);
                        console.log('⏰ Timestamp inicial establecido:', ultimaActualizacion);
                    }
                }
                
                // Función para refrescar los paneles de derivaciones
                function refrescarDerivaciones() {
                    console.log('🔄 Refrescando derivaciones...');
                    console.log('⏰ Última actualización:', ultimaActualizacion);
                    
                    const body = ultimaActualizacion 
                        ? `action=obtener_derivaciones&ultima_actualizacion=${ultimaActualizacion}`
                        : 'action=obtener_derivaciones';
                    
                    console.log('📤 Enviando body:', body);
                    
                    fetch('../../controller/derivacionController.php', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                        body: body
                    })
                    .then(response => response.json())
                    .then(data => {
                        console.log('📡 Respuesta del servidor:', data);
                        if (data.success) {
                            // Siempre actualizar ambos paneles completos
                            if (data.derivacionesEnviadas !== undefined) {
                                console.log('📤 Actualizando derivaciones enviadas:', data.derivacionesEnviadas.length);
                                actualizarPanelDerivaciones('enviadas', data.derivacionesEnviadas);
                            } else {
                                console.log('📤 No hay datos de derivaciones enviadas');
                            }
                            
                            if (data.derivacionesRecibidas !== undefined) {
                                console.log('📥 Actualizando derivaciones recibidas:', data.derivacionesRecibidas.length);
                                actualizarPanelDerivaciones('recibidas', data.derivacionesRecibidas);
                            } else {
                                console.log('📥 No hay datos de derivaciones recibidas');
                            }
                            
                            // Mostrar notificación si hay nuevas derivaciones
                            if (data.nuevasDerivaciones && data.nuevasDerivaciones.length > 0) {
                                console.log('🆕 NUEVAS DERIVACIONES DETECTADAS:', data.nuevasDerivaciones);
                                console.log('📊 Cantidad de nuevas derivaciones:', data.nuevasDerivaciones.length);
                                console.log('🎯 Llamando a mostrarNotificacionNuevasDerivaciones...');
                                mostrarNotificacionNuevasDerivaciones(data.nuevasDerivaciones);
                                
                                // Solo actualizar timestamp cuando hay nuevas derivaciones
                                ultimaActualizacion = new Date().toISOString();
                                localStorage.setItem('ultimaActualizacionDerivaciones', ultimaActualizacion);
                                console.log('⏰ Nueva última actualización guardada:', ultimaActualizacion);
                            } else {
                                console.log('ℹ️ No hay nuevas derivaciones en este refresco');
                            }
                        } else {
                            console.error('❌ Error en la respuesta del servidor:', data);
                        }
                    })
                    .catch(error => {
                        console.error('❌ Error al refrescar derivaciones:', error);
                    });
                }
                
                // Función para convertir fecha DICOM a formato legible
                function formatearFechaDICOM(fechaDICOM) {
                    if (!fechaDICOM || fechaDICOM === 'No disponible') {
                        return 'No disponible';
                    }
                    
                    try {
                        // Formato DICOM: YYYYMMDD
                        if (fechaDICOM.length === 8) {
                            const year = fechaDICOM.substring(0, 4);
                            const month = fechaDICOM.substring(4, 6);
                            const day = fechaDICOM.substring(6, 8);
                            const fecha = new Date(year, month - 1, day); // month - 1 porque JavaScript usa 0-11
                            return fecha.toLocaleDateString('es-ES');
                        }
                        
                        // Si no es formato DICOM, intentar parsear como fecha normal
                        const fecha = new Date(fechaDICOM);
                        if (!isNaN(fecha.getTime())) {
                            return fecha.toLocaleDateString('es-ES');
                        }
                        
                        return 'No disponible';
                    } catch (error) {
                        console.error('Error formateando fecha DICOM:', error, fechaDICOM);
                        return 'No disponible';
                    }
                }
                
                // Función para actualizar un panel específico de derivaciones
                function actualizarPanelDerivaciones(tipo, derivaciones) {
                    console.log(`🔄 Iniciando actualización del panel ${tipo}`);
                    
                    const container = document.getElementById(`contenido-derivaciones-${tipo}`);
                    const contador = document.getElementById(`contador-${tipo}`);
                    const icono = document.getElementById(`icono-${tipo}`);
                    const notificacion = document.getElementById('notificacion-nuevas');

                    if (!container) {
                        console.error(`❌ No se encontró el contenedor: contenido-derivaciones-${tipo}`);
                        return;
                    }
                    
                    if (!contador) {
                        console.error(`❌ No se encontró el contador: contador-${tipo}`);
                        return;
                    }

                    if (!icono) {
                        console.error(`❌ No se encontró el icono: icono-${tipo}`);
                        return;
                    }

                    if (!notificacion) {
                        console.error(`❌ No se encontró la notificación: notificacion-nuevas`);
                        return;
                    }
                    
                    console.log(`✅ Contenedor encontrado para ${tipo}:`, container);
                    console.log(`📊 Datos a procesar:`, derivaciones);
                    
                    if (!Array.isArray(derivaciones)) {
                        console.error(`❌ Los datos no son un array:`, derivaciones);
                        return;
                    }
                    
                    // Obtener el contador anterior para comparar
                    const contadorAnterior = parseInt(contador.textContent) || 0;
                    const contadorNuevo = derivaciones.length;
                    
                    console.log(`📈 Contador anterior: ${contadorAnterior}, Contador nuevo: ${contadorNuevo}`);
                    
                    if (derivaciones.length === 0) {
                        container.style.display = 'none'; // Ocultar el panel si no hay derivaciones
                        icono.textContent = tipo === 'enviadas' ? '📤' : '📥';
                        contador.textContent = '0';
                        notificacion.style.display = 'none';
                        console.log(`📤 Panel ${tipo} actualizado - sin derivaciones`);
                        return;
                    }

                    container.style.display = 'block'; // Mostrar el panel
                    icono.textContent = tipo === 'enviadas' ? '📤' : '📥';
                    contador.textContent = derivaciones.length;

                    // Verificar si hay derivaciones nuevas (no vistas) para el panel de recibidas
                    if (tipo === 'recibidas') {
                        const hayNuevas = derivaciones.some(d => (d.vista || 0) === 0 && (d.descargada || 0) === 0);
                        const userPreference = localStorage.getItem('panel-recibidas-expanded');
                        
                        // DETECTAR SI EL CONTADOR INCREMENTÓ Y MOSTRAR NOTIFICACIÓN
                        if (contadorNuevo > contadorAnterior) {
                            console.log(`🆕 ¡CONTADOR INCREMENTÓ! De ${contadorAnterior} a ${contadorNuevo}`);
                            console.log(`🎯 Mostrando notificación automática por incremento de contador`);
                            
                            // Crear notificación automática basada en las nuevas derivaciones
                            const nuevasDerivaciones = derivaciones.slice(0, contadorNuevo - contadorAnterior);
                            const derivacionesParaNotificacion = nuevasDerivaciones.map(d => ({
                                patientName: d.patientName || 'Paciente',
                                usuario_origen_nombre: d.usuario_origen_nombre || 'Usuario',
                                studyDescription: d.studyDescription || 'Estudio'
                            }));
                            
                            mostrarNotificacionNuevasDerivaciones(derivacionesParaNotificacion);
                        }
                        
                        if (hayNuevas) {
                            // Solo expandir automáticamente si el usuario no lo ha contraído manualmente
                            if (userPreference !== 'false') {
                                expandPanel('recibidas');
                                notificacion.style.display = 'block';
                                notificacion.style.animation = 'pulse 1s infinite';
                                console.log('📤 Panel de recibidas expandido automáticamente - hay derivaciones nuevas');
                            } else {
                                // Mantener contraído pero mostrar notificación
                                collapsePanel('recibidas');
                                notificacion.style.display = 'block';
                                notificacion.style.animation = 'pulse 1s infinite';
                                console.log('📤 Panel de recibidas mantenido contraído por preferencia del usuario - hay derivaciones nuevas');
                            }
                        } else {
                            // No hay nuevas derivaciones, respetar preferencia del usuario
                            if (userPreference === 'true') {
                                expandPanel('recibidas');
                                console.log('📤 Panel de recibidas expandido por preferencia del usuario');
                            } else {
                                collapsePanel('recibidas');
                                console.log('📤 Panel de recibidas contraído por preferencia del usuario');
                            }
                            notificacion.style.display = 'none';
                        }
                    } else {
                        // Para enviadas, respetar preferencia del usuario
                        const userPreference = localStorage.getItem('panel-enviadas-expanded');
                        if (userPreference === 'true') {
                            expandPanel('enviadas');
                            console.log('📤 Panel de enviadas expandido por preferencia del usuario');
                        } else {
                            collapsePanel('enviadas');
                            console.log('📤 Panel de enviadas contraído por preferencia del usuario');
                        }
                    }
                    
                    let html = '';
                    derivaciones.forEach((derivacion, index) => {
                        console.log(`📝 Procesando derivación ${index + 1}:`, derivacion);
                        
                        try {
                            const fechaDerivacion = new Date(derivacion.fecha_derivacion).toLocaleString('es-ES');
                            const fechaRecibido = new Date(derivacion.fecha_derivacion).toLocaleString('es-ES');
                            
                            if (tipo === 'enviadas') {
                                html += `
                                    <div class="derivation-item" style="background: white; padding: 8px; margin-bottom: 6px; border-radius: 4px; border: 1px solid #dee2e6; font-size: 11px;">
                                        <div style="font-weight: bold; color: #495057; margin-bottom: 2px;">👤 ${derivacion.patientName || 'No disponible'}</div>
                                        <div style="color: #6c757d; margin-bottom: 2px;">📅 ${formatearFechaDICOM(derivacion.studyDate)} | ${derivacion.studyDescription || 'No disponible'}</div>
                                        <div style="color: #6c757d; margin-bottom: 2px;">👤 Para: ${derivacion.usuario_destino_nombre}</div>
                                        <div style="color: #6c757d; margin-bottom: 3px;">📅 Derivado: ${fechaDerivacion}</div>
                                        ${derivacion.comentario ? `<div style="color: #28a745; font-style: italic; margin-bottom: 3px;">💬 ${derivacion.comentario}</div>` : ''}
                                        <button onclick="desactivarDerivacion(${derivacion.id})" 
                                                style="background: #dc3545; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 10px; cursor: pointer;">
                                                ❌ Cancelar
                                            </button>
                                        </div>
                                `;
                            } else {
                                // Construir nombre de archivo: nombrepaciente_fechaestudio_modalidad.zip
                                const safePatientName = (derivacion.patientName || 'No disponible').replace(/[^a-zA-Z0-9_-]/g, '_');
                                const studyDate = derivacion.studyDate || '';
                                const modality = derivacion.modality || 'UNKNOWN';
                                // Formatear fecha: mantener formato YYYYMMDD o convertir
                                let fechaFormateada = studyDate;
                                if (studyDate && studyDate.length === 8) {
                                    fechaFormateada = studyDate; // Ya está en formato YYYYMMDD
                                } else if (studyDate) {
                                    const dateObj = new Date(studyDate);
                                    if (!isNaN(dateObj.getTime())) {
                                        fechaFormateada = dateObj.toISOString().slice(0, 10).replace(/-/g, '');
                                    }
                                }
                                const safeModality = modality.replace(/[^a-zA-Z0-9_-]/g, '_');
                                const filename = `${safePatientName}_${fechaFormateada}_${safeModality}.zip`;
                                const downloadUrl = `https://iset.portalpaciente.com.ar/api/isetarc/${derivacion.study_instance_uid}?accept=application/zip&filename=${encodeURIComponent(filename)}`;
                                const viewUrl = derivacion.studyDescription && derivacion.studyDescription.toLowerCase().includes('cone') 
                                    ? `https://iset.portalpaciente.com.ar/volview/index.html?names=[archive.zip]&urls=[../studies/${derivacion.studyID}/archive]`
                                    : `view.php?study=${encodeURIComponent(derivacion.study_instance_uid)}`;
                                
                                // Determinar el estado basado en si ha sido vista o descargada
                                const haSidoVista = (derivacion.vista || 0) == 1;
                                const haSidoDescargada = (derivacion.descargada || 0) == 1;
                                
                                // Color del borde según el estado
                                let colorBorde, colorFondo, indicadorEstado, textoEstado;
                                
                                if (haSidoVista || haSidoDescargada) {
                                    colorBorde = '#28a745'; // Verde para vista/descargada
                                    colorFondo = '#d4edda';
                                    indicadorEstado = '✅';
                                    textoEstado = haSidoVista ? 'Vista' : 'Descargada';
                                } else {
                                    colorBorde = '#ffc107'; // Amarillo para nueva
                                    colorFondo = '#fff3cd';
                                    indicadorEstado = '🆕';
                                    textoEstado = 'Nueva';
                                }
                                
                                html += `
                                    <div class="derivation-item" style="background: ${colorFondo}; padding: 8px; margin-bottom: 6px; border-radius: 4px; border: 2px solid ${colorBorde}; font-size: 11px; position: relative;">
                                        <!-- Indicador de estado -->
                                        <div style="position: absolute; top: 4px; right: 4px; background: ${colorBorde}; color: white; padding: 1px 4px; border-radius: 3px; font-size: 9px; font-weight: bold;">
                                            ${indicadorEstado} ${textoEstado}
                                        </div>
                                        
                                        <div style="font-weight: bold; color: #495057; margin-bottom: 2px;">👤 ${derivacion.patientName || 'No disponible'}</div>
                                        <div style="color: #6c757d; margin-bottom: 2px;">📅 ${formatearFechaDICOM(derivacion.studyDate)} | ${derivacion.studyDescription || 'No disponible'}</div>
                                        <div style="color: #6c757d; margin-bottom: 2px;">👤 De: ${derivacion.usuario_origen_nombre}</div>
                                        <div style="color: #6c757d; margin-bottom: 3px;">📅 Recibido: ${fechaRecibido}</div>
                                        ${derivacion.comentario ? `<div style="color: #28a745; font-style: italic; margin-bottom: 3px;">💬 ${derivacion.comentario}</div>` : ''}
                                        
                                        <div style="display: flex; gap: 4px; margin-top: 4px;">
                                            <button onclick='downloadAndRename("${downloadUrl}", "${safePatientName}", this, ${derivacion.id})' 
                                                    style="background: #3498db; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 9px; cursor: pointer;">
                                                📥 Descargar
                                            </button>
                                            <a href="${viewUrl}" target='_blank' rel='noopener noreferrer' 
                                               onclick="marcarComoVista(${derivacion.id})"
                                               style="background: #3498db; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 9px; cursor: pointer; text-decoration: none; display: inline-block;">
                                                👁️ Ver
                                            </a>
                                            <button onclick="ocultarDerivacionRecibida(${derivacion.id})" 
                                                    style="background: #28a745; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 9px; cursor: pointer;">
                                                ✅ Marcar como vista
                                            </button>
                                        </div>
                                    </div>
                                `;
                            }
                        } catch (error) {
                            console.error(`❌ Error procesando derivación ${index + 1}:`, error, derivacion);
                        }
                    });
                    
                    console.log(`✅ HTML generado para ${tipo}:`, html);
                    container.innerHTML = html;
                    console.log(`✅ Panel ${tipo} actualizado exitosamente`);
                }
                
                // Función para actualizar la lista de pacientes
                function actualizarListaPacientes(forzar = false) {
                    console.log('🔄 Actualizando lista de pacientes...');
                    
                    const ahora = new Date().toISOString();
                    const ultimaActualizacionLista = localStorage.getItem('ultimaActualizacionListaPacientes');
                    
                    if (forzar || !ultimaActualizacionLista || (new Date(ahora) - new Date(ultimaActualizacionLista)) > 30000) {
                        // Recargar si se fuerza o han pasado más de 30 segundos
                        console.log('🔄 Recargando página para actualizar lista de pacientes...');
                        localStorage.setItem('ultimaActualizacionListaPacientes', ahora);
                        window.location.reload();
                    } else {
                        console.log('⏰ Lista de pacientes actualizada recientemente, omitiendo recarga');
                    }
                }
                
                // Iniciar refresco automático cada 15 segundos
                setInterval(() => {
                    refrescarDerivaciones();
                }, 15000);
                
                // Actualizar lista de pacientes cada 2 minutos (separado del refresco de derivaciones)
                setInterval(() => {
                    actualizarListaPacientes();
                }, 120000); // 2 minutos
                
                // También refrescar cuando la página se vuelve visible
                document.addEventListener('visibilitychange', function() {
                    if (!document.hidden) {
                        console.log('👁️ Página visible, refrescando derivaciones...');
                        refrescarDerivaciones();
                    }
                });
                
                // Inicializar estado de los paneles al cargar la página
                document.addEventListener('DOMContentLoaded', function() {
                    console.log('🚀 Inicializando paneles de derivaciones...');
                    
                    // Establecer timestamp inicial para evitar detección falsa de nuevas derivaciones
                    establecerTimestampInicial();
                    
                    // Detectar y eliminar duplicados en la lista de pacientes
                    setTimeout(() => {
                        detectarDuplicadosListaPacientes();
                    }, 1000); // Esperar 1 segundo para que se cargue completamente
                    
                    // Detectar nuevas derivaciones iniciales
                    detectarNuevasDerivacionesIniciales();
                    
                    // Verificar preferencias del usuario para ambos paneles
                    const userPreferenceRecibidas = localStorage.getItem('panel-recibidas-expanded');
                    const userPreferenceEnviadas = localStorage.getItem('panel-enviadas-expanded');
                    
                    // Aplicar preferencias del usuario
                    if (userPreferenceRecibidas === 'true') {
                        expandPanel('recibidas');
                        console.log('Panel de recibidas expandido por preferencia del usuario');
                    } else if (userPreferenceRecibidas === 'false') {
                        collapsePanel('recibidas');
                        console.log('Panel de recibidas contraído por preferencia del usuario');
                    } else {
                        // Si no hay preferencia guardada, verificar si hay derivaciones nuevas
                        const panelRecibidas = document.getElementById('contenido-derivaciones-recibidas');
                        const notificacion = document.getElementById('notificacion-nuevas');
                        
                        if (panelRecibidas && panelRecibidas.children.length > 0) {
                            // Verificar si hay derivaciones nuevas (no vistas)
                            const derivacionesNuevas = panelRecibidas.querySelectorAll('.derivation-item');
                            let hayNuevas = false;
                            
                            derivacionesNuevas.forEach(item => {
                                const indicador = item.querySelector('div[style*="position: absolute"]');
                                if (indicador && indicador.textContent.includes('🆕')) {
                                    hayNuevas = true;
                                }
                            });
                            
                            // Solo expandir si hay derivaciones nuevas
                            if (hayNuevas) {
                                expandPanel('recibidas');
                                if (notificacion) {
                                    notificacion.style.display = 'block';
                                    notificacion.style.animation = 'pulse 1s infinite';
                                }
                                console.log('Panel de recibidas expandido automáticamente - hay derivaciones nuevas');
                            } else {
                                collapsePanel('recibidas');
                                console.log('Panel de recibidas contraído - no hay derivaciones nuevas');
                            }
                        } else {
                            collapsePanel('recibidas');
                            console.log('No hay derivaciones recibidas - panel contraído');
                        }
                    }
                    
                    // Aplicar preferencia para panel de enviadas
                    if (userPreferenceEnviadas === 'true') {
                        expandPanel('enviadas');
                        console.log('Panel de enviadas expandido por preferencia del usuario');
                    } else {
                        collapsePanel('enviadas');
                        console.log('Panel de enviadas contraído por preferencia del usuario');
                    }
                    
                    console.log('✅ Paneles inicializados correctamente');
                });

                // Función para alternar panel
                function togglePanel(tipo) {
                    const container = document.getElementById(`contenido-derivaciones-${tipo}`);
                    const icono = document.getElementById(`icono-${tipo}`);
                    
                    if (container.style.maxHeight === '0px' || container.style.maxHeight === '') {
                        // Expandir panel
                        container.style.maxHeight = '150px';
                        container.style.overflow = 'auto';
                        icono.textContent = tipo === 'enviadas' ? '📤' : '📥';
                        
                        // Guardar preferencia del usuario
                        localStorage.setItem(`panel-${tipo}-expanded`, 'true');
                        console.log(`Panel ${tipo} expandido por el usuario`);
                    } else {
                        // Contraer panel
                        container.style.maxHeight = '0px';
                        container.style.overflow = 'hidden';
                        icono.textContent = tipo === 'enviadas' ? '📤' : '📥';
                        
                        // Guardar preferencia del usuario
                        localStorage.setItem(`panel-${tipo}-expanded`, 'false');
                        console.log(`Panel ${tipo} contraído por el usuario`);
                    }
                }

                // Función para expandir panel (sin guardar preferencia)
                function expandPanel(tipo) {
                    const container = document.getElementById(`contenido-derivaciones-${tipo}`);
                    const icono = document.getElementById(`icono-${tipo}`);
                    
                    container.style.maxHeight = '150px';
                    container.style.overflow = 'auto';
                    icono.textContent = tipo === 'enviadas' ? '📤' : '📥';
                    console.log(`Panel ${tipo} expandido automáticamente`);
                }

                // Función para contraer panel (sin guardar preferencia)
                function collapsePanel(tipo) {
                    const container = document.getElementById(`contenido-derivaciones-${tipo}`);
                    const icono = document.getElementById(`icono-${tipo}`);
                    
                    container.style.maxHeight = '0px';
                    container.style.overflow = 'hidden';
                    icono.textContent = tipo === 'enviadas' ? '📤' : '📥';
                    console.log(`Panel ${tipo} contraído automáticamente`);
                }

                // Sistema de notificaciones personalizado
                function showCustomNotification(type, title, message, duration = 5000) {
                    // Crear elemento de notificación
                    const notification = document.createElement('div');
                    notification.className = `custom-notification ${type}`;
                    
                    // Iconos según el tipo
                    const icons = {
                        success: '✅',
                        error: '❌',
                        warning: '⚠️',
                        info: 'ℹ️'
                    };
                    
                    notification.innerHTML = `
                        <div class="icon">${icons[type] || 'ℹ️'}</div>
                        <div class="content">
                            <div class="title">${title}</div>
                            <div class="message">${message}</div>
                        </div>
                        <button class="close-btn" onclick="this.parentElement.remove()">×</button>
                    `;
                    
                    // Agregar al DOM
                    document.body.appendChild(notification);
                    
                    // Animar entrada
                    setTimeout(() => {
                        notification.classList.add('show');
                    }, 100);
                    
                    // Auto-remover después del tiempo especificado
                    if (duration > 0) {
                        setTimeout(() => {
                            notification.classList.remove('show');
                            setTimeout(() => {
                                if (notification.parentElement) {
                                    notification.parentElement.removeChild(notification);
                                }
                            }, 300);
                        }, duration);
                    }
                    
                    return notification;
                }
                
                // Funciones de conveniencia
                function showSuccess(title, message, duration = 4000) {
                    return showCustomNotification('success', title, message, duration);
                }
                
                function showError(title, message, duration = 6000) {
                    return showCustomNotification('error', title, message, duration);
                }
                
                function showWarning(title, message, duration = 5000) {
                    return showCustomNotification('warning', title, message, duration);
                }
                
                function showInfo(title, message, duration = 4000) {
                    return showCustomNotification('info', title, message, duration);
                }
                </script>

                </body>
                </html>

                <?php
                // Cerrar la conexión al final del archivo
                if (isset($conexionDerivados)) {
                $conexionDerivados->cerrar();
                }
                ?>
