<?php
session_start();

// Validamos que exista una session y ademas que el cargo que exista sea igual a 0 (Root) o 1 (Administradores)
if(!isset($_SESSION['cargo']) || ($_SESSION['cargo'] != '0' && $_SESSION['cargo'] != '1')){
    header('location: ../../index.php');
}

// Definir constante para permitir acceso al modal de derivación
define('ADMIN_ACCESS', true);

// Importar la clase Conexion
require_once('../../model/conexion.php');

// Modificar la consulta inicial (alrededor de la línea 20)
$refphysician = $_SESSION['nombre'];
$usuarioId = $_SESSION['id'];

// Credenciales
$username = "iset";
$password = "iset";

// Función para validar formato de fecha
function isValidDate($date) {
    $d = DateTime::createFromFormat('Y-m-d', $date);
    return $d && $d->format('Y-m-d') === $date;
}

// Función para validar rango de fechas
function isValidDateRange($startDate, $endDate) {
    $start = new DateTime($startDate);
    $end = new DateTime($endDate);
    $today = new DateTime();
    
    // La fecha de inicio no puede ser posterior a la fecha final
    if ($start > $end) {
        return ['valid' => false, 'message' => 'La fecha de inicio no puede ser posterior a la fecha final'];
    }
    
    // Las fechas no pueden ser futuras
    if ($start > $today || $end > $today) {
        return ['valid' => false, 'message' => 'No se pueden seleccionar fechas futuras'];
    }
    
    // Limitar el rango máximo a 3 meses (90 días)
    $diff = $start->diff($end);
    if ($diff->days > 365) {
        return ['valid' => false, 'message' => 'El rango máximo permitido es de 365 días'];
    }
    
    return ['valid' => true, 'message' => ''];
}

// Inicializar variable para mensajes de error
$errorMessage = '';

// Inicializar variables de derivaciones para evitar warnings
// Inicializar variables de derivaciones
$derivacionesEnviadas = null;
$derivacionesRealizadas = null;
$errorDerivaciones = null;

// Función para obtener información del estudio
function obtenerInfoEstudio($studyInstanceUID) {
    $url = "https://iset.portalpaciente.com.ar/api/find";
    $data = json_encode([
        "Level" => "Study",
        "Query" => [
            "StudyInstanceUID" => $studyInstanceUID
        ],
        "Expand" => true
    ]);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: application/json"]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        error_log('Error cURL en obtenerInfoEstudio: ' . curl_error($ch));
        curl_close($ch);
        return null;
    }
    
    curl_close($ch);
    
    $jsonData = json_decode($response, true);
    
    if ($jsonData && count($jsonData) > 0) {
        $study = $jsonData[0];
        return [
            'patientName' => str_replace("^", " ", $study['PatientMainDicomTags']['PatientName'] ?? ''),
            'studyDate' => $study['MainDicomTags']['StudyDate'] ?? '',
            'studyDescription' => str_replace("^", " ", $study['MainDicomTags']['StudyDescription'] ?? ''),
            'modality' => $study['MainDicomTags']['Modality'] ?? 'UNKNOWN',
            'studyID' => $study['ID'] ?? ''
        ];
    }
    
    return null;
}

// Función auxiliar para construir el nombre del archivo con formato: nombrepaciente_fechaestudio_modalidad.zip
function construirNombreArchivo($patientName, $studyDate, $modality = 'UNKNOWN') {
    // Limpiar nombre del paciente
    $safePatientName = preg_replace('/[^a-zA-Z0-9_-]/', '_', $patientName);
    
    // Formatear fecha: convertir YYYYMMDD a YYYYMMDD (mantener formato DICOM)
    $fechaFormateada = $studyDate;
    if (strlen($studyDate) == 8) {
        // Ya está en formato YYYYMMDD
        $fechaFormateada = $studyDate;
    } elseif (strtotime($studyDate)) {
        // Convertir a YYYYMMDD si viene en otro formato
        $fechaFormateada = date('Ymd', strtotime($studyDate));
    }
    
    // Limpiar modalidad
    $safeModality = preg_replace('/[^a-zA-Z0-9_-]/', '_', $modality);
    
    // Construir nombre: nombrepaciente_fechaestudio_modalidad.zip
    return "{$safePatientName}_{$fechaFormateada}_{$safeModality}.zip";
}

// Obtener derivaciones desde la base de datos
try {
    $conexionDerivados = new Conexion();
    $conexionDerivados->conectar();
    
    // Verificar que la tabla derivaciones existe
    $checkTable = "SHOW TABLES LIKE 'derivaciones'";
    $tableExists = $conexionDerivados->query($checkTable);
    
    if (!$tableExists || $tableExists->num_rows == 0) {
        error_log('ADMIN: Tabla derivaciones no existe');
        $errorDerivaciones = "Tabla de derivaciones no encontrada";
        $derivacionesEnviadas = null;
        $derivacionesRealizadas = null;
    } else {
        // Consulta para derivaciones enviadas por el admin actual
        $consultaDerivacionesEnviadas = "SELECT d.id, d.study_instance_uid, d.usuario_origen_id, d.usuario_destino_id, 
                                        d.fecha_derivacion, d.activo, d.comentario, 
                                        COALESCE(d.vista, 0) as vista, COALESCE(d.descargada, 0) as descargada,
                                        d.fecha_vista, d.fecha_descarga,
                                        u.nombre as usuario_destino_nombre, u.email as usuario_destino_email 
                                        FROM derivaciones d 
                                        JOIN usuarios u ON d.usuario_destino_id = u.id 
                                        WHERE d.usuario_origen_id = $usuarioId AND d.activo = 1 
                                        ORDER BY d.fecha_derivacion DESC";
        
        $derivacionesEnviadas = $conexionDerivados->query($consultaDerivacionesEnviadas);
        
        if (!$derivacionesEnviadas) {
            error_log('ADMIN: Error en consulta derivaciones enviadas: ' . $conexionDerivados->error());
            $derivacionesEnviadas = null;
        } else {
            error_log('ADMIN: Derivaciones enviadas encontradas: ' . $derivacionesEnviadas->num_rows);
        }
        
        // Consulta para todas las derivaciones del sistema (para admin)
        $consultaDerivacionesRealizadas = "SELECT d.id, d.study_instance_uid, d.usuario_origen_id, d.usuario_destino_id, 
                                         d.fecha_derivacion, d.activo, d.comentario, 
                                         COALESCE(d.vista, 0) as vista, COALESCE(d.descargada, 0) as descargada,
                                         d.fecha_vista, d.fecha_descarga,
                                         u_origen.nombre as usuario_origen_nombre, u_origen.email as usuario_origen_email,
                                         u_destino.nombre as usuario_destino_nombre, u_destino.email as usuario_destino_email
                                         FROM derivaciones d 
                                         JOIN usuarios u_origen ON d.usuario_origen_id = u_origen.id 
                                         JOIN usuarios u_destino ON d.usuario_destino_id = u_destino.id 
                                         WHERE d.activo = 1 
                                         ORDER BY d.fecha_derivacion DESC";
        
        $derivacionesRealizadas = $conexionDerivados->query($consultaDerivacionesRealizadas);
        
        if (!$derivacionesRealizadas) {
            error_log('ADMIN: Error en consulta derivaciones realizadas: ' . $conexionDerivados->error());
            $derivacionesRealizadas = null;
        } else {
            error_log('ADMIN: Derivaciones realizadas encontradas: ' . $derivacionesRealizadas->num_rows);
        }
    }
} catch (Exception $e) {
    error_log('ADMIN: Error en conexión derivaciones: ' . $e->getMessage());
    $errorDerivaciones = $e->getMessage();
    $derivacionesEnviadas = null;
    $derivacionesRealizadas = null;
}

// Manejo de fechas con validación
try {
    if (isset($_POST['startDate']) && isset($_POST['endDate'])) {
        // Validar formato de fechas
        if (!isValidDate($_POST['startDate']) || !isValidDate($_POST['endDate'])) {
            throw new Exception('Formato de fecha inválido. Use el formato YYYY-MM-DD');
        }
        
        // Validar rango de fechas
        $dateValidation = isValidDateRange($_POST['startDate'], $_POST['endDate']);
        if (!$dateValidation['valid']) {
            throw new Exception($dateValidation['message']);
        }
        
        // Si pasa todas las validaciones, actualizar sesión
        $_SESSION['startDate'] = $_POST['startDate'];
        $_SESSION['endDate'] = $_POST['endDate'];
        $startDate = str_replace('-', '', $_POST['startDate']);
        $endDate = str_replace('-', '', $_POST['endDate']);
    } elseif (!isset($_SESSION['startDate']) || !isset($_SESSION['endDate'])) {
        // Valores por defecto (7 días)
        $_SESSION['startDate'] = date('Y-m-d', strtotime('-7 days'));
        $_SESSION['endDate'] = date('Y-m-d');
        $startDate = date('Ymd', strtotime('-7 days'));
        $endDate = date('Ymd');
    } else {
        // Usar valores de sesión existentes
        $startDate = str_replace('-', '', $_SESSION['startDate']);
        $endDate = str_replace('-', '', $_SESSION['endDate']);
    }
} catch (Exception $e) {
    // En caso de error, usar últimas fechas válidas de la sesión o valores por defecto
    if (isset($_SESSION['startDate']) && isset($_SESSION['endDate'])) {
        $startDate = str_replace('-', '', $_SESSION['startDate']);
        $endDate = str_replace('-', '', $_SESSION['endDate']);
    } else {
        $_SESSION['startDate'] = date('Y-m-d', strtotime('-7 days'));
        $_SESSION['endDate'] = date('Y-m-d');
        $startDate = date('Ymd', strtotime('-7 days'));
        $endDate = date('Ymd');
    }
    $errorMessage = "Error: " . $e->getMessage();
}

// Datos de la solicitud
$url = "https://iset.portalpaciente.com.ar/api/find";
$data = json_encode([
    "Level" => "Study",
    "Query" => [
        "StudyDate" => $startDate . "-" . $endDate
    ],
    "Expand" => true
]);

// Inicializar cURL
$ch = curl_init();

// Configurar opciones de cURL
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, 1);
curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    "Content-Type: application/json"
]);

// Deshabilitar verificación SSL (NO RECOMENDADO PARA PRODUCCIÓN)
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);

// Ejecutar la solicitud y obtener la respuesta
$response = curl_exec($ch);

if (curl_errno($ch)) {
    echo 'Error en la solicitud: ' . curl_error($ch);
} else {
    // Decodificar la respuesta JSON
    $jsonData = json_decode($response, true);

    if ($jsonData === null) {
        echo "Error al decodificar el JSON";
    } else {
        // Verificar si hay al menos un estudio en los resultados
        if (count($jsonData) > 0) {
            // Obtener el parámetro de ordenamiento
            $sortColumn = isset($_GET['sort']) ? $_GET['sort'] : 'StudyDate';
            $sortOrder = isset($_GET['order']) ? $_GET['order'] : 'desc';

            // Crear un array para ordenamiento
            $sortArray = array();
            foreach ($jsonData as $key => $study) {
                switch ($sortColumn) {
                    case 'PatientName':
                        $sortArray[$key] = $study['PatientMainDicomTags']['PatientName'];
                        break;
                    case 'PatientID':
                        $sortArray[$key] = $study['PatientMainDicomTags']['PatientID'];
                        break;
                    case 'StudyDescription':
                        $sortArray[$key] = $study['MainDicomTags']['StudyDescription'];
                        break;
                    case 'StudyDate':
                        $sortArray[$key] = $study['MainDicomTags']['StudyDate'];
                        break;                    
                    case 'Modality':
                        $sortArray[$key] = $study['MainDicomTags']['Modality'];
                        break;                    
                    case 'ID':
                        $sortArray[$key] = $study['ID'];
                        break;
                    case 'ReferringPhysician':
                        $sortArray[$key] = $study['MainDicomTags']['ReferringPhysicianName'] ?? '';
                        break;
                    default:
                        $sortArray[$key] = $study['MainDicomTags']['StudyDate'];
                }
            }

            // Ordenar el array
            if ($sortOrder === 'asc') {
                asort($sortArray);
            } else {
                arsort($sortArray);
            }

            // Reordenar $jsonData basado en $sortArray
            $sortedData = array();
            foreach ($sortArray as $key => $value) {
                $sortedData[] = $jsonData[$key];
            }
            $jsonData = $sortedData;

            
            if ($errorMessage) {
                echo '<div style="background-color: #ffebee; color: #c62828; padding: 10px; margin: 10px 0; border-radius: 4px; border: 1px solid #ef9a9a;">
                        <strong>¡Atención!</strong> ' . htmlspecialchars($errorMessage) . '
                      </div>';
            }


            // Generar el HTML
            echo '<!DOCTYPE html>
            <html lang="es">
            <head>
                <meta charset="utf-8">
                <meta name="viewport" content="width=device-width, initial-scale=1">
                <title>ISET-PROFESIONALES</title>
                <style>
                    body {
                         font-family: Arial, sans-serif;
                         line-height: 1.6;
                         color: #333;
                         max-width: 1200px;
                         margin: 0 auto;
                         padding: 20px;
                         background-color: #f4f4f4;
                     }
                     h1, h3 {
                         color: #2c3e50;
                     }
                     .welcome-message {
                         background-color: #3498db;
                         color: white;
                         padding: 10px;
                         border-radius: 5px;
                         margin-bottom: 20px;
                     }
                     .logout-button {
                         background-color: #e74c3c;
                         color: white;
                         border: none;
                         padding: 10px 15px;
                         border-radius: 5px;
                         cursor: pointer;
                         transition: background-color 0.3s;
                     }
                     .logout-button:hover {
                         background-color: #c0392b;
                     }
                     table {
                         width: 100%;
                         border-collapse: separate;
                         border-spacing: 0;
                         background-color: white;
                         border-radius: 10px;
                         overflow: hidden;
                         box-shadow: 0 0 20px rgba(0, 0, 0, 0.1);
                     }
                     th, td {
                         padding: 15px;
                         text-align: left;
                         border-bottom: 1px solid #ecf0f1;
                     }
                     th {
                         background-color: #4CAF50;
                         color: white;
                         font-weight: bold;
                         text-transform: uppercase;
                         letter-spacing: 0.03em;
                         cursor: pointer;
                     }
                     th:hover {
                         background-color: #45a049;
                     }
                     tr:nth-child(even) {
                         background-color: #f9f9f9;
                     }
                     tr:hover {
                         background-color: #f5f5f5;
                     }
                     .actions {
                         display: flex;
                         justify-content: space-around;
                     }
                     .action-button {
                         padding: 8px 12px;
                         border-radius: 5px;
                         text-decoration: none;
                         color: white;
                         font-weight: bold;
                         transition: background-color 0.3s;
                     }
                     .download {
                         background-color: #3498db;
                     }
                     .download:hover {
                         background-color: #2980b9;
                     }
                     .download.error {
                         background-color: #e74c3c;
                     }
                     .download.success {
                         background-color: #2ecc71;
                     }
                     .view {
                         background-color: #3498db;
                     }
                     .view:hover {
                         background-color: #2980b9;
                     }
                     .spinner {
                         display: inline-block;
                         width: 20px;
                         height: 20px;
                         border: 3px solid rgba(255,255,255,.3);
                         border-radius: 50%;
                         border-top-color: #fff;
                         animation: spin 1s ease-in-out infinite;
                         margin-right: 10px;
                     }
                     @keyframes spin {
                         to { transform: rotate(360deg); }
                     }
                     .download:disabled {
                         background-color: #95a5a6;
                         cursor: not-allowed;
                     }
                     #searchInput {
                         width: 100%;
                         padding: 12px 20px;
                         margin: 8px 0;
                         display: inline-block;
                         border: 1px solid #ccc;
                         border-radius: 4px;
                         box-sizing: border-box;
                     }
                     .logo-container {
                         text-align: center;
                         margin-bottom: 20px;
                     }
                     .logo {
                         max-width: 200px;
                         height: auto;
                     }
                    .date-filter {
                        background-color: white;
                        padding: 20px;
                        border-radius: 10px;
                        margin: 20px 0;
                        box-shadow: 0 0 20px rgba(0, 0, 0, 0.1);
                    }
                    .date-filter form {
                        display: flex;
                        gap: 20px;
                        align-items: center;
                    }
                    .date-filter input[type="date"] {
                        padding: 8px;
                        border: 1px solid #ddd;
                        border-radius: 4px;
                    }
                    .filter-button {
                        background-color: #4CAF50;
                        color: white;
                        border: none;
                        padding: 10px 20px;
                        border-radius: 5px;
                        cursor: pointer;
                        transition: background-color 0.3s;
                    }
                    .filter-button:hover {
                        background-color: #45a049;
                    }
                    .header-buttons {
                        display: flex;
                        justify-content: space-between;
                        align-items: center;
                        margin-bottom: 20px;
                    }
                    .left-buttons {
                        display: flex;
                        align-items: center;
                        gap: 10px;
                    }
                    .btn-primary {
                        background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
                        color: white;
                        border: none;
                        padding: 10px 15px;
                        border-radius: 5px;
                        text-decoration: none;
                        display: inline-block;
                        transition: all 0.3s ease;
                        font-weight: 500;
                    }
                    .btn-primary:hover {
                        background: linear-gradient(135deg, #1d4ed8 0%, #1e40af 100%);
                        color: white;
                        text-decoration: none;
                        transform: translateY(-1px);
                        box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
                    }
                    .download-app-button {
                        background-color: #27ae60;
                        color: white;
                        border: none;
                        padding: 10px 15px;
                        border-radius: 5px;
                        cursor: pointer;
                        text-decoration: none;
                        display: inline-block;
                        transition: background-color 0.3s;
                    }
                    .download-app-button:hover {
                        background-color: #229954;
                        color: white;
                        text-decoration: none;
                    }
                    
                    .download-bluesky-button {
                        background-color: #8e44ad;
                        color: white;
                        border: none;
                        padding: 10px 15px;
                        border-radius: 5px;
                        cursor: pointer;
                        text-decoration: none;
                        display: inline-block;
                        transition: background-color 0.3s;
                    }
                    .download-bluesky-button:hover {
                        background-color: #7d3c98;
                    }
                    
                    .download-buttons-container {
                        display: flex;
                        gap: 10px;
                        flex-wrap: wrap;
                    }
                    
                    /* Estilos para paneles plegables */
                    .derivation-panel {
                        transition: all 0.3s ease;
                    }
                    
                    .derivation-title {
                        transition: background-color 0.3s ease;
                    }
                    
                    .derivation-title:hover {
                        background-color: #e9ecef;
                        border-radius: 4px;
                        padding: 4px;
                        margin: -4px;
                    }
                    
                    /* Animación de pulso para notificaciones */
                    @keyframes pulse {
                        0% { transform: scale(1); }
                        50% { transform: scale(1.2); }
                        100% { transform: scale(1); }
                    }
                    
                    /* Estilos para notificaciones */
                    .notification-badge {
                        background: #ff4444;
                        color: white;
                        border-radius: 50%;
                        padding: 2px 6px;
                        font-size: 10px;
                        animation: bounce 2s infinite;
                    }
                    
                    /* Estilos para iconos de estado */
                    .panel-icon {
                        transition: transform 0.3s ease;
                    }
                    
                    .panel-icon:hover {
                        transform: scale(1.1);
                    }
                    
                    /* Estilos responsivos */
                    @media (max-width: 768px) {
                        .derivations-container {
                            flex-direction: column;
                        }
                        
                        .derivation-panel {
                            min-width: 100%;
                        }
                        
                        .download-buttons-container {
                            flex-direction: column;
                            gap: 8px;
                        }
                        
                        .download-app-button,
                        .download-bluesky-button {
                            margin-right: 0;
                            text-align: center;
                            font-size: 14px;
                            padding: 12px 15px;
                        }
                        
                        .header-buttons {
                            flex-direction: column;
                            gap: 15px;
                        }
                    }
                    
                    .derive {
                        background-color: #f39c12;
                    }
                    .derive:hover {
                        background-color: #e67e22;
                    }
                    
                    /* Botón flotante para volver al inicio */
                    .scroll-to-top {
                        position: fixed;
                        bottom: 30px;
                        right: 30px;
                        width: 50px;
                        height: 50px;
                        background: linear-gradient(135deg, #3498db, #2980b9);
                        color: white;
                        border: none;
                        border-radius: 50%;
                        cursor: pointer;
                        box-shadow: 0 4px 12px rgba(52, 152, 219, 0.3);
                        z-index: 1000;
                        opacity: 0;
                        visibility: hidden;
                        transform: translateY(20px);
                        transition: all 0.3s ease;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        font-size: 18px;
                    }
                    
                    .scroll-to-top.show {
                        opacity: 1;
                        visibility: visible;
                        transform: translateY(0);
                    }
                    
                    .scroll-to-top:hover {
                        background: linear-gradient(135deg, #2980b9, #1f5f8b);
                        transform: translateY(-2px);
                        box-shadow: 0 6px 16px rgba(52, 152, 219, 0.4);
                    }
                    
                    .scroll-to-top:active {
                        transform: translateY(0);
                    }
                    
                    /* Animación de pulso suave */
                    @keyframes gentle-pulse {
                        0%, 100% { transform: scale(1); }
                        50% { transform: scale(1.05); }
                    }
                    
                    .scroll-to-top.show {
                        animation: gentle-pulse 2s ease-in-out infinite;
                    }
                    
                    /* Responsive para el botón flotante */
                    @media (max-width: 768px) {
                        .scroll-to-top {
                            bottom: 20px;
                            right: 20px;
                            width: 45px;
                            height: 45px;
                            font-size: 16px;
                        }
                    }
                    
                    /* Estilos para paneles de derivaciones con scroll */
                    .derivation-panel {
                        transition: all 0.3s ease;
                    }
                    
                    .derivation-content {
                        max-height: 0px;
                        overflow: hidden;
                        transition: all 0.3s ease;
                    }
                    
                    .derivation-content.expanded {
                        max-height: 300px; /* Altura fija para mostrar ~3 tarjetas */
                        overflow-y: auto;
                        overflow-x: hidden;
                        scrollbar-width: thin;
                        scrollbar-color: #c1c1c1 #f1f1f1;
                    }
                    
                    /* Estilos para scrollbar webkit (Chrome, Safari, Edge) */
                    .derivation-content.expanded::-webkit-scrollbar {
                        width: 8px;
                    }
                    
                    .derivation-content.expanded::-webkit-scrollbar-track {
                        background: #f1f1f1;
                        border-radius: 4px;
                    }
                    
                    .derivation-content.expanded::-webkit-scrollbar-thumb {
                        background: #c1c1c1;
                        border-radius: 4px;
                    }
                    
                    .derivation-content.expanded::-webkit-scrollbar-thumb:hover {
                        background: #a8a8a8;
                    }
                    
                    /* Estilos para scrollbar Firefox */
                    .derivation-content.expanded {
                        scrollbar-width: thin;
                        scrollbar-color: #c1c1c1 #f1f1f1;
                    }
                    
                    /* Asegurar que las tarjetas de derivación tengan altura consistente */
                    .derivation-item {
                        min-height: 80px;
                        margin-bottom: 8px;
                        padding: 10px;
                        border-radius: 6px;
                        box-shadow: 0 1px 3px rgba(0,0,0,0.1);
                        transition: all 0.2s ease;
                    }
                    
                    .derivation-item:hover {
                        box-shadow: 0 2px 6px rgba(0,0,0,0.15);
                        transform: translateY(-1px);
                    }
                    
                    /* Responsive para paneles de derivaciones */
                    @media (max-width: 768px) {
                        .derivation-content.expanded {
                            max-height: 250px;
                        }
                        
                        .derivation-item {
                            min-height: 70px;
                            padding: 8px;
                        }
                    }
                </style>
                 <script>
                 function downloadAndRename(url, patientName, buttonElement, derivacionId) {
                     buttonElement.disabled = true;
                     buttonElement.innerHTML = "<span class=\'spinner\'></span>Descargando...";
                     buttonElement.classList.remove("error", "success");
 
                     fetch(url)
                         .then(response => {
                             if (!response.ok) {
                                 throw new Error("HTTP error " + response.status);
                             }
                             return response.blob();
                         })
                         .then(blob => {
                             const url = window.URL.createObjectURL(blob);
                             const a = document.createElement("a");
                             a.style.display = "none";
                             a.href = url;
                             a.download = patientName + ".zip";
                             document.body.appendChild(a);
                             a.click();
                             window.URL.revokeObjectURL(url);
                             
                             buttonElement.classList.add("success");
                             buttonElement.innerHTML = "Descarga exitosa";
                             setTimeout(() => {
                                 buttonElement.classList.remove("success");
                                 buttonElement.innerHTML = "Descargar";
                             }, 3000);

                             // Marcar como descargada y vista en el backend si es una derivación
                             if (derivacionId) {
                                 Promise.all([
                                     // Marcar como descargada
                                     fetch("../../controller/derivacionController.php", {
                                         method: "POST",
                                         headers: {
                                             "Content-Type": "application/x-www-form-urlencoded",
                                         },
                                         body: "action=marcar_descargada&derivacion_id=" + derivacionId
                                     }),
                                     // Marcar como vista
                                     fetch("../../controller/derivacionController.php", {
                                         method: "POST",
                                         headers: {
                                             "Content-Type": "application/x-www-form-urlencoded",
                                         },
                                         body: "action=marcar_vista&derivacion_id=" + derivacionId
                                     })
                                 ])
                                 .then(responses => Promise.all(responses.map(r => r.json())))
                                 .then(results => {
                                     const [descargadaResult, vistaResult] = results;
                                     
                                     if (descargadaResult.success) {
                                         console.log("Derivación marcada como descargada:", derivacionId);
                                     } else {
                                         console.error("Error al marcar como descargada:", descargadaResult.message);
                                     }
                                     
                                     if (vistaResult.success) {
                                         console.log("Derivación marcada como vista:", derivacionId);
                                     } else {
                                         console.error("Error al marcar como vista:", vistaResult.message);
                                     }
                                 })
                                 .catch(error => {
                                     console.error("Error al marcar derivación:", error);
                                 });
                             }
                         })
                         .catch(error => {
                             console.error("Error en la descarga:", error);
                             alert("Hubo un error en la descarga. Por favor, inténtelo de nuevo.");
                             
                             buttonElement.classList.add("error");
                             buttonElement.innerHTML = "Error en descarga";
                             setTimeout(() => {
                                 buttonElement.classList.remove("error");
                                 buttonElement.innerHTML = "Descargar";
                             }, 3000);
                         })
                         .finally(() => {
                             buttonElement.disabled = false;
                         });
                 }
 
                 function searchTable() {
                    var input, filter, table, tr, td, i, txtValue;
                    input = document.getElementById("searchInput");
                    filter = input.value.toUpperCase();
                    table = document.querySelector("table");
                    tr = table.getElementsByTagName("tr");
                    var visibleCount = 0;
                    for (i = 0; i < tr.length; i++) {
                        td = tr[i].getElementsByTagName("td");
                        if (td.length > 0) { // Asegurarse de que no es una fila de encabezado
                            var rowVisible = false;
                            for (var j = 0; j < td.length; j++) {
                                if (td[j]) {
                                    txtValue = td[j].textContent || td[j].innerText;
                                    if (txtValue.toUpperCase().indexOf(filter) > -1) {
                                        rowVisible = true;
                                        break;
                                    }
                                }
                            }
                            if (rowVisible) {
                                tr[i].style.display = "";
                                visibleCount++;
                            } else {
                                tr[i].style.display = "none";
                            }
                        }
                    }
                    // Actualizar el contador de estudios
                    document.getElementById("studyCount").textContent = visibleCount;
                }
                 </script>
             </head>
            <body>
                <div class="logo-container">
                    <img src="isettacselogotransparent.png" alt="Logo ISET" class="logo">
                </div>

                <div class="welcome-message">
                    <h1>Hola estimado colega ' . strtoupper($_SESSION['nombre']) . '</h1>
                    <p>Estos son los estudios de tus pacientes, solicitados a ISET:</p>
                </div>
                
                <div class="date-filter">
                    <form method="post">
                        <label>
                            Fecha inicio:
                            <input type="date" 
                                name="startDate" 
                                value="' . htmlspecialchars($_SESSION['startDate']) . '" 
                                max="' . date('Y-m-d') . '"
                                required>
                        </label>
                        <label>
                            Fecha fin:
                            <input type="date" 
                                name="endDate" 
                                value="' . htmlspecialchars($_SESSION['endDate']) . '" 
                                max="' . date('Y-m-d') . '"
                                required>
                        </label>
                        <button type="submit" class="filter-button">Filtrar</button>
                    </form>
                </div>

                <div class="header-buttons">
                    <div class="left-buttons">
                        <a href="user_management.php" class="btn btn-primary" style="background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%); color: white; border: none; padding: 10px 15px; border-radius: 5px; text-decoration: none; display: inline-block; margin-right: 10px; transition: all 0.3s ease;">
                            <i class="fa fa-users"></i> Administrar Usuarios
                        </a>
                        <a href="../../controller/cerrarSesion.php">
                            <button class="logout-button" type="button">Cerrar sesión</button>
                        </a>
                    </div>
                    <div class="download-buttons-container">
                        <a href="../../controller/downloadApp.php" class="download-app-button">
                            📱 Descargar OnDemand 3D Communicator
                        </a>
                        <a href="../../controller/downloadBlueSky.php" class="download-bluesky-button">
                            🧠 Descargar BlueSkyPlan
                        </a>
                    </div>
                </div>';

            // Mostrar mensaje de error si existe
            if ($errorDerivaciones) {
                echo '<div style="background: #f8d7da; color: #721c24; padding: 10px; border-radius: 5px; margin-bottom: 15px;">' 
                     . '⚠️ Error en derivaciones: ' . htmlspecialchars($errorDerivaciones) . '</div>';
            }
            
            // Panel de Derivaciones (solo si no hay errores)
            if (!$errorDerivaciones) {
                echo '<!-- Panel de Derivaciones -->
                <div class="derivations-container" style="display: flex; gap: 15px; margin-bottom: 20px; flex-wrap: wrap;">
                <!-- Derivaciones Enviadas -->
                <div class="derivation-panel" style="flex: 1; min-width: 300px; background: #f8f9fa; border-radius: 8px; padding: 15px; border: 1px solid #dee2e6;">
                    <div class="derivation-title" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; font-weight: bold; color: #495057; cursor: pointer;" onclick="togglePanel(\'enviadas\')">
                        <span>📤 Derivaciones Enviadas <span id="contador-enviadas" style="background: #007bff; color: white; padding: 2px 8px; border-radius: 12px; font-size: 11px; font-weight: normal;">' . ($derivacionesEnviadas ? $derivacionesEnviadas->num_rows : 0) . '</span></span>
                        <div style="display: flex; align-items: center; gap: 8px;">
                            <span id="icono-enviadas" style="font-size: 16px;">📤</span>
                            <button onclick="event.stopPropagation(); refrescarManual()" style="background: none; border: none; color: #6c757d; cursor: pointer; font-size: 14px;" title="Refrescar">🔄</button>
                        </div>
                    </div>
                    <div id="contenido-derivaciones-enviadas" class="derivation-content">';

            if ($derivacionesEnviadas && $derivacionesEnviadas->num_rows > 0) {
                error_log('DEBUG: Mostrando ' . $derivacionesEnviadas->num_rows . ' derivaciones enviadas');
                while ($derivacion = $derivacionesEnviadas->fetch_assoc()) {
                    $infoEstudio = obtenerInfoEstudio($derivacion['study_instance_uid']);
                    $patientName = $infoEstudio['patientName'] ?? 'No disponible';
                    $studyDate = $infoEstudio['studyDate'] ?? 'No disponible';
                    $studyDescription = $infoEstudio['studyDescription'] ?? 'No disponible';
                    
                    echo '<div class="derivation-item" style="background: white; border: 1px solid #dee2e6; font-size: 11px;">
                        <div style="font-weight: bold; color: #495057; margin-bottom: 2px;">👤 ' . htmlspecialchars($patientName) . '</div>
                        <div style="color: #6c757d; margin-bottom: 2px;">📅 ' . ($studyDate ? date('d/m/Y', strtotime($studyDate)) : 'No disponible') . ' | ' . htmlspecialchars($studyDescription) . '</div>
                        <div style="color: #6c757d; margin-bottom: 2px;">👤 Para: ' . htmlspecialchars($derivacion['usuario_destino_nombre']) . '</div>
                        <div style="color: #6c757d; margin-bottom: 3px;">📅 Derivado: ' . date('d/m/Y H:i', strtotime($derivacion['fecha_derivacion'])) . '</div>';
                    
                    if (!empty($derivacion['comentario'])) {
                        echo '<div style="color: #28a745; font-style: italic; margin-bottom: 3px;">💬 ' . htmlspecialchars($derivacion['comentario']) . '</div>';
                    }
                    
                    echo '<button onclick="desactivarDerivacion(' . $derivacion['id'] . ')" 
                            style="background: #dc3545; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 10px; cursor: pointer;">
                            ❌ Cancelar
                        </button>
                    </div>';
                }
            } else {
                error_log('DEBUG: No hay derivaciones enviadas para mostrar');
                echo '<p style="color: #6c757d; font-style: italic; font-size: 12px;">No has enviado derivaciones</p>';
            }

            echo '</div>
                </div>

                <!-- Derivaciones Realizadas (Global del Sistema) -->
                <div class="derivation-panel" style="flex: 1; min-width: 300px; background: #f8f9fa; border-radius: 8px; padding: 15px; border: 1px solid #dee2e6;">
                    <div class="derivation-title" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; font-weight: bold; color: #495057; cursor: pointer;" onclick="togglePanel(\'realizadas\')">
                        <span>🌐 Derivaciones Realizadas <span id="contador-realizadas" style="background: #6f42c1; color: white; padding: 2px 8px; border-radius: 12px; font-size: 11px; font-weight: normal;">' . ($derivacionesRealizadas ? $derivacionesRealizadas->num_rows : 0) . '</span></span>
                        <div style="display: flex; align-items: center; gap: 8px;">
                            <span id="icono-realizadas" style="font-size: 16px;">🌐</span>
                            <button onclick="event.stopPropagation(); refrescarManual()" style="background: none; border: none; color: #6c757d; cursor: pointer; font-size: 14px;" title="Refrescar">🔄</button>
                        </div>
                    </div>
                    <div id="contenido-derivaciones-realizadas" class="derivation-content">';

            if ($derivacionesRealizadas && $derivacionesRealizadas->num_rows > 0) {
                error_log('DEBUG: Mostrando ' . $derivacionesRealizadas->num_rows . ' derivaciones realizadas');
                while ($derivacion = $derivacionesRealizadas->fetch_assoc()) {
                    $infoEstudio = obtenerInfoEstudio($derivacion['study_instance_uid']);
                    $patientName = $infoEstudio['patientName'] ?? 'No disponible';
                    $studyDate = $infoEstudio['studyDate'] ?? 'No disponible';
                    $studyDescription = $infoEstudio['studyDescription'] ?? 'No disponible';
                    
                    // Determinar el estado y color
                    $haSidoVista = isset($derivacion['vista']) ? $derivacion['vista'] == 1 : false;
                    $haSidoDescargada = isset($derivacion['descargada']) ? $derivacion['descargada'] == 1 : false;
                    
                    if ($haSidoVista || $haSidoDescargada) {
                        $colorBorde = '#28a745'; // Verde para vista/descargada
                        $colorFondo = '#d4edda';
                        $indicadorEstado = '✅';
                        $textoEstado = $haSidoVista ? 'Vista' : 'Descargada';
                    } else {
                        $colorBorde = '#ffc107'; // Amarillo para nueva
                        $colorFondo = '#fff3cd';
                        $indicadorEstado = '🆕';
                        $textoEstado = 'Nueva';
                    }
                    
                    echo '<div class="derivation-item" style="background: ' . $colorFondo . '; border: 2px solid ' . $colorBorde . '; font-size: 11px; position: relative;">
                        <div style="font-weight: bold; color: #495057; margin-bottom: 2px;">👤 ' . htmlspecialchars($patientName) . '</div>
                        <div style="color: #6c757d; margin-bottom: 2px;">📅 ' . ($studyDate ? date('d/m/Y', strtotime($studyDate)) : 'No disponible') . ' | ' . htmlspecialchars($studyDescription) . '</div>
                        <div style="color: #6c757d; margin-bottom: 2px;">📤 De: ' . htmlspecialchars($derivacion['usuario_origen_nombre']) . ' → 📥 Para: ' . htmlspecialchars($derivacion['usuario_destino_nombre']) . '</div>
                        <div style="color: #6c757d; margin-bottom: 2px;">📅 Derivado: ' . date('d/m/Y H:i', strtotime($derivacion['fecha_derivacion'])) . ' | ' . $indicadorEstado . ' ' . $textoEstado . '</div>';
                    
                    if (!empty($derivacion['comentario'])) {
                        echo '<div style="color: #28a745; font-style: italic; margin-bottom: 3px;">💬 ' . htmlspecialchars($derivacion['comentario']) . '</div>';
                    }
                    
                    // Solo mostrar botón de cancelar si es el administrador actual
                    if ($derivacion['usuario_origen_id'] == $usuarioId) {
                        echo '<button onclick="desactivarDerivacion(' . $derivacion['id'] . ')" 
                                style="background: #dc3545; color: white; border: none; padding: 2px 6px; border-radius: 3px; font-size: 10px; cursor: pointer;">
                                ❌ Cancelar
                            </button>';
                    }
                    
                    echo '</div>';
                }
            } else {
                error_log('DEBUG: No hay derivaciones realizadas para mostrar');
                echo '<p style="color: #6c757d; font-style: italic; font-size: 12px;">No hay derivaciones realizadas en el sistema</p>';
            }

            echo '</div>
                </div>
            </div>';
            
            } else {
                echo '<div style="background: #fff3cd; color: #856404; padding: 15px; border-radius: 5px; margin-bottom: 20px;">' 
                     . '📋 Los paneles de derivaciones no están disponibles temporalmente.</div>';
            }

            echo '<h3>Estudios: <span id="studyCount">' . count($jsonData) . '</span> encontrados</h3>
                <input type="text" id="searchInput" onkeyup="searchTable()" placeholder="Buscar en la lista...">
                <table>
                    <thead>
                        <tr>
                            <th><a href="?sort=PatientName&order=' . ($sortColumn == 'PatientName' && $sortOrder == 'asc' ? 'desc' : 'asc') . '">NOMBRE</a></th>
                            <th><a href="?sort=PatientID&order=' . ($sortColumn == 'PatientID' && $sortOrder == 'asc' ? 'desc' : 'asc') . '">ID</a></th>
                            <th><a href="?sort=StudyDescription&order=' . ($sortColumn == 'StudyDescription' && $sortOrder == 'asc' ? 'desc' : 'asc') . '">DESCRIPCIÓN</a></th>                            
                            <th><a href="?sort=StudyDate&order=' . ($sortColumn == 'StudyDate' && $sortOrder == 'asc' ? 'desc' : 'asc') . '">FECHA</a></th>
                            <th><a href="?sort=ReferringPhysician&order=' . ($sortColumn == 'ReferringPhysician' && $sortOrder == 'asc' ? 'desc' : 'asc') . '">PROFESIONAL</a></th>
                            <th>ACCIONES</th>
                        </tr>
                    </thead>
                    <tbody>';

            foreach ($jsonData as $study) {
                $patientName = str_replace("^", " ", $study['PatientMainDicomTags']['PatientName'] ?? '');
                $patientid = str_replace("^", " ", $study['PatientMainDicomTags']['PatientID'] ?? '');
                $studydescription = str_replace("^", " ", $study['MainDicomTags']['StudyDescription'] ?? '');
                $referringPhysician = str_replace("^", " ", $study['MainDicomTags']['ReferringPhysicianName'] ?? 'No especificado');
                $studyInstanceUID = $study['MainDicomTags']['StudyInstanceUID'];                
                $studyID = $study['ID'];
                $studyDate = $study['MainDicomTags']['StudyDate'] ?? '';
                $modality = $study['MainDicomTags']['Modality'] ?? 'UNKNOWN';
                $filename = construirNombreArchivo($patientName, $studyDate, $modality);
                $downloadUrl = "https://iset.portalpaciente.com.ar/api/isetarc/{$studyInstanceUID}?accept=application/zip&filename=" . urlencode($filename);
                
                if (stripos($studydescription, 'Cone') !== false) {
                    $viewUrl = "https://iset.portalpaciente.com.ar/volview/index.html?names=[archive.zip]&urls=[../studies/{$studyID}/archive]";
                } else {
                    $viewUrl = "view.php?study=" . urlencode($study['MainDicomTags']['StudyInstanceUID']);
                }

                echo "<tr>
                        <td>" . htmlspecialchars($patientName) . "</td>
                        <td>" . htmlspecialchars($patientid) . "</td>
                        <td>" . htmlspecialchars($studydescription ?: 'No disponible') . "</td>
                        <td>" . date("d/m/Y", strtotime($study['MainDicomTags']['StudyDate'])) . "</td>
                        <td>" . htmlspecialchars($referringPhysician) . "</td>
                        <td class='actions'>
                            <button onclick='downloadAndRename(\"" . htmlspecialchars($downloadUrl) . "\", \"" . htmlspecialchars($safePatientName) . "\", this)' class='action-button download'>Descargar</button>
                            
                            " . (!isset($study['es_derivado']) ? '<button onclick="mostrarModalDerivarMejorado(\'' . htmlspecialchars($studyInstanceUID) . '\', \'' . htmlspecialchars($patientName) . '\', \'' . htmlspecialchars($studydescription) . '\', \'' . $study['MainDicomTags']['StudyDate'] . '\')" class="action-button derive">Derivar</button>' : '') . "
                            
                            <a href='" . htmlspecialchars($viewUrl) . "' target='_blank' rel='noopener noreferrer' class='action-button view'>Visualizar</a>
                        </td>
                      </tr>";
            }

            echo '</tbody>
                </table>';
                
                // Incluir el modal mejorado de derivación
                include 'derivacion_modal.php';
                
                echo '<script>
                // Función mejorada para mostrar el modal de derivación con información del estudio
                function mostrarModalDerivarMejorado(studyUID, patientName, studyDescription, studyDate) {
                    const estudioInfo = {
                        patientName: patientName,
                        studyDescription: studyDescription,
                        studyDate: studyDate ? new Date(studyDate).toLocaleDateString("es-ES") : "No disponible"
                    };
                    
                    // Llamar a la función del modal mejorado
                    mostrarModalDerivar(studyUID, estudioInfo);
                }
                
                // Funciones para los paneles de derivaciones
                function togglePanel(tipo) {
                    const contenido = document.getElementById("contenido-derivaciones-" + tipo);
                    const icono = document.getElementById("icono-" + tipo);
                    
                    if (contenido.classList.contains("expanded")) {
                        // Cerrar panel
                        contenido.classList.remove("expanded");
                        contenido.style.maxHeight = "0px";
                        icono.textContent = tipo === "enviadas" ? "📤" : tipo === "realizadas" ? "🌐" : "📥";
                        localStorage.setItem("panel-" + tipo + "-expanded", "false");
                    } else {
                        // Abrir panel
                        contenido.classList.add("expanded");
                        contenido.style.maxHeight = "300px"; // Altura fija para ~3 tarjetas
                        icono.textContent = tipo === "enviadas" ? "📤" : tipo === "realizadas" ? "🌐" : "📥";
                        localStorage.setItem("panel-" + tipo + "-expanded", "true");
                    }
                }
                
                function refrescarManual() {
                    location.reload();
                }
                
                function desactivarDerivacion(derivacionId) {
                    if (confirm("¿Estás seguro de que deseas cancelar esta derivación?")) {
                        fetch("../../controller/derivacionController.php", {
                            method: "POST",
                            headers: {
                                "Content-Type": "application/x-www-form-urlencoded",
                            },
                            body: "action=desactivar&derivacion_id=" + derivacionId
                        })
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                alert("Derivación cancelada exitosamente");
                                location.reload();
                            } else {
                                alert("Error al cancelar: " + data.message);
                            }
                        })
                        .catch(error => {
                            alert("Error de conexión. Por favor, inténtelo de nuevo.");
                        });
                    }
                }
                
                function marcarComoVista(derivacionId) {
                    fetch("../../controller/derivacionController.php", {
                        method: "POST",
                        headers: {
                            "Content-Type": "application/x-www-form-urlencoded",
                        },
                        body: "action=marcar_vista&derivacion_id=" + derivacionId
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            console.log("Derivación marcada como vista exitosamente");
                        } else {
                            console.error("Error al marcar como vista:", data.message);
                        }
                    })
                    .catch(error => {
                        console.error("Error en la petición:", error);
                    });
                }
                
                // Auto-expandir paneles si hay derivaciones
                document.addEventListener("DOMContentLoaded", function() {
                    const contadorEnviadas = document.getElementById("contador-enviadas");
                    const contadorRealizadas = document.getElementById("contador-realizadas");
                    
                    if (contadorEnviadas && parseInt(contadorEnviadas.textContent) > 0) {
                        const contenidoEnviadas = document.getElementById("contenido-derivaciones-enviadas");
                        contenidoEnviadas.classList.add("expanded");
                        contenidoEnviadas.style.maxHeight = "300px";
                    }
                    
                    if (contadorRealizadas && parseInt(contadorRealizadas.textContent) > 0) {
                        const contenidoRealizadas = document.getElementById("contenido-derivaciones-realizadas");
                        contenidoRealizadas.classList.add("expanded");
                        contenidoRealizadas.style.maxHeight = "300px";
                    }
                });
                
                // Funcionalidad del botón flotante para volver al inicio
                function scrollToTop() {
                    window.scrollTo({
                        top: 0,
                        behavior: "smooth"
                    });
                }
                
                // Controlar visibilidad del botón flotante
                function toggleScrollButton() {
                    const scrollBtn = document.getElementById("scrollToTopBtn");
                    if (window.pageYOffset > 300) { // Mostrar después de 300px de scroll
                        scrollBtn.classList.add("show");
                    } else {
                        scrollBtn.classList.remove("show");
                    }
                }
                
                // Event listeners para el botón flotante
                window.addEventListener("scroll", toggleScrollButton);
                window.addEventListener("load", toggleScrollButton);
                </script>
                
                <!-- Botón flotante para volver al inicio -->
                <button id="scrollToTopBtn" class="scroll-to-top" title="Volver al inicio" onclick="scrollToTop()">
                    ↑
                </button>
            </body>
            </html>';
        } else {
            echo "No se encontraron estudios para el rango de fechas seleccionado.";
        }
    }
}
// Cerrar cURL
curl_close($ch);

?>