<?php
/**
 * Modal mejorado para derivación de estudios
 * Incluye filtrado en tiempo real de usuarios
 */

// Asegurar que solo se acceda desde el contexto correcto
if (!defined('ADMIN_ACCESS')) {
    die('Acceso denegado');
}
?>

<!-- CSS específico para el modal de derivación -->
<style>
.modal-derivar {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    z-index: 1000;
    animation: fadeIn 0.3s ease;
}

.modal-derivar.show {
    display: flex;
    align-items: center;
    justify-content: center;
}

.modal-content-derivar {
    background: white;
    padding: 25px;
    border-radius: 12px;
    min-width: 450px;
    max-width: 600px;
    max-height: 90vh;
    overflow-y: auto;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
    animation: slideIn 0.3s ease;
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    padding-bottom: 15px;
    border-bottom: 2px solid #e5e7eb;
}

.modal-title {
    color: #1f2937;
    font-size: 20px;
    font-weight: 600;
    margin: 0;
}

.close-button {
    background: none;
    border: none;
    font-size: 24px;
    color: #6b7280;
    cursor: pointer;
    padding: 5px;
    line-height: 1;
    transition: color 0.2s ease;
}

.close-button:hover {
    color: #dc2626;
}

.form-group {
    margin-bottom: 20px;
}

.form-label {
    display: block;
    font-weight: 600;
    color: #374151;
    margin-bottom: 8px;
    font-size: 14px;
}

.form-input, .form-textarea, .form-select {
    width: 100%;
    padding: 12px;
    border: 2px solid #d1d5db;
    border-radius: 8px;
    font-size: 14px;
    transition: border-color 0.2s ease, box-shadow 0.2s ease;
    box-sizing: border-box;
}

.form-input:focus, .form-textarea:focus, .form-select:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.search-input {
    position: relative;
}

.search-input::before {
    content: "🔍";
    position: absolute;
    left: 12px;
    top: 50%;
    transform: translateY(-50%);
    color: #9ca3af;
    z-index: 1;
}

.search-input input {
    padding-left: 40px;
}

.users-list {
    max-height: 200px;
    overflow-y: auto;
    border: 2px solid #d1d5db;
    border-radius: 8px;
    background: white;
    margin-top: 5px;
}

.user-item {
    padding: 12px 16px;
    cursor: pointer;
    transition: background-color 0.2s ease;
    border-bottom: 1px solid #f3f4f6;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.user-item:hover {
    background-color: #f9fafb;
}

.user-item.selected {
    background-color: #dbeafe;
    border-color: #3b82f6;
}

.user-item:last-child {
    border-bottom: none;
}

.user-info {
    flex: 1;
}

.user-name {
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 2px;
}

.user-email {
    font-size: 12px;
    color: #6b7280;
}

.user-badge {
    background: #10b981;
    color: white;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 10px;
    font-weight: 600;
}

.no-users {
    padding: 20px;
    text-align: center;
    color: #6b7280;
    font-style: italic;
}

.modal-actions {
    display: flex;
    justify-content: flex-end;
    gap: 12px;
    margin-top: 25px;
    padding-top: 20px;
    border-top: 1px solid #e5e7eb;
}

.btn {
    padding: 12px 24px;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    font-size: 14px;
}

.btn-cancel {
    background: #f3f4f6;
    color: #374151;
}

.btn-cancel:hover {
    background: #e5e7eb;
}

.btn-submit {
    background: #f59e0b;
    color: white;
}

.btn-submit:hover {
    background: #d97706;
}

.btn-submit:disabled {
    background: #9ca3af;
    cursor: not-allowed;
}

.loading-spinner {
    display: inline-block;
    width: 16px;
    height: 16px;
    border: 2px solid #ffffff;
    border-radius: 50%;
    border-top-color: transparent;
    animation: spin 1s ease-in-out infinite;
    margin-right: 8px;
}

.selected-user-info {
    background: #ecfdf5;
    border: 1px solid #10b981;
    border-radius: 8px;
    padding: 12px;
    margin-top: 10px;
    display: none;
}

.selected-user-info.show {
    display: block;
}

.selected-user-name {
    font-weight: 600;
    color: #065f46;
    margin-bottom: 4px;
}

.selected-user-email {
    color: #047857;
    font-size: 13px;
}

@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}

@keyframes slideIn {
    from { 
        opacity: 0;
        transform: scale(0.9) translateY(-20px);
    }
    to { 
        opacity: 1;
        transform: scale(1) translateY(0);
    }
}

@keyframes spin {
    to { transform: rotate(360deg); }
}

/* Responsive */
@media (max-width: 768px) {
    .modal-content-derivar {
        min-width: 90%;
        margin: 10px;
        padding: 20px;
    }
    
    .modal-actions {
        flex-direction: column;
    }
    
    .btn {
        width: 100%;
    }
}
</style>

<!-- Modal para derivar estudio -->
<div id="modalDerivar" class="modal-derivar">
    <div class="modal-content-derivar">
        <div class="modal-header">
            <h3 class="modal-title">📤 Derivar Estudio</h3>
            <button type="button" class="close-button" onclick="cerrarModalDerivar()">&times;</button>
        </div>
        
        <form id="formDerivar">
            <input type="hidden" id="studyUID" name="study_uid">
            
            <!-- Información del estudio -->
            <div class="form-group">
                <div id="estudioInfo" style="background: #f8f9fa; padding: 12px; border-radius: 8px; margin-bottom: 15px; display: none;">
                    <div style="font-weight: 600; color: #495057; margin-bottom: 4px;">📋 Información del Estudio</div>
                    <div id="estudioDetalles" style="color: #6c757d; font-size: 13px;"></div>
                </div>
            </div>
            
            <!-- Búsqueda de usuarios -->
            <div class="form-group">
                <label class="form-label">🔍 Buscar Usuario Destino</label>
                <div class="search-input">
                    <input 
                        type="text" 
                        id="searchUsuario" 
                        class="form-input"
                        placeholder="Escriba el nombre o email del usuario..." 
                        oninput="filtrarUsuarios(this.value)"
                        autocomplete="off">
                </div>
                
                <!-- Lista de usuarios filtrada -->
                <div id="usersListContainer" class="users-list" style="display: none;">
                    <div id="usersList"></div>
                </div>
                
                <!-- Usuario seleccionado -->
                <div id="selectedUserInfo" class="selected-user-info">
                    <div class="selected-user-name" id="selectedUserName"></div>
                    <div class="selected-user-email" id="selectedUserEmail"></div>
                </div>
                
                <!-- Select oculto para el formulario -->
                <select id="usuarioDestino" name="usuario_destino" style="display: none;">
                    <option value="">Seleccionar usuario...</option>
                </select>
            </div>
            
            <!-- Comentario -->
            <div class="form-group">
                <label class="form-label">💬 Comentario (opcional)</label>
                <textarea 
                    id="comentario" 
                    name="comentario" 
                    class="form-textarea"
                    placeholder="Agregue un comentario sobre la derivación..."
                    rows="3"
                    maxlength="500"></textarea>
                <div style="font-size: 12px; color: #6b7280; text-align: right; margin-top: 4px;">
                    <span id="comentarioCounter">0</span>/500 caracteres
                </div>
            </div>
            
            <!-- Acciones -->
            <div class="modal-actions">
                <button type="button" class="btn btn-cancel" onclick="cerrarModalDerivar()">
                    ❌ Cancelar
                </button>
                <button type="submit" class="btn btn-submit" id="btnDerivar" disabled>
                    <span id="derivarText">📤 Derivar Estudio</span>
                </button>
            </div>
        </form>
    </div>
</div>

<script>
// Variables globales para la funcionalidad de derivación
let usuariosDisponibles = [];
let usuariosFiltrados = [];
let usuarioSeleccionado = null;
let isLoading = false;

// Cargar usuarios al iniciar la página
document.addEventListener('DOMContentLoaded', function() {
    cargarUsuarios();
});

// Función para cargar usuarios disponibles
function cargarUsuarios() {
    fetch("../../controller/derivacionController.php", {
        method: "POST",
        headers: {"Content-Type": "application/x-www-form-urlencoded"},
        body: "action=obtener_usuarios"
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            usuariosDisponibles = data.usuarios;
            usuariosFiltrados = [...usuariosDisponibles];
            console.log('Usuarios cargados:', usuariosDisponibles.length);
        } else {
            console.error('Error al cargar usuarios:', data.message);
            mostrarNotificacion('Error al cargar la lista de usuarios', 'error');
        }
    })
    .catch(error => {
        console.error('Error en la petición de usuarios:', error);
        mostrarNotificacion('Error de conexión al cargar usuarios', 'error');
    });
}

// Función para mostrar el modal de derivación
function mostrarModalDerivar(studyUID, estudioInfo = null) {
    if (isLoading) return;
    
    // Limpiar formulario anterior
    document.getElementById("studyUID").value = studyUID;
    document.getElementById("searchUsuario").value = "";
    document.getElementById("comentario").value = "";
    document.getElementById("usuarioDestino").value = "";
    usuarioSeleccionado = null;
    
    // Mostrar información del estudio si está disponible
    if (estudioInfo) {
        const estudioInfoDiv = document.getElementById("estudioInfo");
        const estudioDetalles = document.getElementById("estudioDetalles");
        estudioDetalles.innerHTML = `
            <strong>Paciente:</strong> ${estudioInfo.patientName || 'No disponible'}<br>
            <strong>Fecha:</strong> ${estudioInfo.studyDate || 'No disponible'}<br>
            <strong>Descripción:</strong> ${estudioInfo.studyDescription || 'No disponible'}
        `;
        estudioInfoDiv.style.display = 'block';
    }
    
    // Ocultar lista de usuarios y reset
    document.getElementById("usersListContainer").style.display = "none";
    document.getElementById("selectedUserInfo").classList.remove("show");
    updateSubmitButton();
    
    // Mostrar modal con animación
    const modal = document.getElementById("modalDerivar");
    modal.classList.add("show");
    
    // Enfocar el campo de búsqueda
    setTimeout(() => {
        document.getElementById("searchUsuario").focus();
    }, 100);
}

// Función para filtrar usuarios en tiempo real
function filtrarUsuarios(searchTerm) {
    const searchInput = document.getElementById("searchUsuario");
    const usersListContainer = document.getElementById("usersListContainer");
    const usersList = document.getElementById("usersList");
    
    if (!searchTerm || searchTerm.trim() === "") {
        usersListContainer.style.display = "none";
        usuariosFiltrados = [];
        return;
    }
    
    const term = searchTerm.toLowerCase().trim();
    usuariosFiltrados = usuariosDisponibles.filter(usuario => {
        const nombre = usuario.nombre ? usuario.nombre.toLowerCase() : "";
        const email = usuario.email ? usuario.email.toLowerCase() : "";
        return nombre.includes(term) || email.includes(term);
    });
    
    // Mostrar lista si hay resultados
    if (usuariosFiltrados.length > 0) {
        renderUsersList(usuariosFiltrados);
        usersListContainer.style.display = "block";
    } else {
        usersList.innerHTML = '<div class="no-users">No se encontraron usuarios que coincidan con la búsqueda</div>';
        usersListContainer.style.display = "block";
    }
}

// Función para renderizar la lista de usuarios
function renderUsersList(usuarios) {
    const usersList = document.getElementById("usersList");
    
    usersList.innerHTML = usuarios.map(usuario => `
        <div class="user-item" onclick="seleccionarUsuario(${usuario.id}, '${escapeHtml(usuario.nombre)}', '${escapeHtml(usuario.email)}')">
            <div class="user-info">
                <div class="user-name">${escapeHtml(usuario.nombre)}</div>
                <div class="user-email">${escapeHtml(usuario.email)}</div>
            </div>
            <div class="user-badge">Seleccionar</div>
        </div>
    `).join('');
}

// Función para seleccionar un usuario
function seleccionarUsuario(userId, userName, userEmail) {
    usuarioSeleccionado = {
        id: userId,
        nombre: userName,
        email: userEmail
    };
    
    // Actualizar campo de búsqueda
    document.getElementById("searchUsuario").value = userName;
    
    // Ocultar lista
    document.getElementById("usersListContainer").style.display = "none";
    
    // Mostrar información del usuario seleccionado
    document.getElementById("selectedUserName").textContent = userName;
    document.getElementById("selectedUserEmail").textContent = userEmail;
    document.getElementById("selectedUserInfo").classList.add("show");
    
    // Actualizar select oculto
    const select = document.getElementById("usuarioDestino");
    select.value = userId;
    
    // Actualizar botón de envío
    updateSubmitButton();
}

// Función para actualizar el estado del botón de envío
function updateSubmitButton() {
    const btn = document.getElementById("btnDerivar");
    const hasUser = usuarioSeleccionado && usuarioSeleccionado.id;
    
    btn.disabled = !hasUser || isLoading;
    
    if (hasUser && !isLoading) {
        btn.classList.remove("btn-disabled");
    } else {
        btn.classList.add("btn-disabled");
    }
}

// Función para cerrar el modal
function cerrarModalDerivar() {
    if (isLoading) return;
    
    const modal = document.getElementById("modalDerivar");
    modal.classList.remove("show");
    
    // Limpiar formulario
    setTimeout(() => {
        document.getElementById("formDerivar").reset();
        document.getElementById("searchUsuario").value = "";
        document.getElementById("usersListContainer").style.display = "none";
        document.getElementById("selectedUserInfo").classList.remove("show");
        document.getElementById("estudioInfo").style.display = "none";
        usuarioSeleccionado = null;
        // Limpiar el campo oculto también
        document.getElementById("usuarioDestino").value = "";
        updateSubmitButton();
    }, 300);
}

// Función auxiliar para escape HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Contador de caracteres para comentario
document.getElementById("comentario").addEventListener('input', function() {
    const counter = document.getElementById("comentarioCounter");
    counter.textContent = this.value.length;
    
    if (this.value.length > 450) {
        counter.style.color = '#dc2626';
    } else {
        counter.style.color = '#6b7280';
    }
});

// Manejo del envío del formulario
document.getElementById("formDerivar").addEventListener("submit", function(e) {
    e.preventDefault();
    
    // Validación manual del usuario seleccionado
    if (!usuarioSeleccionado || !usuarioSeleccionado.id) {
        mostrarNotificacion('Por favor, seleccione un usuario destino antes de continuar', 'error');
        // Enfocar el campo de búsqueda
        document.getElementById("searchUsuario").focus();
        return;
    }
    
    if (isLoading) return;
    
    isLoading = true;
    const btn = document.getElementById("btnDerivar");
    const btnText = document.getElementById("derivarText");
    
    // Cambiar estado del botón
    btn.disabled = true;
    btnText.innerHTML = '<span class="loading-spinner"></span>Derivando...';
    
    const studyUID = document.getElementById("studyUID").value;
    const comentario = document.getElementById("comentario").value;
    
    const body = `action=derivar&study_uid=${encodeURIComponent(studyUID)}&usuario_destino=${encodeURIComponent(usuarioSeleccionado.id)}&comentario=${encodeURIComponent(comentario)}`;
    
    fetch("../../controller/derivacionController.php", {
        method: "POST",
        headers: {
            "Content-Type": "application/x-www-form-urlencoded",
        },
        body: body
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            mostrarNotificacion('Estudio derivado exitosamente a ' + usuarioSeleccionado.nombre, 'success');
            cerrarModalDerivar();
            
            // Recargar la página después de un breve delay
            setTimeout(() => {
                location.reload();
            }, 1500);
        } else {
            mostrarNotificacion('Error al derivar: ' + data.message, 'error');
        }
    })
    .catch(error => {
        console.error('Error en derivación:', error);
        mostrarNotificacion('Error de conexión. Por favor, inténtelo de nuevo.', 'error');
    })
    .finally(() => {
        isLoading = false;
        btn.disabled = false;
        btnText.innerHTML = '📤 Derivar Estudio';
        updateSubmitButton();
    });
});

// Función para mostrar notificaciones
function mostrarNotificacion(mensaje, tipo = 'info') {
    // Crear elemento de notificación
    const notificacion = document.createElement('div');
    notificacion.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 8px;
        color: white;
        font-weight: 600;
        z-index: 10000;
        max-width: 400px;
        animation: slideInRight 0.3s ease;
        box-shadow: 0 4px 12px rgba(0,0,0,0.2);
    `;
    
    // Colores según el tipo
    switch (tipo) {
        case 'success':
            notificacion.style.background = '#10b981';
            notificacion.innerHTML = '✅ ' + mensaje;
            break;
        case 'error':
            notificacion.style.background = '#dc2626';
            notificacion.innerHTML = '❌ ' + mensaje;
            break;
        default:
            notificacion.style.background = '#3b82f6';
            notificacion.innerHTML = 'ℹ️ ' + mensaje;
    }
    
    document.body.appendChild(notificacion);
    
    // Remover después de 4 segundos
    setTimeout(() => {
        notificacion.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => {
            if (notificacion.parentNode) {
                notificacion.parentNode.removeChild(notificacion);
            }
        }, 300);
    }, 4000);
}

// Agregar estilos de animación para notificaciones
const notificationStyles = document.createElement('style');
notificationStyles.textContent = `
    @keyframes slideInRight {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    @keyframes slideOutRight {
        from {
            transform: translateX(0);
            opacity: 1;
        }
        to {
            transform: translateX(100%);
            opacity: 0;
        }
    }
`;
document.head.appendChild(notificationStyles);

// Cerrar modal al hacer clic fuera
document.getElementById("modalDerivar").addEventListener('click', function(e) {
    if (e.target === this && !isLoading) {
        cerrarModalDerivar();
    }
});

// Cerrar modal con Escape
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape' && document.getElementById("modalDerivar").classList.contains('show') && !isLoading) {
        cerrarModalDerivar();
    }
});
</script>
