<?php
session_start();

// Verificar si el usuario está logueado y es admin
if (!isset($_SESSION['cargo']) || $_SESSION['cargo'] != 1) {
    die('Acceso denegado. Solo administradores pueden ejecutar este script.');
}

require_once('../../model/conexion.php');

echo "<h2>Verificación de Base de Datos - Derivaciones</h2>";
echo "<hr>";

// Crear conexión
$conexion = new Conexion();

try {
    $conexion->conectar();
    echo "<p style='color: green;'>✅ Conexión a base de datos exitosa</p>";
    
    $usuarioId = $_SESSION['id'];
    echo "<p><strong>Usuario Admin ID:</strong> $usuarioId</p>";
    
    // 1. Verificar tabla derivaciones
    echo "<h3>1. Verificando tabla derivaciones</h3>";
    $checkTable = "SHOW TABLES LIKE 'derivaciones'";
    $result = $conexion->query($checkTable);
    
    if ($result && $result->num_rows > 0) {
        echo "<p style='color: green;'>✅ Tabla 'derivaciones' existe</p>";
        
        // Verificar estructura
        $structure = $conexion->query("DESCRIBE derivaciones");
        echo "<h4>Estructura de la tabla:</h4>";
        echo "<table border='1' style='border-collapse: collapse;'>";
        echo "<tr><th>Campo</th><th>Tipo</th><th>Null</th><th>Key</th><th>Default</th></tr>";
        while ($row = $structure->fetch_assoc()) {
            echo "<tr>";
            echo "<td>{$row['Field']}</td>";
            echo "<td>{$row['Type']}</td>";
            echo "<td>{$row['Null']}</td>";
            echo "<td>{$row['Key']}</td>";
            echo "<td>{$row['Default']}</td>";
            echo "</tr>";
        }
        echo "</table>";
        
    } else {
        echo "<p style='color: red;'>❌ Tabla 'derivaciones' NO existe</p>";
        echo "<p>Creando tabla derivaciones...</p>";
        
        $createTable = "
        CREATE TABLE IF NOT EXISTS derivaciones (
            id INT AUTO_INCREMENT PRIMARY KEY,
            study_instance_uid VARCHAR(255) NOT NULL,
            usuario_origen_id INT NOT NULL,
            usuario_destino_id INT NOT NULL,
            fecha_derivacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            activo TINYINT(1) DEFAULT 1,
            comentario TEXT,
            vista TINYINT(1) DEFAULT 0,
            descargada TINYINT(1) DEFAULT 0,
            fecha_vista TIMESTAMP NULL,
            fecha_descarga TIMESTAMP NULL,
            INDEX idx_usuario_origen (usuario_origen_id),
            INDEX idx_usuario_destino (usuario_destino_id),
            INDEX idx_study_uid (study_instance_uid),
            INDEX idx_activo (activo)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ";
        
        if ($conexion->query($createTable)) {
            echo "<p style='color: green;'>✅ Tabla 'derivaciones' creada exitosamente</p>";
        } else {
            echo "<p style='color: red;'>❌ Error creando tabla: " . $conexion->error() . "</p>";
        }
    }
    
    // 2. Verificar datos existentes
    echo "<h3>2. Verificando datos existentes</h3>";
    $countTotal = $conexion->query("SELECT COUNT(*) as total FROM derivaciones");
    $totalRows = $countTotal->fetch_assoc()['total'];
    echo "<p><strong>Total de derivaciones en la tabla:</strong> $totalRows</p>";
    
    if ($totalRows == 0) {
        echo "<p style='color: orange;'>⚠️ No hay datos en la tabla. Insertando datos de prueba...</p>";
        
        // Obtener usuarios para datos de prueba
        $usuarios = $conexion->query("SELECT id, nombre FROM usuarios LIMIT 5");
        $userIds = [];
        while ($user = $usuarios->fetch_assoc()) {
            $userIds[] = $user['id'];
            echo "<p>Usuario disponible: ID {$user['id']} - {$user['nombre']}</p>";
        }
        
        if (count($userIds) >= 2) {
            // Insertar datos de prueba
            $testData = [
                [
                    'study_uid' => '1.2.3.4.5.6.7.8.9.10.11.12.13.14.15.16.17.18.19.20.21.22.23.24.25.26.27.28.29.30.31.32.33.34.35.36.37.38.39.40.41.42.43.44.45.46.47.48.49.50.51.52.53.54.55.56.57.58.59.60.61.62.63.64.65.66.67.68.69.70.71.72.73.74.75.76.77.78.79.80.81.82.83.84.85.86.87.88.89.90.91.92.93.94.95.96.97.98.99.100',
                    'origen' => $userIds[0],
                    'destino' => $userIds[1],
                    'comentario' => 'Derivación de prueba 1 - Revisar estudio urgente'
                ],
                [
                    'study_uid' => '2.3.4.5.6.7.8.9.10.11.12.13.14.15.16.17.18.19.20.21.22.23.24.25.26.27.28.29.30.31.32.33.34.35.36.37.38.39.40.41.42.43.44.45.46.47.48.49.50.51.52.53.54.55.56.57.58.59.60.61.62.63.64.65.66.67.68.69.70.71.72.73.74.75.76.77.78.79.80.81.82.83.84.85.86.87.88.89.90.91.92.93.94.95.96.97.98.99.101',
                    'origen' => $usuarioId, // Admin actual
                    'destino' => $userIds[1],
                    'comentario' => 'Derivación del administrador - Caso complejo'
                ],
                [
                    'study_uid' => '3.4.5.6.7.8.9.10.11.12.13.14.15.16.17.18.19.20.21.22.23.24.25.26.27.28.29.30.31.32.33.34.35.36.37.38.39.40.41.42.43.44.45.46.47.48.49.50.51.52.53.54.55.56.57.58.59.60.61.62.63.64.65.66.67.68.69.70.71.72.73.74.75.76.77.78.79.80.81.82.83.84.85.86.87.88.89.90.91.92.93.94.95.96.97.98.99.102',
                    'origen' => $userIds[1],
                    'destino' => $usuarioId, // Para admin
                    'comentario' => 'Derivación para el administrador'
                ]
            ];
            
            foreach ($testData as $data) {
                $insertQuery = "INSERT INTO derivaciones (study_instance_uid, usuario_origen_id, usuario_destino_id, comentario, activo) 
                               VALUES ('{$data['study_uid']}', {$data['origen']}, {$data['destino']}, '{$data['comentario']}', 1)";
                
                if ($conexion->query($insertQuery)) {
                    echo "<p style='color: green;'>✅ Datos de prueba insertados</p>";
                } else {
                    echo "<p style='color: red;'>❌ Error insertando datos: " . $conexion->error() . "</p>";
                }
            }
        }
    }
    
    // 3. Probar consultas de admin/index.php
    echo "<h3>3. Probando consultas de admin/index.php</h3>";
    
    // Consulta de derivaciones enviadas (por el admin actual)
    echo "<h4>Derivaciones Enviadas (por admin actual):</h4>";
    $consultaEnviadas = "SELECT d.id, d.study_instance_uid, d.usuario_origen_id, d.usuario_destino_id, 
                        d.fecha_derivacion, d.activo, d.comentario, 
                        COALESCE(d.vista, 0) as vista, COALESCE(d.descargada, 0) as descargada,
                        d.fecha_vista, d.fecha_descarga,
                        u.nombre as usuario_destino_nombre, u.email as usuario_destino_email 
                        FROM derivaciones d 
                        JOIN usuarios u ON d.usuario_destino_id = u.id 
                        WHERE d.usuario_origen_id = $usuarioId AND d.activo = 1 
                        ORDER BY d.fecha_derivacion DESC";
    
    echo "<p><strong>Consulta SQL:</strong></p>";
    echo "<pre>" . htmlspecialchars($consultaEnviadas) . "</pre>";
    
    $resultEnviadas = $conexion->query($consultaEnviadas);
    if ($resultEnviadas) {
        echo "<p style='color: green;'>✅ Consulta ejecutada exitosamente</p>";
        echo "<p><strong>Filas encontradas:</strong> " . $resultEnviadas->num_rows . "</p>";
        
        if ($resultEnviadas->num_rows > 0) {
            echo "<table border='1' style='border-collapse: collapse;'>";
            echo "<tr><th>ID</th><th>Study UID</th><th>Destino</th><th>Fecha</th><th>Comentario</th></tr>";
            while ($row = $resultEnviadas->fetch_assoc()) {
                echo "<tr>";
                echo "<td>{$row['id']}</td>";
                echo "<td>" . substr($row['study_instance_uid'], 0, 30) . "...</td>";
                echo "<td>{$row['usuario_destino_nombre']}</td>";
                echo "<td>{$row['fecha_derivacion']}</td>";
                echo "<td>{$row['comentario']}</td>";
                echo "</tr>";
            }
            echo "</table>";
        }
    } else {
        echo "<p style='color: red;'>❌ Error en consulta: " . $conexion->error() . "</p>";
    }
    
    // Consulta de derivaciones realizadas (todas en el sistema)
    echo "<h4>Derivaciones Realizadas (todas en el sistema):</h4>";
    $consultaRealizadas = "SELECT d.id, d.study_instance_uid, d.usuario_origen_id, d.usuario_destino_id, 
                          d.fecha_derivacion, d.activo, d.comentario, 
                          COALESCE(d.vista, 0) as vista, COALESCE(d.descargada, 0) as descargada,
                          d.fecha_vista, d.fecha_descarga,
                          u_origen.nombre as usuario_origen_nombre, u_origen.email as usuario_origen_email,
                          u_destino.nombre as usuario_destino_nombre, u_destino.email as usuario_destino_email
                          FROM derivaciones d 
                          JOIN usuarios u_origen ON d.usuario_origen_id = u_origen.id 
                          JOIN usuarios u_destino ON d.usuario_destino_id = u_destino.id 
                          WHERE d.activo = 1 
                          ORDER BY d.fecha_derivacion DESC";
    
    echo "<p><strong>Consulta SQL:</strong></p>";
    echo "<pre>" . htmlspecialchars($consultaRealizadas) . "</pre>";
    
    $resultRealizadas = $conexion->query($consultaRealizadas);
    if ($resultRealizadas) {
        echo "<p style='color: green;'>✅ Consulta ejecutada exitosamente</p>";
        echo "<p><strong>Filas encontradas:</strong> " . $resultRealizadas->num_rows . "</p>";
        
        if ($resultRealizadas->num_rows > 0) {
            echo "<table border='1' style='border-collapse: collapse;'>";
            echo "<tr><th>ID</th><th>Study UID</th><th>Origen</th><th>Destino</th><th>Fecha</th><th>Comentario</th></tr>";
            while ($row = $resultRealizadas->fetch_assoc()) {
                echo "<tr>";
                echo "<td>{$row['id']}</td>";
                echo "<td>" . substr($row['study_instance_uid'], 0, 30) . "...</td>";
                echo "<td>{$row['usuario_origen_nombre']}</td>";
                echo "<td>{$row['usuario_destino_nombre']}</td>";
                echo "<td>{$row['fecha_derivacion']}</td>";
                echo "<td>{$row['comentario']}</td>";
                echo "</tr>";
            }
            echo "</table>";
        }
    } else {
        echo "<p style='color: red;'>❌ Error en consulta: " . $conexion->error() . "</p>";
    }
    
    echo "<hr>";
    echo "<h3>4. Resumen y Recomendaciones</h3>";
    echo "<p>Si las consultas funcionan aquí pero no en admin/index.php, el problema podría estar en:</p>";
    echo "<ul>";
    echo "<li>Variables de sesión no inicializadas correctamente</li>";
    echo "<li>Conexión a base de datos diferente</li>";
    echo "<li>Errores de JavaScript que impiden mostrar los paneles</li>";
    echo "<li>CSS que oculta los elementos</li>";
    echo "</ul>";
    
    echo "<p><a href='index.php' style='background: #007bff; color: white; padding: 10px 15px; text-decoration: none; border-radius: 5px;'>🔙 Volver a Admin Panel</a></p>";
    
} catch (Exception $e) {
    echo "<p style='color: red;'>❌ Error: " . $e->getMessage() . "</p>";
} finally {
    if (isset($conexion)) {
        $conexion->cerrar();
    }
}
?>
