<?php
/**
 * Script de diagnóstico para verificar ReferringPhysicianName en estudios
 * Uso: Acceder desde navegador con parámetros:
 * - paciente: nombre del paciente a buscar (ej: "Gallardo")
 * - fecha: fecha en formato YYYYMMDD (ej: "20251212")
 */

// Configuración
$username = "iset";
$password = "iset";

// Obtener parámetros
$paciente = isset($_GET['paciente']) ? $_GET['paciente'] : '';
$fecha = isset($_GET['fecha']) ? $_GET['fecha'] : date('Ymd');

if (empty($paciente)) {
    echo "<h2>Diagnóstico de Estudios - ReferringPhysicianName</h2>";
    echo "<form method='GET'>";
    echo "Nombre del paciente (parcial): <input type='text' name='paciente' required><br><br>";
    echo "Fecha (YYYYMMDD): <input type='text' name='fecha' value='" . date('Ymd') . "'><br><br>";
    echo "<input type='submit' value='Buscar'>";
    echo "</form>";
    exit;
}

echo "<h2>🔍 Diagnóstico de Estudios</h2>";
echo "<p><strong>Paciente buscado:</strong> $paciente</p>";
echo "<p><strong>Fecha:</strong> $fecha</p>";
echo "<hr>";

// Buscar estudios por fecha
$url = "https://iset.portalpaciente.com.ar/api/find";
$data = json_encode([
    "Level" => "Study",
    "Query" => [
        "StudyDate" => $fecha
    ],
    "Expand" => true
]);

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, 1);
curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: application/json"]);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);

$response = curl_exec($ch);

if (curl_errno($ch)) {
    echo "<p style='color:red;'>Error cURL: " . curl_error($ch) . "</p>";
    curl_close($ch);
    exit;
}

curl_close($ch);

$jsonData = json_decode($response, true);

if ($jsonData === null) {
    echo "<p style='color:red;'>Error al decodificar JSON</p>";
    exit;
}

if (count($jsonData) == 0) {
    echo "<p style='color:orange;'>No se encontraron estudios para la fecha $fecha</p>";
    exit;
}

echo "<p><strong>Total de estudios encontrados en la fecha:</strong> " . count($jsonData) . "</p>";

// Filtrar por paciente
$estudiosEncontrados = 0;
$tabla = "<table border='1' cellpadding='10' style='border-collapse: collapse; width: 100%;'>";
$tabla .= "<thead style='background-color: #f0f0f0;'>";
$tabla .= "<tr>";
$tabla .= "<th>#</th>";
$tabla .= "<th>Paciente</th>";
$tabla .= "<th>PatientID</th>";
$tabla .= "<th>Descripción</th>";
$tabla .= "<th>Fecha</th>";
$tabla .= "<th>ReferringPhysicianName</th>";
$tabla .= "<th>Estado</th>";
$tabla .= "</tr>";
$tabla .= "</thead>";
$tabla .= "<tbody>";

foreach ($jsonData as $index => $study) {
    $patientName = str_replace("^", " ", $study['PatientMainDicomTags']['PatientName'] ?? '');
    
    // Buscar coincidencia parcial (case insensitive)
    if (stripos($patientName, $paciente) !== false) {
        $estudiosEncontrados++;
        
        $patientID = $study['PatientMainDicomTags']['PatientID'] ?? 'N/A';
        $studyDescription = str_replace("^", " ", $study['MainDicomTags']['StudyDescription'] ?? '');
        $studyDate = $study['MainDicomTags']['StudyDate'] ?? '';
        $referringPhysician = $study['MainDicomTags']['ReferringPhysicianName'] ?? '';
        
        // Determinar estado
        $estado = "";
        $colorEstado = "";
        if (empty($referringPhysician)) {
            $estado = "⚠️ VACÍO";
            $colorEstado = "background-color: #ffcccc;";
            $referringPhysician = "<em>(vacío)</em>";
        } elseif (trim($referringPhysician) !== $referringPhysician) {
            $estado = "⚠️ CON ESPACIOS";
            $colorEstado = "background-color: #fff3cd;";
            $referringPhysician = "'" . $referringPhysician . "'";
        } else {
            $estado = "✅ OK";
            $colorEstado = "background-color: #d4edda;";
        }
        
        $tabla .= "<tr style='$colorEstado'>";
        $tabla .= "<td>$estudiosEncontrados</td>";
        $tabla .= "<td><strong>$patientName</strong></td>";
        $tabla .= "<td>$patientID</td>";
        $tabla .= "<td>$studyDescription</td>";
        $tabla .= "<td>$studyDate</td>";
        $tabla .= "<td>$referringPhysician</td>";
        $tabla .= "<td>$estado</td>";
        $tabla .= "</tr>";
    }
}

$tabla .= "</tbody>";
$tabla .= "</table>";

if ($estudiosEncontrados == 0) {
    echo "<p style='color:orange;'>No se encontraron estudios para el paciente '$paciente' en la fecha $fecha</p>";
} else {
    echo "<p><strong>✅ Estudios encontrados para el paciente:</strong> $estudiosEncontrados</p>";
    echo $tabla;
    
    echo "<hr>";
    echo "<h3>📊 Análisis:</h3>";
    echo "<ul>";
    echo "<li><strong style='color: green;'>✅ OK:</strong> El campo ReferringPhysicianName tiene un valor válido</li>";
    echo "<li><strong style='color: red;'>⚠️ VACÍO:</strong> El campo está vacío - estos estudios NO aparecerán en la búsqueda por médico</li>";
    echo "<li><strong style='color: orange;'>⚠️ CON ESPACIOS:</strong> El campo tiene espacios al inicio o final - puede causar problemas de búsqueda</li>";
    echo "</ul>";
    
    echo "<hr>";
    echo "<h3>🔍 Prueba de búsquedas específicas:</h3>";
    
    // Obtener valores únicos de ReferringPhysicianName
    $valoresUnicos = [];
    foreach ($jsonData as $study) {
        $patientName = str_replace("^", " ", $study['PatientMainDicomTags']['PatientName'] ?? '');
        if (stripos($patientName, $paciente) !== false) {
            $ref = $study['MainDicomTags']['ReferringPhysicianName'] ?? '';
            if (!empty($ref)) {
                $valoresUnicos[$ref] = ($valoresUnicos[$ref] ?? 0) + 1;
            }
        }
    }
    
    if (count($valoresUnicos) > 0) {
        echo "<p>Valores únicos de ReferringPhysicianName encontrados:</p>";
        echo "<ul>";
        foreach ($valoresUnicos as $valor => $cantidad) {
            $valorMostrar = htmlspecialchars($valor);
            echo "<li><strong>'$valorMostrar'</strong> ($cantidad estudio" . ($cantidad > 1 ? 's' : '') . ")</li>";
        }
        echo "</ul>";
    }
}

echo "<hr>";
echo "<p><a href='diagnostico_estudios.php'>← Nueva búsqueda</a></p>";
?>


