<?php
/**
 * ANÁLISIS DE ESTUDIOS ESPECÍFICOS POR PatientID
 * Búsqueda por campo DICOM PatientID
 */

header('Content-Type: text/html; charset=utf-8');

echo "<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <title>Análisis de Estudios por PatientID</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { background: white; padding: 20px; border-radius: 8px; margin-bottom: 20px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        h1 { color: #333; border-bottom: 3px solid #007bff; padding-bottom: 10px; }
        h2 { color: #007bff; margin-top: 30px; }
        h3 { color: #28a745; margin-top: 20px; }
        table { width: 100%; border-collapse: collapse; margin: 15px 0; font-size: 13px; }
        th, td { border: 1px solid #ddd; padding: 10px; text-align: left; vertical-align: top; }
        th { background-color: #007bff; color: white; font-weight: bold; }
        .error { background-color: #ffebee; color: #c62828; padding: 10px; border-left: 4px solid #c62828; margin: 10px 0; }
        .success { background-color: #e8f5e9; color: #2e7d32; padding: 10px; border-left: 4px solid #2e7d32; margin: 10px 0; }
        .warning { background-color: #fff3cd; color: #856404; padding: 10px; border-left: 4px solid #856404; margin: 10px 0; }
        .info { background-color: #e3f2fd; color: #1565c0; padding: 10px; border-left: 4px solid #1565c0; margin: 10px 0; }
        .code { background: #f4f4f4; padding: 10px; border-radius: 4px; font-family: monospace; margin: 10px 0; overflow-x: auto; }
        .field { margin: 5px 0; padding: 5px; background: #f9f9f9; }
        .field-name { font-weight: bold; color: #007bff; display: inline-block; width: 250px; }
        .field-value { color: #333; }
        .highlight { background-color: #fff59d; padding: 2px 5px; font-weight: bold; }
        .compare-table { font-size: 12px; }
        .compare-table td { padding: 8px; }
        pre { background: #f4f4f4; padding: 10px; border-radius: 4px; overflow-x: auto; font-size: 12px; }
        .visible { background-color: #d4edda; }
        .hidden { background-color: #ffcccc; }
    </style>
</head>
<body>";

echo "<h1>🔬 ANÁLISIS DE ESTUDIOS POR PatientID</h1>";

// PatientIDs a analizar
$patientIds = ['32132018', '51446591', '20380974'];

// Función para hacer consulta POST (búsqueda)
function consultarOrthancPOST($query) {
    $url = "https://iset.portalpaciente.com.ar/api/find";
    
    $data = json_encode([
        "Level" => "Study",
        "Query" => $query,
        "Expand" => true
    ]);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: application/json"]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        $error = curl_error($ch);
        curl_close($ch);
        return ['error' => $error];
    }
    
    curl_close($ch);
    return json_decode($response, true);
}

// Array para almacenar información de todos los estudios por paciente
$todosPacientes = [];
$todosEstudios = [];

// Analizar cada paciente
foreach ($patientIds as $patientId) {
    echo "<div class='container'>";
    echo "<h2>🏥 Paciente ID: $patientId</h2>";
    
    // Buscar por PatientID
    echo "<div class='code'>Query: { \"PatientID\": \"$patientId\" }</div>";
    
    $resultados = consultarOrthancPOST(["PatientID" => $patientId]);
    
    if (isset($resultados['error'])) {
        echo "<div class='error'>❌ Error al buscar estudios: " . htmlspecialchars($resultados['error']) . "</div>";
        echo "</div>";
        continue;
    }
    
    if (!is_array($resultados) || count($resultados) == 0) {
        echo "<div class='error'>❌ No se encontraron estudios para PatientID: $patientId</div>";
        echo "</div>";
        continue;
    }
    
    $cantidadEstudios = count($resultados);
    echo "<div class='success'>✅ Se encontraron <strong>$cantidadEstudios</strong> estudio(s) para este paciente</div>";
    
    // Obtener información del primer estudio para mostrar datos del paciente
    $primerEstudio = $resultados[0];
    $patientName = str_replace('^', ' ', $primerEstudio['PatientMainDicomTags']['PatientName'] ?? 'N/A');
    
    echo "<div class='info'>";
    echo "<strong>👤 Nombre del Paciente:</strong> " . htmlspecialchars($patientName) . "<br>";
    echo "<strong>🔢 Patient ID:</strong> " . htmlspecialchars($patientId);
    echo "</div>";
    
    // Mostrar tabla con todos los estudios de este paciente
    echo "<h3>📋 Lista de Estudios del Paciente:</h3>";
    echo "<table>";
    echo "<tr><th>#</th><th>Orthanc ID</th><th>StudyID</th><th>Fecha</th><th>Descripción</th><th>ReferringPhysicianName</th><th>Estado</th></tr>";
    
    $referringPhysicians = [];
    
    foreach ($resultados as $index => $study) {
        $orthancId = $study['ID'] ?? 'N/A';
        $studyId = $study['MainDicomTags']['StudyID'] ?? 'N/A';
        $studyDate = $study['MainDicomTags']['StudyDate'] ?? 'N/A';
        $studyDescription = str_replace('^', ' ', $study['MainDicomTags']['StudyDescription'] ?? 'N/A');
        $referringPhysician = $study['MainDicomTags']['ReferringPhysicianName'] ?? '';
        $studyInstanceUID = $study['MainDicomTags']['StudyInstanceUID'] ?? '';
        
        // Almacenar información
        $estudioInfo = [
            'patientId' => $patientId,
            'patientName' => $patientName,
            'orthancId' => $orthancId,
            'studyId' => $studyId,
            'studyDate' => $studyDate,
            'studyDescription' => $studyDescription,
            'referringPhysician' => $referringPhysician,
            'studyInstanceUID' => $studyInstanceUID
        ];
        
        $todosEstudios[] = $estudioInfo;
        
        if (!empty($referringPhysician)) {
            $referringPhysicians[] = $referringPhysician;
        }
        
        // Determinar estado
        $refPhysDisplay = '';
        $estado = '';
        $rowClass = '';
        
        if (empty($referringPhysician)) {
            $refPhysDisplay = "<em style='color:#999;'>(vacío)</em>";
            $estado = "⚠️ SIN MÉDICO";
            $rowClass = "class='hidden'";
        } else {
            $refPhysDisplay = "<strong>" . htmlspecialchars($referringPhysician) . "</strong>";
            $estado = "✅ OK";
            $rowClass = "class='visible'";
        }
        
        echo "<tr $rowClass>";
        echo "<td>" . ($index + 1) . "</td>";
        echo "<td>" . htmlspecialchars(substr($orthancId, 0, 12)) . "...</td>";
        echo "<td>" . htmlspecialchars($studyId) . "</td>";
        echo "<td>" . htmlspecialchars($studyDate) . "</td>";
        echo "<td>" . htmlspecialchars($studyDescription) . "</td>";
        echo "<td>$refPhysDisplay</td>";
        echo "<td>$estado</td>";
        echo "</tr>";
    }
    
    echo "</table>";
    
    // Análisis de ReferringPhysician para este paciente
    $valoresUnicos = array_unique($referringPhysicians);
    
    if (count($referringPhysicians) == 0) {
        echo "<div class='error'>";
        echo "❌ <strong>NINGUNO</strong> de los $cantidadEstudios estudios tiene ReferringPhysicianName<br>";
        echo "Por eso no aparecen en la vista de usuario.";
        echo "</div>";
    } elseif (count($referringPhysicians) < $cantidadEstudios) {
        echo "<div class='warning'>";
        echo "⚠️ <strong>PROBLEMA ENCONTRADO:</strong><br>";
        echo "• " . count($referringPhysicians) . " estudio(s) CON ReferringPhysicianName ✅<br>";
        echo "• " . ($cantidadEstudios - count($referringPhysicians)) . " estudio(s) SIN ReferringPhysicianName ❌<br><br>";
        echo "Los estudios sin médico referente NO aparecerán en la búsqueda de usuario.";
        echo "</div>";
    } else {
        echo "<div class='success'>";
        echo "✅ Todos los $cantidadEstudios estudios tienen ReferringPhysicianName";
        echo "</div>";
        
        if (count($valoresUnicos) == 1) {
            echo "<div class='success'>";
            echo "✅ <strong>Y todos tienen el MISMO médico:</strong> " . htmlspecialchars($valoresUnicos[0]);
            echo "</div>";
        } else {
            echo "<div class='warning'>";
            echo "⚠️ <strong>Pero tienen DIFERENTES médicos:</strong><br>";
            foreach ($valoresUnicos as $valor) {
                echo "• '" . htmlspecialchars($valor) . "'<br>";
            }
            echo "</div>";
        }
    }
    
    $todosPacientes[$patientId] = [
        'patientName' => $patientName,
        'cantidadEstudios' => $cantidadEstudios,
        'referringPhysicians' => $valoresUnicos
    ];
    
    echo "</div>";
}

// ANÁLISIS GLOBAL
if (count($todosEstudios) > 0) {
    echo "<div class='container'>";
    echo "<h2>🔍 ANÁLISIS GLOBAL</h2>";
    
    echo "<h3>📊 Resumen General:</h3>";
    echo "<table>";
    echo "<tr><th>PatientID</th><th>Paciente</th><th>Total Estudios</th><th>Médicos Únicos</th></tr>";
    
    foreach ($todosPacientes as $pid => $info) {
        echo "<tr>";
        echo "<td><strong>$pid</strong></td>";
        echo "<td>" . htmlspecialchars($info['patientName']) . "</td>";
        echo "<td><strong>" . $info['cantidadEstudios'] . "</strong></td>";
        echo "<td>" . implode(', ', array_map('htmlspecialchars', $info['referringPhysicians'])) . "</td>";
        echo "</tr>";
    }
    
    echo "</table>";
    
    // PRUEBA DE BÚSQUEDA
    echo "<h3>🔬 PRUEBA: Simulación de búsqueda como USUARIO</h3>";
    
    // Obtener todos los médicos únicos
    $todosMedicos = [];
    foreach ($todosEstudios as $estudio) {
        if (!empty($estudio['referringPhysician'])) {
            $todosMedicos[] = $estudio['referringPhysician'];
        }
    }
    $medicosUnicos = array_unique($todosMedicos);
    
    foreach ($medicosUnicos as $medico) {
        echo "<div class='container' style='background: #f9f9f9; margin: 20px 0;'>";
        echo "<h4>👨‍⚕️ Médico: " . htmlspecialchars($medico) . "</h4>";
        
        echo "<div class='code'>Query: { \"ReferringPhysicianName\": \"" . htmlspecialchars($medico) . "\" }</div>";
        
        $resultadoBusqueda = consultarOrthancPOST(["ReferringPhysicianName" => $medico]);
        
        if (isset($resultadoBusqueda['error'])) {
            echo "<div class='error'>❌ Error: " . htmlspecialchars($resultadoBusqueda['error']) . "</div>";
        } else {
            $cantidadEncontrados = count($resultadoBusqueda);
            echo "<div class='info'>";
            echo "📊 <strong>Orthanc devuelve:</strong> $cantidadEncontrados estudio(s) en total para este médico<br>";
            echo "</div>";
            
            // Filtrar solo los estudios de nuestros pacientes
            $estudiosDeNuestrosPacientes = [];
            foreach ($resultadoBusqueda as $study) {
                $pid = $study['PatientMainDicomTags']['PatientID'] ?? '';
                if (in_array($pid, $patientIds)) {
                    $estudiosDeNuestrosPacientes[] = $study;
                }
            }
            
            echo "<div class='warning'>";
            echo "📋 <strong>De nuestros 3 pacientes:</strong> " . count($estudiosDeNuestrosPacientes) . " estudio(s)<br>";
            echo "</div>";
            
            if (count($estudiosDeNuestrosPacientes) > 0) {
                echo "<table style='font-size: 12px;'>";
                echo "<tr><th>PatientID</th><th>Paciente</th><th>Fecha</th><th>Descripción</th></tr>";
                
                foreach ($estudiosDeNuestrosPacientes as $study) {
                    $pid = $study['PatientMainDicomTags']['PatientID'] ?? '';
                    $pname = str_replace('^', ' ', $study['PatientMainDicomTags']['PatientName'] ?? '');
                    $sdate = $study['MainDicomTags']['StudyDate'] ?? '';
                    $sdesc = str_replace('^', ' ', $study['MainDicomTags']['StudyDescription'] ?? '');
                    
                    echo "<tr class='visible'>";
                    echo "<td>$pid</td>";
                    echo "<td>" . htmlspecialchars($pname) . "</td>";
                    echo "<td>$sdate</td>";
                    echo "<td>" . htmlspecialchars($sdesc) . "</td>";
                    echo "</tr>";
                }
                
                echo "</table>";
            }
            
            // Verificar si faltan estudios
            $cantidadEsperada = 0;
            foreach ($todosEstudios as $estudio) {
                if ($estudio['referringPhysician'] == $medico) {
                    $cantidadEsperada++;
                }
            }
            
            if (count($estudiosDeNuestrosPacientes) == $cantidadEsperada) {
                echo "<div class='success'>";
                echo "✅ <strong>TODOS los estudios esperados aparecen en la búsqueda de Orthanc</strong>";
                echo "</div>";
            } else {
                echo "<div class='error'>";
                echo "❌ <strong>FALTAN ESTUDIOS:</strong><br>";
                echo "• Esperados: $cantidadEsperada<br>";
                echo "• Encontrados: " . count($estudiosDeNuestrosPacientes);
                echo "</div>";
            }
        }
        
        echo "</div>";
    }
    
    echo "</div>";
    
    // DIAGNÓSTICO FINAL
    echo "<div class='container'>";
    echo "<h2>🎯 DIAGNÓSTICO FINAL</h2>";
    
    $totalEstudios = count($todosEstudios);
    $estudiosConMedico = 0;
    $estudiosSinMedico = 0;
    
    foreach ($todosEstudios as $estudio) {
        if (!empty($estudio['referringPhysician'])) {
            $estudiosConMedico++;
        } else {
            $estudiosSinMedico++;
        }
    }
    
    echo "<div class='info'>";
    echo "<strong>📊 RESUMEN:</strong><br>";
    echo "• Total de estudios analizados: <strong>$totalEstudios</strong><br>";
    echo "• Con ReferringPhysicianName: <strong>$estudiosConMedico</strong> ✅<br>";
    echo "• Sin ReferringPhysicianName: <strong>$estudiosSinMedico</strong> ❌<br>";
    echo "</div>";
    
    if ($estudiosSinMedico > 0) {
        echo "<div class='error'>";
        echo "<strong>❌ PROBLEMA IDENTIFICADO: Datos Incompletos</strong><br><br>";
        echo "$estudiosSinMedico estudio(s) no tienen el campo ReferringPhysicianName.<br>";
        echo "Estos estudios NO pueden aparecer en la búsqueda por médico.<br><br>";
        echo "<strong>SOLUCIONES:</strong><br>";
        echo "1. Corregir los datos en Orthanc (completar el campo ReferringPhysicianName)<br>";
        echo "2. Modificar el código para usar otra estrategia de búsqueda<br>";
        echo "3. Implementar asignación manual de estudios a usuarios<br>";
        echo "</div>";
    } else {
        echo "<div class='success'>";
        echo "<strong>✅ Todos los estudios tienen ReferringPhysicianName</strong><br><br>";
        echo "Si aún así no aparecen todos en la interfaz de usuario, el problema está en el código PHP de procesamiento.";
        echo "</div>";
    }
    
    echo "</div>";
}

echo "</body></html>";
?>
