<?php
/**
 * ANÁLISIS DIRECTO DE PROBLEMAS CON ESTUDIOS
 * Análisis de casos específicos reportados
 */

header('Content-Type: text/html; charset=utf-8');

echo "<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <title>Análisis Directo - Problemas con Estudios</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { background: white; padding: 20px; border-radius: 8px; margin-bottom: 20px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        h1 { color: #333; border-bottom: 3px solid #007bff; padding-bottom: 10px; }
        h2 { color: #007bff; margin-top: 30px; }
        h3 { color: #28a745; margin-top: 20px; }
        table { width: 100%; border-collapse: collapse; margin: 15px 0; }
        th, td { border: 1px solid #ddd; padding: 12px; text-align: left; }
        th { background-color: #007bff; color: white; font-weight: bold; }
        .error { background-color: #ffebee; color: #c62828; padding: 10px; border-left: 4px solid #c62828; margin: 10px 0; }
        .success { background-color: #e8f5e9; color: #2e7d32; padding: 10px; border-left: 4px solid #2e7d32; margin: 10px 0; }
        .warning { background-color: #fff3cd; color: #856404; padding: 10px; border-left: 4px solid #856404; margin: 10px 0; }
        .info { background-color: #e3f2fd; color: #1565c0; padding: 10px; border-left: 4px solid #1565c0; margin: 10px 0; }
        .highlight { background-color: #fff59d; font-weight: bold; }
        .code { background: #f4f4f4; padding: 10px; border-radius: 4px; font-family: monospace; margin: 10px 0; }
        tr.found { background-color: #d4edda; }
        tr.not-found { background-color: #ffcccc; }
        .badge { padding: 4px 8px; border-radius: 4px; font-size: 12px; font-weight: bold; }
        .badge-success { background: #28a745; color: white; }
        .badge-danger { background: #dc3545; color: white; }
        .badge-warning { background: #ffc107; color: black; }
    </style>
</head>
<body>";

echo "<h1>🔬 ANÁLISIS DIRECTO DE PROBLEMAS CON ESTUDIOS</h1>";

// Casos a analizar
$casos = [
    [
        'usuario' => 'MARIANO TAHHAN',
        'paciente' => 'Tahhan Santiago Alejandro',
        'fecha' => '20251212',
        'estudios_esperados' => 3,
        'estudios_visibles' => 1
    ],
    [
        'usuario' => 'ADRIANA CHAUD',
        'paciente' => 'Gallardo Juarez Maria',
        'fecha' => '20251212',
        'estudios_esperados' => 3,
        'estudios_visibles' => 1
    ]
];

$url = "https://iset.portalpaciente.com.ar/api/find";

// Función para hacer consulta
function consultarOrthanc($query) {
    global $url;
    
    $data = json_encode([
        "Level" => "Study",
        "Query" => $query,
        "Expand" => true
    ]);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: application/json"]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        $error = curl_error($ch);
        curl_close($ch);
        return ['error' => $error];
    }
    
    curl_close($ch);
    return json_decode($response, true);
}

// Analizar cada caso
foreach ($casos as $index => $caso) {
    echo "<div class='container'>";
    echo "<h2>📋 CASO #" . ($index + 1) . ": " . htmlspecialchars($caso['usuario']) . "</h2>";
    
    echo "<div class='info'>";
    echo "<strong>👤 Usuario:</strong> " . htmlspecialchars($caso['usuario']) . "<br>";
    echo "<strong>🏥 Paciente:</strong> " . htmlspecialchars($caso['paciente']) . "<br>";
    echo "<strong>📅 Fecha:</strong> " . $caso['fecha'] . "<br>";
    echo "<strong>📊 Estudios esperados:</strong> " . $caso['estudios_esperados'] . "<br>";
    echo "<strong>👁️ Estudios visibles:</strong> " . $caso['estudios_visibles'] . " <span class='badge badge-danger'>PROBLEMA</span>";
    echo "</div>";
    
    // PASO 1: Buscar por fecha (como lo hace el admin)
    echo "<h3>🔍 PASO 1: Búsqueda por FECHA (como admin)</h3>";
    echo "<div class='code'>Query: { \"StudyDate\": \"" . $caso['fecha'] . "\" }</div>";
    
    $resultadoFecha = consultarOrthanc(["StudyDate" => $caso['fecha']]);
    
    if (isset($resultadoFecha['error'])) {
        echo "<div class='error'>❌ Error en consulta: " . $resultadoFecha['error'] . "</div>";
        continue;
    }
    
    if (!is_array($resultadoFecha)) {
        echo "<div class='error'>❌ Respuesta inválida de Orthanc</div>";
        continue;
    }
    
    // Buscar estudios del paciente
    $palabrasClave = explode(' ', strtolower($caso['paciente']));
    $estudiosDelPaciente = [];
    
    foreach ($resultadoFecha as $study) {
        $patientName = str_replace("^", " ", $study['PatientMainDicomTags']['PatientName'] ?? '');
        $patientNameLower = strtolower($patientName);
        
        // Verificar si coincide con al menos 2 palabras del nombre
        $coincidencias = 0;
        foreach ($palabrasClave as $palabra) {
            if (strlen($palabra) > 2 && strpos($patientNameLower, $palabra) !== false) {
                $coincidencias++;
            }
        }
        
        if ($coincidencias >= 2) {
            $estudiosDelPaciente[] = $study;
        }
    }
    
    echo "<div class='success'>✅ Total de estudios en la fecha: " . count($resultadoFecha) . "</div>";
    echo "<div class='success'>✅ Estudios del paciente encontrados: " . count($estudiosDelPaciente) . "</div>";
    
    if (count($estudiosDelPaciente) == 0) {
        echo "<div class='error'>❌ No se encontraron estudios para este paciente en esta fecha</div>";
        continue;
    }
    
    // Mostrar detalles de cada estudio
    echo "<table>";
    echo "<tr><th>#</th><th>Paciente</th><th>PatientID</th><th>Descripción</th><th>ReferringPhysicianName</th><th>Estado</th></tr>";
    
    foreach ($estudiosDelPaciente as $idx => $study) {
        $patientName = str_replace("^", " ", $study['PatientMainDicomTags']['PatientName'] ?? '');
        $patientID = $study['PatientMainDicomTags']['PatientID'] ?? 'N/A';
        $studyDescription = str_replace("^", " ", $study['MainDicomTags']['StudyDescription'] ?? '');
        $referringPhysician = $study['MainDicomTags']['ReferringPhysicianName'] ?? '';
        
        $clase = "";
        $estado = "";
        
        if (empty($referringPhysician)) {
            $clase = "not-found";
            $estado = "<span class='badge badge-danger'>VACÍO</span>";
            $referringPhysician = "<em style='color: #999;'>(campo vacío)</em>";
        } elseif (trim($referringPhysician) == $caso['usuario']) {
            $clase = "found";
            $estado = "<span class='badge badge-success'>COINCIDE</span>";
        } else {
            $clase = "not-found";
            $estado = "<span class='badge badge-warning'>NO COINCIDE</span>";
        }
        
        echo "<tr class='$clase'>";
        echo "<td>" . ($idx + 1) . "</td>";
        echo "<td>" . htmlspecialchars($patientName) . "</td>";
        echo "<td>" . htmlspecialchars($patientID) . "</td>";
        echo "<td>" . htmlspecialchars($studyDescription) . "</td>";
        echo "<td><strong>" . $referringPhysician . "</strong></td>";
        echo "<td>$estado</td>";
        echo "</tr>";
    }
    
    echo "</table>";
    
    // PASO 2: Buscar como lo hace el usuario
    echo "<h3>🔍 PASO 2: Búsqueda por MÉDICO REFERENTE (como usuario)</h3>";
    echo "<div class='code'>Query: { \"ReferringPhysicianName\": \"" . htmlspecialchars($caso['usuario']) . "\" }</div>";
    
    $resultadoUsuario = consultarOrthanc(["ReferringPhysicianName" => $caso['usuario']]);
    
    if (isset($resultadoUsuario['error'])) {
        echo "<div class='error'>❌ Error en consulta: " . $resultadoUsuario['error'] . "</div>";
    } else {
        // Filtrar solo estudios del paciente en esa fecha
        $estudiosVisibles = [];
        foreach ($resultadoUsuario as $study) {
            $patientName = str_replace("^", " ", $study['PatientMainDicomTags']['PatientName'] ?? '');
            $studyDate = $study['MainDicomTags']['StudyDate'] ?? '';
            $patientNameLower = strtolower($patientName);
            
            $coincidencias = 0;
            foreach ($palabrasClave as $palabra) {
                if (strlen($palabra) > 2 && strpos($patientNameLower, $palabra) !== false) {
                    $coincidencias++;
                }
            }
            
            if ($coincidencias >= 2 && $studyDate == $caso['fecha']) {
                $estudiosVisibles[] = $study;
            }
        }
        
        echo "<div class='warning'>⚠️ Estudios encontrados para el usuario: " . count($resultadoUsuario) . "</div>";
        echo "<div class='warning'>⚠️ Estudios del paciente visibles: " . count($estudiosVisibles) . "</div>";
    }
    
    // DIAGNÓSTICO FINAL
    echo "<h3>📊 DIAGNÓSTICO FINAL</h3>";
    
    $estudiosConProblema = 0;
    $motivosProblema = [];
    
    foreach ($estudiosDelPaciente as $study) {
        $referringPhysician = $study['MainDicomTags']['ReferringPhysicianName'] ?? '';
        
        if (empty($referringPhysician)) {
            $estudiosConProblema++;
            $motivosProblema[] = "Campo ReferringPhysicianName VACÍO";
        } elseif (trim($referringPhysician) != $caso['usuario']) {
            $estudiosConProblema++;
            $motivosProblema[] = "ReferringPhysicianName = '" . htmlspecialchars($referringPhysician) . "' (NO coincide con '" . htmlspecialchars($caso['usuario']) . "')";
        }
    }
    
    if ($estudiosConProblema > 0) {
        echo "<div class='error'>";
        echo "<strong>❌ PROBLEMA IDENTIFICADO:</strong><br><br>";
        echo "De los " . count($estudiosDelPaciente) . " estudios del paciente:<br>";
        echo "• <strong>" . (count($estudiosDelPaciente) - $estudiosConProblema) . " estudio(s)</strong> tienen el campo correcto ✅<br>";
        echo "• <strong>" . $estudiosConProblema . " estudio(s)</strong> tienen el campo incorrecto o vacío ❌<br><br>";
        echo "<strong>Motivos específicos:</strong><br>";
        foreach (array_unique($motivosProblema) as $motivo) {
            echo "• " . $motivo . "<br>";
        }
        echo "</div>";
        
        echo "<div class='info'>";
        echo "<strong>🔧 CAUSA RAÍZ:</strong><br>";
        echo "Los estudios fueron cargados en el sistema DICOM sin el campo <code>ReferringPhysicianName</code> correcto o con valores inconsistentes.";
        echo "</div>";
    } else {
        echo "<div class='success'>✅ Todos los estudios tienen el campo ReferringPhysicianName correcto</div>";
    }
    
    echo "</div>"; // Fin container
}

// CONCLUSIONES Y RECOMENDACIONES
echo "<div class='container'>";
echo "<h2>🎯 CONCLUSIONES GENERALES</h2>";

echo "<div class='error'>";
echo "<strong>❌ PROBLEMA IDENTIFICADO:</strong><br><br>";
echo "El problema es de <strong>CALIDAD DE DATOS</strong>, no de código.<br><br>";
echo "Los estudios fueron cargados en el servidor DICOM (Orthanc) sin el campo <code>ReferringPhysicianName</code> o con valores incorrectos.<br><br>";
echo "Cuando un usuario inicia sesión, el sistema busca estudios donde <code>ReferringPhysicianName</code> coincida EXACTAMENTE con el nombre del usuario. Si el campo está vacío o tiene otro valor, el estudio NO aparece.";
echo "</div>";

echo "<h3>🔧 SOLUCIONES POSIBLES:</h3>";

echo "<div class='info'>";
echo "<strong>OPCIÓN 1: Corregir los datos en Orthanc (IDEAL pero complejo)</strong><br>";
echo "• Actualizar el campo ReferringPhysicianName en los estudios que tienen el valor incorrecto<br>";
echo "• Requiere acceso administrativo a Orthanc y scripts de corrección<br>";
echo "• Es la solución más limpia pero requiere más trabajo<br>";
echo "</div>";

echo "<div class='info'>";
echo "<strong>OPCIÓN 2: Modificar el código para búsqueda flexible (RÁPIDO) ⭐ RECOMENDADO</strong><br>";
echo "• Usar comodines en la búsqueda: <code>\"ReferringPhysicianName\": \"*NOMBRE*\"</code><br>";
echo "• Permite coincidencias parciales y campos con espacios extras<br>";
echo "• Puede mostrar estudios no deseados si hay coincidencias ambiguas<br>";
echo "</div>";

echo "<div class='info'>";
echo "<strong>OPCIÓN 3: Sistema de asignación manual (COMPLETO)</strong><br>";
echo "• Crear tabla en base de datos que vincule estudios con usuarios<br>";
echo "• Permitir asignación manual de estudios desde el panel admin<br>";
echo "• Mayor control pero requiere trabajo manual<br>";
echo "</div>";

echo "<div class='info'>";
echo "<strong>OPCIÓN 4: Búsqueda híbrida (EQUILIBRADO) ⭐ RECOMENDADO</strong><br>";
echo "• Buscar por fecha (como admin) y filtrar en PHP por múltiples criterios<br>";
echo "• Permite búsqueda por ReferringPhysicianName + PatientID + asignaciones manuales<br>";
echo "• Más flexible y controlable<br>";
echo "</div>";

echo "<div class='warning'>";
echo "<strong>⚠️ IMPORTANTE:</strong><br>";
echo "Para evitar este problema en el futuro, es necesario:<br>";
echo "1. Configurar correctamente el equipo/software que envía estudios a Orthanc<br>";
echo "2. Validar que el campo ReferringPhysicianName se complete correctamente al momento de la carga<br>";
echo "3. Implementar validaciones automáticas<br>";
echo "</div>";

echo "</div>";

echo "</body></html>";
?>


