const express = require('express');
const axios = require('axios');
const cors = require('cors');
const app = express();

// Configuración
const PORT = 3001;
const ORTHANC_URL = 'http://localhost:8042';
const ORTHANC_AUTH = {
    username: 'iset',
    password: 'iset'
};

// Middleware
app.use(express.json());
app.use(cors());

// Middleware de logging para depuración
app.use((req, res, next) => {
    console.log(`[${new Date().toISOString()}] ${req.method} ${req.path} - Query:`, req.query);
    next();
});

// Ruta GET para el navegador
app.get('/api/find', (req, res) => {
    res.send(`
        <html>
            <head>
                <title>ISET API Proxy</title>
                <style>
                    body {
                        font-family: Arial, sans-serif;
                        max-width: 800px;
                        margin: 40px auto;
                        padding: 20px;
                        line-height: 1.6;
                    }
                    .container {
                        background-color: #f5f5f5;
                        border-radius: 8px;
                        padding: 20px;
                        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                    }
                    h1 {
                        color: #333;
                        border-bottom: 2px solid #4CAF50;
                        padding-bottom: 10px;
                    }
                    .status {
                        color: #4CAF50;
                        font-weight: bold;
                    }
                    .endpoint {
                        background-color: #e8f5e9;
                        padding: 10px;
                        border-radius: 4px;
                        margin: 10px 0;
                    }
                </style>
            </head>
            <body>
                <div class="container">
                    <h1>ISET API Proxy</h1>
                    <p class="status">✅ El servicio está funcionando correctamente</p>
                    <p>Esta API actúa como proxy para el servidor Orthanc.</p>
                    
                    <div class="endpoint">
                        <strong>Endpoint:</strong> <code>/api/find</code><br>
                        <strong>Método:</strong> <code>POST</code><br>
                        <strong>Descripción:</strong> Consulta/búsqueda de estudios
                    </div>
                    
                    <div class="endpoint">
                        <strong>Endpoint:</strong> <code>/api/studies/:studyID/archive</code><br>
                        <strong>Método:</strong> <code>GET</code><br>
                        <strong>Descripción:</strong> Descarga de estudios como ZIP<br>
                        <strong>Parámetros:</strong> <code>?filename=nombre.zip</code>
                    </div>
                    
                    <p><strong>Estado:</strong> <code>Activo</code></p>
                </div>
            </body>
        </html>
    `);
});

// Ruta POST para las peticiones de búsqueda (find)
app.post('/api/find', async (req, res) => {
    try {
        const response = await axios({
            method: 'post',
            url: `${ORTHANC_URL}/tools/find`,
            auth: ORTHANC_AUTH,
            data: req.body,
            headers: {
                'Content-Type': 'application/json'
            }
        });
        res.json(response.data);
    } catch (error) {
        console.error('Error en la petición /api/find:', error.message);
        res.status(500).json({
            error: 'Error al procesar la petición',
            details: error.message
        });
    }
});

// Ruta de prueba para verificar que el proxy funciona
app.get('/test', (req, res) => {
    res.json({ 
        status: 'ok', 
        message: 'El proxy está funcionando correctamente',
        timestamp: new Date().toISOString()
    });
});

// Ruta GET para /api/studies (información)
// IMPORTANTE: Esta ruta debe ir ANTES de /api/studies/* para que tenga prioridad
// También debe ir ANTES de /api/studies/:studyID/archive
app.get('/api/studies', (req, res) => {
    console.log('✅ Petición GET recibida en /api/studies (ruta exacta)');
    console.log('Request path:', req.path);
    console.log('Request originalUrl:', req.originalUrl);
    console.log('Request url:', req.url);
    res.send(`
        <html>
            <head>
                <title>ISET API Proxy - Estudios</title>
                <style>
                    body {
                        font-family: Arial, sans-serif;
                        max-width: 800px;
                        margin: 40px auto;
                        padding: 20px;
                        line-height: 1.6;
                    }
                    .container {
                        background-color: #f5f5f5;
                        border-radius: 8px;
                        padding: 20px;
                        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                    }
                    h1 {
                        color: #333;
                        border-bottom: 2px solid #4CAF50;
                        padding-bottom: 10px;
                    }
                    .status {
                        color: #4CAF50;
                        font-weight: bold;
                    }
                    .endpoint {
                        background-color: #e8f5e9;
                        padding: 10px;
                        border-radius: 4px;
                        margin: 10px 0;
                    }
                    .warning {
                        background-color: #fff3cd;
                        padding: 10px;
                        border-radius: 4px;
                        margin: 10px 0;
                        border-left: 4px solid #ffc107;
                    }
                    code {
                        background-color: #f0f0f0;
                        padding: 2px 6px;
                        border-radius: 3px;
                        font-family: 'Courier New', monospace;
                    }
                </style>
            </head>
            <body>
                <div class="container">
                    <h1>ISET API Proxy - Estudios</h1>
                    <p class="status">✅ El servicio está funcionando correctamente</p>
                    
                    <div class="warning">
                        <strong>⚠️ Nota:</strong> Este endpoint requiere un <code>studyID</code> específico.
                    </div>
                    
                    <div class="endpoint">
                        <strong>Endpoint de descarga:</strong><br>
                        <code>GET /api/studies/:studyID/archive?filename=nombre.zip</code><br>
                        <strong>Ejemplo:</strong><br>
                        <code>/api/studies/abc123-def456-ghi789/archive?filename=paciente.zip</code>
                    </div>
                    
                    <div class="endpoint">
                        <strong>Otros endpoints disponibles:</strong><br>
                        <code>GET /api/studies/:studyID</code> - Información del estudio<br>
                        <code>GET /api/studies/:studyID/archive</code> - Descarga del estudio<br>
                        <code>GET /api/find</code> - Búsqueda de estudios
                    </div>
                    
                    <p><strong>Estado:</strong> <code>Activo</code></p>
                </div>
            </body>
        </html>
    `);
});

// Ruta GET para descargar estudios (archive)
// Maneja: /api/studies/:studyID/archive?filename=nombre.zip
app.get('/api/studies/:studyID/archive', async (req, res) => {
    try {
        const { studyID } = req.params;
        const { filename } = req.query;
        
        // Construir la URL de Orthanc
        const orthancUrl = `${ORTHANC_URL}/studies/${studyID}/archive`;
        
        // Hacer la petición a Orthanc con autenticación
        const response = await axios({
            method: 'get',
            url: orthancUrl,
            auth: ORTHANC_AUTH,
            responseType: 'stream', // Importante: stream para archivos binarios
            headers: {
                'Accept': 'application/zip'
            }
        });
        
        // Configurar headers de respuesta
        res.setHeader('Content-Type', 'application/zip');
        if (filename) {
            res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
        } else {
            res.setHeader('Content-Disposition', `attachment; filename="study_${studyID}.zip"`);
        }
        
        // Pasar el stream directamente al cliente
        response.data.pipe(res);
        
    } catch (error) {
        console.error('Error en la descarga del estudio:', error.message);
        if (error.response) {
            res.status(error.response.status).json({
                error: 'Error al descargar el estudio',
                details: error.response.data || error.message
            });
        } else {
            res.status(500).json({
                error: 'Error al procesar la descarga',
                details: error.message
            });
        }
    }
});

// Ruta catch-all para otras peticiones a /api/studies
// IMPORTANTE: Esta ruta debe ir DESPUÉS de las rutas específicas
// NOTA: Esta ruta NO debe capturar /api/studies exacto, solo /api/studies/*
app.all('/api/studies/*', async (req, res) => {
    console.log(`📥 Petición ${req.method} recibida en /api/studies/* (catch-all): ${req.path}`);
    console.log('Original URL:', req.originalUrl);
    try {
        // Extraer la ruta después de /api/studies
        const path = req.path.replace('/api/studies', '');
        const orthancUrl = `${ORTHANC_URL}/studies${path}`;
        console.log(`🔄 Proxying a Orthanc: ${orthancUrl}`);
        
        // Pasar los query parameters
        const queryString = new URLSearchParams(req.query).toString();
        const fullUrl = queryString ? `${orthancUrl}?${queryString}` : orthancUrl;
        
        const response = await axios({
            method: req.method.toLowerCase(),
            url: fullUrl,
            auth: ORTHANC_AUTH,
            data: req.body,
            headers: {
                'Content-Type': req.get('Content-Type') || 'application/json'
            },
            // Si es una descarga, usar stream
            responseType: req.path.includes('/archive') ? 'stream' : 'json'
        });
        
        // Si es stream, pasarlo directamente
        if (response.data.pipe) {
            response.data.pipe(res);
        } else {
            res.json(response.data);
        }
        
    } catch (error) {
        console.error('Error en la petición a /api/studies:', error.message);
        if (error.response) {
            res.status(error.response.status).json({
                error: 'Error en la petición',
                details: error.response.data || error.message
            });
        } else {
            res.status(500).json({
                error: 'Error al procesar la petición',
                details: error.message
            });
        }
    }
});

// Iniciar el servidor
app.listen(PORT, () => {
    console.log(`Servidor proxy ejecutándose en el puerto ${PORT}`);
    console.log(`Endpoints disponibles:`);
    console.log(`  - POST /api/find - Búsqueda de estudios`);
    console.log(`  - GET /api/studies/:studyID/archive - Descarga de estudios`);
    console.log(`  - ALL /api/studies/* - Proxy general para Orthanc`);
});

