-- =====================================================
-- SCRIPT DE VERIFICACIÓN PARA DERIVACIONES OCULTAS
-- =====================================================
-- Fecha: 2025-01-08
-- Descripción: Script para verificar que la migración se aplicó correctamente
-- Uso: Ejecutar después de aplicar la migración principal
-- =====================================================

-- 1. VERIFICAR QUE LA COLUMNA EXISTE
-- =====================================================
SELECT 
    COLUMN_NAME,
    DATA_TYPE,
    IS_NULLABLE,
    COLUMN_DEFAULT,
    COLUMN_COMMENT
FROM INFORMATION_SCHEMA.COLUMNS 
WHERE TABLE_SCHEMA = 'login' 
  AND TABLE_NAME = 'derivaciones' 
  AND COLUMN_NAME = 'oculta_por_usuario';

-- 2. VERIFICAR LA ESTRUCTURA DE LA TABLA
-- =====================================================
DESCRIBE derivaciones;

-- 3. VERIFICAR FUNCIONES JSON DISPONIBLES
-- =====================================================
-- Probar JSON_CONTAINS
SELECT JSON_CONTAINS('[1,2,3]', '2') as test_json_contains;

-- Probar JSON_ARRAY_APPEND
SELECT JSON_ARRAY_APPEND('[1,2]', '$', 3) as test_json_append;

-- Probar JSON_REMOVE
SELECT JSON_REMOVE('[1,2,3]', '$[1]') as test_json_remove;

-- Probar JSON_SEARCH
SELECT JSON_SEARCH('[1,2,3]', 'one', 2) as test_json_search;

-- 4. VERIFICAR DATOS EXISTENTES
-- =====================================================
-- Contar derivaciones por estado de ocultación
SELECT 
    CASE 
        WHEN oculta_por_usuario IS NULL THEN 'No ocultas'
        WHEN JSON_LENGTH(oculta_por_usuario) = 0 THEN 'Array vacío'
        ELSE CONCAT('Ocultas por ', JSON_LENGTH(oculta_por_usuario), ' usuario(s)')
    END as estado_ocultacion,
    COUNT(*) as cantidad
FROM derivaciones 
WHERE activo = 1
GROUP BY 
    CASE 
        WHEN oculta_por_usuario IS NULL THEN 'No ocultas'
        WHEN JSON_LENGTH(oculta_por_usuario) = 0 THEN 'Array vacío'
        ELSE CONCAT('Ocultas por ', JSON_LENGTH(oculta_por_usuario), ' usuario(s)')
    END;

-- 5. VERIFICAR CONSULTAS DE FILTRADO
-- =====================================================
-- Probar consulta para derivaciones normales (usuario 23 como ejemplo)
SELECT 
    d.id,
    d.study_instance_uid,
    d.usuario_destino_id,
    d.oculta_por_usuario,
    CASE 
        WHEN d.oculta_por_usuario IS NULL THEN 'Normal'
        WHEN JSON_CONTAINS(d.oculta_por_usuario, CAST(23 AS JSON)) = 0 THEN 'Normal'
        ELSE 'Oculta'
    END as estado_para_usuario_23
FROM derivaciones d 
WHERE d.usuario_destino_id = 23 
  AND d.activo = 1 
LIMIT 5;

-- 6. VERIFICAR OPERACIONES DE OCULTAR/MOSTRAR
-- =====================================================
-- Crear una derivación de prueba para testing
-- (Solo ejecutar si quieres hacer pruebas)

-- INSERT INTO derivaciones (study_instance_uid, usuario_origen_id, usuario_destino_id, fecha_derivacion, activo, comentario)
-- VALUES ('TEST-12345', 1, 23, NOW(), 1, 'Derivación de prueba para testing');

-- Ocultar derivación de prueba (reemplazar ID con el ID real)
-- UPDATE derivaciones 
-- SET oculta_por_usuario = JSON_ARRAY_APPEND(COALESCE(oculta_por_usuario, '[]'), '$', 23) 
-- WHERE id = [ID_DE_PRUEBA] AND usuario_destino_id = 23;

-- Verificar que se ocultó correctamente
-- SELECT id, oculta_por_usuario FROM derivaciones WHERE id = [ID_DE_PRUEBA];

-- Mostrar derivación de prueba (revertir ocultación)
-- UPDATE derivaciones 
-- SET oculta_por_usuario = JSON_REMOVE(oculta_por_usuario, JSON_UNQUOTE(JSON_SEARCH(oculta_por_usuario, 'one', 23))) 
-- WHERE id = [ID_DE_PRUEBA] AND usuario_destino_id = 23;

-- Verificar que se mostró correctamente
-- SELECT id, oculta_por_usuario FROM derivaciones WHERE id = [ID_DE_PRUEBA];

-- 7. LIMPIAR DATOS DE PRUEBA (OPCIONAL)
-- =====================================================
-- DELETE FROM derivaciones WHERE study_instance_uid = 'TEST-12345';

-- =====================================================
-- RESULTADOS ESPERADOS
-- =====================================================

-- 1. La columna 'oculta_por_usuario' debe existir con tipo JSON
-- 2. Las funciones JSON deben devolver valores correctos
-- 3. Las consultas de filtrado deben funcionar sin errores
-- 4. Las operaciones de ocultar/mostrar deben ejecutarse correctamente

-- =====================================================
-- FIN DEL SCRIPT DE VERIFICACIÓN
-- =====================================================
