# MIGRACIÓN DE BASE DE DATOS - DERIVACIONES OCULTAS

## 📋 RESUMEN EJECUTIVO

**Fecha:** 2025-01-08  
**Funcionalidad:** Sistema de derivaciones ocultas para usuarios  
**Base de datos:** `login`  
**Tabla:** `derivaciones`  
**Compatibilidad:** MySQL 5.7+

---

## 🔧 MODIFICACIONES REALIZADAS

### 1. **Nueva Columna en Tabla `derivaciones`**
```sql
ALTER TABLE derivaciones 
ADD COLUMN oculta_por_usuario JSON DEFAULT NULL 
COMMENT 'Array JSON de IDs de usuarios que han ocultado esta derivación';
```

### 2. **Tipo de Datos: JSON**
- **Valor por defecto:** `NULL` (derivación visible)
- **Valores posibles:**
  - `NULL` - No oculta por ningún usuario
  - `'[]'` - Array vacío (equivalente a NULL)
  - `'[23]'` - Solo usuario 23 la ha ocultado
  - `'[23, 45, 67]'` - Múltiples usuarios la han ocultado

---

## 📁 ARCHIVOS CREADOS

### 1. **`db/migration_completa_derivaciones_ocultas.sql`**
- Script completo de migración
- Ejemplos de consultas
- Notas de compatibilidad MySQL 5.7
- Script de rollback

### 2. **`db/verificacion_derivaciones_ocultas.sql`**
- Script de verificación post-migración
- Pruebas de funciones JSON
- Validación de consultas
- Tests de operaciones

---

## 🚀 INSTRUCCIONES PARA REPLICAR EN SERVIDOR LINUX

### **Paso 1: Aplicar la Migración**
```bash
# Conectar a MySQL
mysql -u root -p

# Seleccionar base de datos
USE login;

# Ejecutar migración
SOURCE /ruta/a/migration_completa_derivaciones_ocultas.sql;
```

### **Paso 2: Verificar la Migración**
```bash
# Ejecutar script de verificación
SOURCE /ruta/a/verificacion_derivaciones_ocultas.sql;
```

### **Paso 3: Verificar Funcionalidad**
```sql
-- Verificar que la columna existe
DESCRIBE derivaciones;

-- Probar funciones JSON
SELECT JSON_CONTAINS('[1,2,3]', '2') as test;
```

---

## 🔍 FUNCIONES JSON UTILIZADAS

| Función | Propósito | Ejemplo |
|---------|-----------|---------|
| `JSON_CONTAINS()` | Verificar si usuario ocultó derivación | `JSON_CONTAINS(oculta_por_usuario, CAST(23 AS JSON))` |
| `JSON_ARRAY_APPEND()` | Agregar usuario al array de ocultos | `JSON_ARRAY_APPEND(oculta_por_usuario, '$', 23)` |
| `JSON_REMOVE()` | Quitar usuario del array de ocultos | `JSON_REMOVE(oculta_por_usuario, JSON_SEARCH(...))` |
| `JSON_SEARCH()` | Encontrar posición del usuario en array | `JSON_SEARCH(oculta_por_usuario, 'one', 23)` |

---

## 📊 CONSULTAS PRINCIPALES

### **Derivaciones Normales (No Ocultas)**
```sql
SELECT d.*, u.nombre as usuario_origen_nombre 
FROM derivaciones d 
JOIN usuarios u ON d.usuario_origen_id = u.id 
WHERE d.usuario_destino_id = ? AND d.activo = 1 
AND (d.oculta_por_usuario IS NULL OR JSON_CONTAINS(d.oculta_por_usuario, CAST(? AS JSON)) = 0)
ORDER BY d.fecha_derivacion DESC;
```

### **Derivaciones Ocultas**
```sql
SELECT d.*, u.nombre as usuario_origen_nombre 
FROM derivaciones d 
JOIN usuarios u ON d.usuario_origen_id = u.id 
WHERE d.usuario_destino_id = ? AND d.activo = 1 
AND d.oculta_por_usuario IS NOT NULL AND JSON_CONTAINS(d.oculta_por_usuario, CAST(? AS JSON)) = 1
ORDER BY d.fecha_derivacion DESC;
```

### **Ocultar Derivación**
```sql
UPDATE derivaciones 
SET oculta_por_usuario = JSON_ARRAY_APPEND(COALESCE(oculta_por_usuario, '[]'), '$', ?) 
WHERE id = ? AND usuario_destino_id = ?;
```

### **Mostrar Derivación**
```sql
UPDATE derivaciones 
SET oculta_por_usuario = JSON_REMOVE(oculta_por_usuario, JSON_UNQUOTE(JSON_SEARCH(oculta_por_usuario, 'one', ?))) 
WHERE id = ? AND usuario_destino_id = ?;
```

---

## ⚠️ CONSIDERACIONES IMPORTANTES

### **Compatibilidad MySQL 5.7**
- ✅ Soporte completo para tipo JSON
- ✅ Todas las funciones JSON disponibles
- ✅ Rendimiento optimizado para operaciones JSON

### **Rendimiento**
- Las consultas JSON pueden ser más lentas que consultas tradicionales
- Considera agregar índices si tienes muchas derivaciones
- El filtrado por usuario destino sigue siendo eficiente

### **Backup**
- **IMPORTANTE:** Hacer backup antes de aplicar la migración
- La nueva columna es compatible con versiones anteriores
- Rollback disponible si es necesario

---

## 🔄 ROLLBACK (Si es necesario)

```sql
-- Eliminar la columna (revertir migración)
ALTER TABLE derivaciones DROP COLUMN oculta_por_usuario;
```

---

## ✅ CHECKLIST DE IMPLEMENTACIÓN

- [ ] Backup de base de datos realizado
- [ ] Migración aplicada en servidor de desarrollo
- [ ] Script de verificación ejecutado
- [ ] Funciones JSON probadas
- [ ] Consultas de filtrado verificadas
- [ ] Aplicación web probada
- [ ] Migración aplicada en servidor de producción
- [ ] Funcionalidad verificada en producción

---

**Archivos adjuntos:**
- `migration_completa_derivaciones_ocultas.sql`
- `verificacion_derivaciones_ocultas.sql`
