const express = require('express');
const axios = require('axios');
const cors = require('cors');
const app = express();

// Configuración
const PORT = 3001;
const ORTHANC_URL = 'http://localhost:8042';
const ORTHANC_AUTH = {
    username: 'iset',
    password: 'iset'
};

// Configuración para dcm4chee-arc
const DCM4CHEE_ARC_URL = 'http://localhost:7080';
const DCM4CHEE_ARC_AET = 'AS_RECEIVED'; // Application Entity Title

// Middleware
app.use(express.json());
app.use(cors());

// Middleware de logging para depuración
app.use((req, res, next) => {
    console.log(`[${new Date().toISOString()}] ${req.method} ${req.path} - Query:`, req.query);
    next();
});

// Ruta GET para el navegador
app.get('/api/find', (req, res) => {
    res.send(`
        <html>
            <head>
                <title>ISET API Proxy</title>
                <style>
                    body {
                        font-family: Arial, sans-serif;
                        max-width: 800px;
                        margin: 40px auto;
                        padding: 20px;
                        line-height: 1.6;
                    }
                    .container {
                        background-color: #f5f5f5;
                        border-radius: 8px;
                        padding: 20px;
                        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                    }
                    h1 {
                        color: #333;
                        border-bottom: 2px solid #4CAF50;
                        padding-bottom: 10px;
                    }
                    .status {
                        color: #4CAF50;
                        font-weight: bold;
                    }
                    .endpoint {
                        background-color: #e8f5e9;
                        padding: 10px;
                        border-radius: 4px;
                        margin: 10px 0;
                    }
                </style>
            </head>
            <body>
                <div class="container">
                    <h1>ISET API Proxy</h1>
                    <p class="status">✅ El servicio está funcionando correctamente</p>
                    <p>Esta API actúa como proxy para el servidor Orthanc.</p>

                    <div class="endpoint">
                        <strong>Endpoint:</strong> <code>/api/find</code><br>
                        <strong>Método:</strong> <code>POST</code><br>
                        <strong>Descripción:</strong> Consulta/búsqueda de estudios
                    </div>

                    <div class="endpoint">
                        <strong>Endpoint:</strong> <code>/api/studies/:studyID/archive</code><br>
                        <strong>Método:</strong> <code>GET</code><br>
                        <strong>Descripción:</strong> Descarga de estudios como ZIP desde Orthanc<br>
                        <strong>Parámetros:</strong> <code>?filename=nombre.zip</code>
                    </div>

                    <div class="endpoint">
                        <strong>Endpoint:</strong> <code>/api/isetarc/:studyInstanceUID</code><br>
                        <strong>Método:</strong> <code>GET</code><br>
                        <strong>Descripción:</strong> Descarga de estudios desde dcm4chee-arc<br>
                        <strong>Parámetros:</strong> <code>?accept=application/zip&dicomdir=true&filename=nombre.zip</code>
                    </div>

                    <p><strong>Estado:</strong> <code>Activo</code></p>
                </div>
            </body>
        </html>
    `);
});

// Ruta POST para las peticiones de búsqueda (find)
app.post('/api/find', async (req, res) => {
    try {
        const response = await axios({
            method: 'post',
            url: `${ORTHANC_URL}/tools/find`,
            auth: ORTHANC_AUTH,
            data: req.body,
            headers: {
                'Content-Type': 'application/json'
            }
        });
        res.json(response.data);
    } catch (error) {
        console.error('Error en la petición /api/find:', error.message);
        res.status(500).json({
            error: 'Error al procesar la petición',
            details: error.message
        });
    }
});

// Ruta de prueba para verificar que el proxy funciona
app.get('/test', (req, res) => {
    res.json({
        status: 'ok',
        message: 'El proxy está funcionando correctamente',
        timestamp: new Date().toISOString()
    });
});

// Ruta GET para /api/studies (información)
// IMPORTANTE: Esta ruta debe ir ANTES de /api/studies/* para que tenga prioridad
// También debe ir ANTES de /api/studies/:studyID/archive
app.get('/api/studies', (req, res) => {
    console.log('✅ Petición GET recibida en /api/studies (ruta exacta)');
    console.log('Request path:', req.path);
    console.log('Request originalUrl:', req.originalUrl);
    console.log('Request url:', req.url);
    res.send(`
        <html>
            <head>
                <title>ISET API Proxy - Estudios</title>
                <style>
                    body {
                        font-family: Arial, sans-serif;
                        max-width: 800px;
                        margin: 40px auto;
                        padding: 20px;
                        line-height: 1.6;
                    }
                    .container {
                        background-color: #f5f5f5;
                        border-radius: 8px;
                        padding: 20px;
                        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                    }
                    h1 {
                        color: #333;
                        border-bottom: 2px solid #4CAF50;
                        padding-bottom: 10px;
                    }
                    .status {
                        color: #4CAF50;
                        font-weight: bold;
                    }
                    .endpoint {
                        background-color: #e8f5e9;
                        padding: 10px;
                        border-radius: 4px;
                        margin: 10px 0;
                    }
                    .warning {
                        background-color: #fff3cd;
                        padding: 10px;
                        border-radius: 4px;
                        margin: 10px 0;
                        border-left: 4px solid #ffc107;
                    }
                    code {
                        background-color: #f0f0f0;
                        padding: 2px 6px;
                        border-radius: 3px;
                        font-family: 'Courier New', monospace;
                    }
                </style>
            </head>
            <body>
                <div class="container">
                    <h1>ISET API Proxy - Estudios</h1>
                    <p class="status">✅ El servicio está funcionando correctamente</p>

                    <div class="warning">
                        <strong>⚠️ Nota:</strong> Este endpoint requiere un <code>studyID</code> específico.
                    </div>

                    <div class="endpoint">
                        <strong>Endpoint de descarga (Orthanc):</strong><br>
                        <code>GET /api/studies/:studyID/archive?filename=nombre.zip</code><br>
                        <strong>Ejemplo:</strong><br>
                        <code>/api/studies/abc123-def456-ghi789/archive?filename=paciente.zip</code>
                    </div>

                    <div class="endpoint">
                        <strong>Endpoint de descarga (dcm4chee-arc):</strong><br>
                        <code>GET /api/isetarc/:studyInstanceUID?accept=application/zip&dicomdir=true&filename=nombre.zip</code><br>
                        <strong>Ejemplo:</strong><br>
                        <code>/api/isetarc/1.2.840.113619.2.55.3.123456789?accept=application/zip&dicomdir=true&filename=paciente.zip</code>
                    </div>

                    <div class="endpoint">
                        <strong>Otros endpoints disponibles:</strong><br>
                        <code>GET /api/studies/:studyID</code> - Información del estudio<br>
                        <code>GET /api/studies/:studyID/archive</code> - Descarga del estudio (Orthanc)<br>
                        <code>GET /api/isetarc/:studyInstanceUID</code> - Descarga del estudio (dcm4chee-arc)<br>
                        <code>POST /api/find</code> - Búsqueda de estudios
                    </div>

                    <p><strong>Estado:</strong> <code>Activo</code></p>
                </div>
            </body>
        </html>
    `);
});

// Función auxiliar para descargar estudios desde Orthanc (método original sin DICOMDIR)
async function downloadStudyArchive(studyID, filename, req, res) {
    try {
        // Construir la URL de Orthanc usando el método original
        const orthancUrl = `${ORTHANC_URL}/studies/${studyID}/archive`;
        
        console.log(`📦 Descargando estudio ${studyID} desde Orthanc usando /studies/{id}/archive`);
        console.log('URL:', orthancUrl);
        
        // Hacer la petición GET a Orthanc con autenticación
        const response = await axios({
            method: 'get',
            url: orthancUrl,
            auth: ORTHANC_AUTH,
            responseType: 'stream', // Importante: stream para archivos binarios
            headers: {
                'Accept': 'application/zip'
            }
        });
        
        // Configurar headers de respuesta
        res.setHeader('Content-Type', 'application/zip');
        if (filename) {
            res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
        } else {
            res.setHeader('Content-Disposition', `attachment; filename="study_${studyID}.zip"`);
        }
        
        // Pasar el stream directamente al cliente
        response.data.pipe(res);
        
    } catch (error) {
        console.error('Error en la descarga del estudio desde Orthanc:', error.message);
        if (error.response) {
            res.status(error.response.status).json({
                error: 'Error al descargar el estudio',
                details: error.response.data || error.message
            });
        } else {
            res.status(500).json({
                error: 'Error al procesar la descarga',
                details: error.message
            });
        }
    }
}

// Función auxiliar para descargar estudios desde dcm4chee-arc
async function downloadFromDcm4cheeArc(studyInstanceUID, filename, req, res) {
    try {
        // Construir la URL de dcm4chee-arc
        // Formato: /dcm4chee-arc/aets/AET/rs/studies/{StudyInstanceUID}
        const dcm4cheeUrl = `${DCM4CHEE_ARC_URL}/dcm4chee-arc/aets/${DCM4CHEE_ARC_AET}/rs/studies/${studyInstanceUID}`;
        
        // Obtener parámetros de query string
        const accept = req.query.accept || 'application/zip';
        const dicomdir = req.query.dicomdir || 'false';
        
        // Construir URL completa con parámetros
        const queryParams = new URLSearchParams();
        if (accept) queryParams.append('accept', accept);
        if (dicomdir) queryParams.append('dicomdir', dicomdir);
        const fullUrl = queryParams.toString() ? `${dcm4cheeUrl}?${queryParams.toString()}` : dcm4cheeUrl;
        
        console.log(`📦 Descargando estudio ${studyInstanceUID} desde dcm4chee-arc`);
        console.log('URL:', fullUrl);
        
        // Personalizar el nombre del archivo descargado ANTES de hacer la petición
        // Si no se proporciona filename, usar un nombre por defecto
        let downloadFilename = filename;
        if (!downloadFilename) {
            // Generar nombre por defecto basado en StudyInstanceUID
            downloadFilename = `study_${studyInstanceUID}.zip`;
        }
        
        // Asegurar que el nombre del archivo tenga extensión .zip si no la tiene
        if (!downloadFilename.endsWith('.zip')) {
            downloadFilename += '.zip';
        }
        
        console.log(`📥 Nombre de archivo: ${downloadFilename}`);
        
        // Hacer la petición GET a dcm4chee-arc
        // Nota: dcm4chee-arc puede requerir autenticación, ajustar según sea necesario
        const response = await axios({
            method: 'get',
            url: fullUrl,
            responseType: 'stream', // Importante: stream para archivos binarios
            headers: {
                'Accept': accept
            },
            // Timeout más largo para descargas grandes
            timeout: 300000 // 5 minutos
        });
        
        // Configurar headers de respuesta ANTES de hacer el pipe
        // IMPORTANTE: Los headers deben establecerse antes de cualquier escritura
        
        // Preparar el nombre del archivo seguro
        const safeFilename = downloadFilename.replace(/[^a-zA-Z0-9._-]/g, '_');
        const encodedFilename = encodeURIComponent(downloadFilename);
        
        // Establecer headers usando writeHead para mayor control
        // Esto asegura que los headers se establezcan antes de cualquier escritura
        const headers = {
            'Content-Type': accept,
            'Content-Disposition': `attachment; filename="${safeFilename}"; filename*=UTF-8''${encodedFilename}`
        };
        
        // Si hay headers de Content-Length en la respuesta, pasarlos también
        if (response.headers['content-length']) {
            headers['Content-Length'] = response.headers['content-length'];
        }
        
        console.log(`📤 Headers establecidos:`, headers);
        console.log(`📤 Content-Disposition: ${headers['Content-Disposition']}`);
        
        // Establecer los headers usando writeHead (más explícito)
        res.writeHead(200, headers);
        
        // Pasar el stream directamente al cliente
        // Los headers ya están establecidos con writeHead, así que no pueden ser sobrescritos
        response.data.pipe(res);
        
    } catch (error) {
        console.error('Error en la descarga del estudio desde dcm4chee-arc:', error.message);
        if (error.response) {
            // Extraer información del error sin referencias circulares
            let errorDetails = error.message;
            let errorData = null;
            
            // Intentar leer el error de dcm4chee-arc
            if (error.response.data) {
                try {
                    if (typeof error.response.data === 'string') {
                        errorData = error.response.data;
                        errorDetails = error.response.data;
                    } else if (typeof error.response.data === 'object') {
                        // Intentar serializar solo los datos, no el objeto completo
                        errorData = JSON.parse(JSON.stringify(error.response.data));
                        errorDetails = JSON.stringify(error.response.data);
                    }
                } catch (e) {
                    errorDetails = error.message;
                }
            }
            
            console.error(`❌ Error ${error.response.status} de dcm4chee-arc:`, errorDetails);
            console.error('URL solicitada:', fullUrl);
            
            res.status(error.response.status).json({
                error: 'Error al descargar el estudio desde dcm4chee-arc',
                details: errorDetails,
                status: error.response.status
            });
        } else {
            console.error('Error sin respuesta de dcm4chee-arc:', error.message);
            res.status(500).json({
                error: 'Error al procesar la descarga desde dcm4chee-arc',
                details: error.message
            });
        }
    }
}

// Ruta GET para descargar estudios desde Orthanc (archive)
// Maneja: /api/studies/:studyID/archive?filename=nombre.zip
app.get('/api/studies/:studyID/archive', async (req, res) => {
    const { studyID } = req.params;
    const { filename } = req.query;
    await downloadStudyArchive(studyID, filename, req, res);
});

// Ruta GET para descargar estudios desde dcm4chee-arc (isetarc)
// Maneja: /api/isetarc/:studyInstanceUID?accept=application/zip&dicomdir=true&filename=nombre.zip
app.get('/api/isetarc/:studyInstanceUID', async (req, res) => {
    const { studyInstanceUID } = req.params;
    const { filename } = req.query;
    await downloadFromDcm4cheeArc(studyInstanceUID, filename, req, res);
});

// Ruta catch-all para otras peticiones a /api/studies
// IMPORTANTE: Esta ruta debe ir DESPUÉS de las rutas específicas
// NOTA: Esta ruta NO debe capturar /api/studies exacto, solo /api/studies/*
// EXCLUYE las rutas /archive que se manejan arriba
app.all('/api/studies/*', async (req, res) => {
    // NO manejar rutas /archive aquí, ya tienen su propia ruta específica
    // Si llegamos aquí con /archive, significa que la ruta específica no lo capturó
    // En ese caso, devolver un error 404
    if (req.path.includes('/archive')) {
        console.log(`⚠️ Ruta /archive detectada en catch-all - esto no debería pasar`);
        return res.status(404).json({
            error: 'Ruta no encontrada',
            message: 'La ruta /archive debería ser manejada por la ruta específica'
        });
    }
    
    console.log(`📥 Petición ${req.method} recibida en /api/studies/* (catch-all): ${req.path}`);
    console.log('Original URL:', req.originalUrl);
    try {
        // Extraer la ruta después de /api/studies
        const path = req.path.replace('/api/studies', '');
        const orthancUrl = `${ORTHANC_URL}/studies${path}`;
        console.log(`🔄 Proxying a Orthanc: ${orthancUrl}`);

        // Pasar los query parameters
        const queryString = new URLSearchParams(req.query).toString();
        const fullUrl = queryString ? `${orthancUrl}?${queryString}` : orthancUrl;

        const response = await axios({
            method: req.method.toLowerCase(),
            url: fullUrl,
            auth: ORTHANC_AUTH,
            data: req.body,
            headers: {
                'Content-Type': req.get('Content-Type') || 'application/json'
            },
            // Si es una descarga, usar stream
            responseType: req.path.includes('/archive') ? 'stream' : 'json'
        });

        // Si es stream, pasarlo directamente
        if (response.data.pipe) {
            response.data.pipe(res);
        } else {
            res.json(response.data);
        }

    } catch (error) {
        console.error('Error en la petición a /api/studies:', error.message);
        if (error.response) {
            res.status(error.response.status).json({
                error: 'Error en la petición',
                details: error.response.data || error.message
            });
        } else {
            res.status(500).json({
                error: 'Error al procesar la petición',
                details: error.message
            });
        }
    }
});

// Iniciar el servidor
app.listen(PORT, () => {
    console.log(`Servidor proxy ejecutándose en el puerto ${PORT}`);
    console.log(`Endpoints disponibles:`);
    console.log(`  - POST /api/find - Búsqueda de estudios`);
    console.log(`  - GET /api/studies/:studyID/archive - Descarga de estudios (Orthanc)`);
    console.log(`  - GET /api/isetarc/:studyInstanceUID - Descarga de estudios (dcm4chee-arc)`);
    console.log(`  - ALL /api/studies/* - Proxy general para Orthanc`);
});

