-- =====================================================
-- MIGRACIÓN COMPLETA PARA FUNCIONALIDAD DE DERIVACIONES OCULTAS
-- =====================================================
-- Fecha: 2025-01-08
-- Descripción: Script completo para replicar la funcionalidad de derivaciones ocultas
-- Compatible con: MySQL 5.7+
-- Base de datos: login
-- Tabla: derivaciones
-- =====================================================

-- 1. AGREGAR COLUMNA PARA DERIVACIONES OCULTAS
-- =====================================================
-- Esta columna almacena un array JSON con los IDs de usuarios que han ocultado esta derivación
-- Ejemplo: [23, 45, 67] significa que los usuarios 23, 45 y 67 han ocultado esta derivación

ALTER TABLE derivaciones 
ADD COLUMN oculta_por_usuario JSON DEFAULT NULL 
COMMENT 'Array JSON de IDs de usuarios que han ocultado esta derivación';

-- =====================================================
-- VERIFICACIÓN DE LA MIGRACIÓN
-- =====================================================

-- Verificar que la columna se agregó correctamente
DESCRIBE derivaciones;

-- Verificar la estructura de la tabla
SHOW CREATE TABLE derivaciones;

-- =====================================================
-- CONSULTAS DE EJEMPLO PARA VERIFICAR FUNCIONALIDAD
-- =====================================================

-- 1. Obtener derivaciones recibidas NORMALES (no ocultas por el usuario 23)
-- SELECT d.*, u.nombre as usuario_origen_nombre, u.email as usuario_origen_email 
-- FROM derivaciones d 
-- JOIN usuarios u ON d.usuario_origen_id = u.id 
-- WHERE d.usuario_destino_id = 23 AND d.activo = 1 
-- AND (d.oculta_por_usuario IS NULL OR JSON_CONTAINS(d.oculta_por_usuario, CAST(23 AS JSON)) = 0)
-- ORDER BY d.fecha_derivacion DESC;

-- 2. Obtener derivaciones recibidas OCULTAS (ocultas por el usuario 23)
-- SELECT d.*, u.nombre as usuario_origen_nombre, u.email as usuario_origen_email 
-- FROM derivaciones d 
-- JOIN usuarios u ON d.usuario_origen_id = u.id 
-- WHERE d.usuario_destino_id = 23 AND d.activo = 1 
-- AND d.oculta_por_usuario IS NOT NULL AND JSON_CONTAINS(d.oculta_por_usuario, CAST(23 AS JSON)) = 1
-- ORDER BY d.fecha_derivacion DESC;

-- 3. Ocultar una derivación (agregar usuario al array JSON)
-- UPDATE derivaciones 
-- SET oculta_por_usuario = JSON_ARRAY_APPEND(COALESCE(oculta_por_usuario, '[]'), '$', 23) 
-- WHERE id = 48 AND usuario_destino_id = 23;

-- 4. Mostrar una derivación (quitar usuario del array JSON)
-- UPDATE derivaciones 
-- SET oculta_por_usuario = JSON_REMOVE(oculta_por_usuario, JSON_UNQUOTE(JSON_SEARCH(oculta_por_usuario, 'one', 23))) 
-- WHERE id = 48 AND usuario_destino_id = 23;

-- =====================================================
-- NOTAS IMPORTANTES PARA MYSQL 5.7
-- =====================================================

-- 1. COMPATIBILIDAD JSON:
--    - MySQL 5.7+ soporta completamente el tipo de datos JSON
--    - Las funciones JSON_CONTAINS, JSON_ARRAY_APPEND, JSON_REMOVE están disponibles
--    - JSON_SEARCH también está disponible para búsquedas

-- 2. FUNCIONES JSON UTILIZADAS:
--    - JSON_CONTAINS(): Verifica si un valor existe en un array JSON
--    - JSON_ARRAY_APPEND(): Agrega un elemento al final de un array JSON
--    - JSON_REMOVE(): Remueve un elemento de un array JSON
--    - JSON_SEARCH(): Busca la posición de un valor en un array JSON
--    - JSON_UNQUOTE(): Remueve las comillas de un string JSON

-- 3. EJEMPLOS DE VALORES EN LA COLUMNA:
--    - NULL: La derivación no ha sido ocultada por ningún usuario
--    - '[]': Array vacío (equivalente a NULL)
--    - '[23]': Solo el usuario 23 ha ocultado esta derivación
--    - '[23, 45, 67]': Los usuarios 23, 45 y 67 han ocultado esta derivación

-- 4. ÍNDICES RECOMENDADOS (OPCIONAL):
--    Si tienes muchas derivaciones, puedes agregar un índice virtual:
--    ALTER TABLE derivaciones ADD INDEX idx_oculta_usuario ((CAST(oculta_por_usuario AS CHAR(255) ARRAY)));

-- =====================================================
-- SCRIPT DE ROLLBACK (POR SI NECESITAS REVERTIR)
-- =====================================================

-- Para revertir la migración (eliminar la columna):
-- ALTER TABLE derivaciones DROP COLUMN oculta_por_usuario;

-- =====================================================
-- FIN DEL SCRIPT DE MIGRACIÓN
-- =====================================================
