<?php
// =====================================================
// CONFIGURACIÓN DE SERVIDORES ORTHANC
// =====================================================
// Fecha: 2025-01-08
// Descripción: Configuración centralizada de URLs de Orthanc
// =====================================================

class OrthancConfig {
    
    // =====================================================
    // CONFIGURACIÓN PRINCIPAL - CAMBIAR AQUÍ LAS URLs
    // =====================================================
    
    // Servidor Orthanc principal (para consultas de estudios)
    const ORTHANC_API_URL = "https://iset.portalpaciente.com.ar/api/find";
    
    // Servidor Orthanc local (para derivaciones)
    const ORTHANC_LOCAL_URL = "http://localhost:3001/api/find";
    
    // Servidor de descarga de estudios
    const DOWNLOAD_SERVER_URL = "https://iset.portalpaciente.com.ar/api/studies";
    
    // Servidor de visualización
    const VIEWER_SERVER_URL = "https://iset.portalpaciente.com.ar/viewer";
    
    // Servidor de visualización con puerto específico
    const VIEWER_SERVER_URL_WITH_PORT = "http://iset.portalpaciente.com.ar:8042/volview";
    
    // Servidor de descarga de derivaciones
    const DERIVATION_DOWNLOAD_URL = "https://iset.portalpaciente.com.ar/api/studies";
    
    // =====================================================
    // MÉTODOS PARA OBTENER URLs
    // =====================================================
    
    /**
     * Obtiene la URL para consultas de estudios principales
     */
    public static function getMainApiUrl() {
        return self::ORTHANC_API_URL;
    }
    
    /**
     * Obtiene la URL para consultas de estudios derivados
     */
    public static function getDerivationApiUrl() {
        return self::ORTHANC_LOCAL_URL;
    }
    
    /**
     * Obtiene la URL para descargar un estudio
     */
    public static function getDownloadUrl($studyInstanceUID) {
        return self::DOWNLOAD_SERVER_URL . "/" . urlencode($studyInstanceUID) . "/archive";
    }
    
    /**
     * Obtiene la URL para visualizar un estudio
     */
    public static function getViewerUrl($studyInstanceUID) {
        return self::VIEWER_SERVER_URL . "/" . urlencode($studyInstanceUID);
    }
    
    /**
     * Obtiene la URL para visualizar un estudio con volview
     */
    public static function getVolviewUrl($studyID) {
        return self::VIEWER_SERVER_URL_WITH_PORT . "/index.html?names=[archive.zip]&urls=[../studies/{$studyID}/archive]";
    }
    
    /**
     * Obtiene la URL para descargar una derivación
     */
    public static function getDerivationDownloadUrl($studyInstanceUID) {
        return self::DERIVATION_DOWNLOAD_URL . "/" . urlencode($studyInstanceUID) . "/archive";
    }
    
    // =====================================================
    // CONFIGURACIÓN PARA DIFERENTES AMBIENTES
    // =====================================================
    
    /**
     * Configuración para ambiente de desarrollo
     */
    public static function setDevelopmentConfig() {
        // Descomentar y modificar según tu servidor de desarrollo
        /*
        self::ORTHANC_API_URL = "http://localhost:8042/api/find";
        self::ORTHANC_LOCAL_URL = "http://localhost:8042/api/find";
        self::DOWNLOAD_SERVER_URL = "http://localhost:3001/api/studies";
        self::VIEWER_SERVER_URL = "http://localhost:8042/viewer";
        self::VIEWER_SERVER_URL_WITH_PORT = "http://localhost:8042/volview";
        self::DERIVATION_DOWNLOAD_URL = "http://localhost:8042/api/studies";
        */
    }
    
    /**
     * Configuración para ambiente de producción
     */
    public static function setProductionConfig() {
        // Descomentar y modificar según tu servidor de producción
        /*
        self::ORTHANC_API_URL = "https://tu-servidor.com/api/find";
        self::ORTHANC_LOCAL_URL = "https://tu-servidor.com/api/find";
        self::DOWNLOAD_SERVER_URL = "https://tu-servidor.com/api/studies";
        self::VIEWER_SERVER_URL = "https://tu-servidor.com/viewer";
        self::VIEWER_SERVER_URL_WITH_PORT = "https://tu-servidor.com:8042/volview";
        self::DERIVATION_DOWNLOAD_URL = "https://tu-servidor.com/api/studies";
        */
    }
}

// =====================================================
// INSTRUCCIONES DE USO
// =====================================================

/*
INSTRUCCIONES PARA CAMBIAR LAS URLs:

1. CAMBIO SIMPLE (recomendado):
   - Modifica las constantes en la parte superior de esta clase
   - Cambia los valores de las constantes según tu servidor

2. CAMBIO POR AMBIENTE:
   - Descomenta y modifica los métodos setDevelopmentConfig() o setProductionConfig()
   - Llama al método correspondiente al inicio de tu aplicación

3. EJEMPLO DE CAMBIO:
   // Antes:
   const ORTHANC_API_URL = "https://iset.portalpaciente.com.ar/api/find";
   
   // Después:
   const ORTHANC_API_URL = "https://tu-nuevo-servidor.com/api/find";

4. ARCHIVOS QUE NECESITAN SER ACTUALIZADOS:
   - view/user/index.php
   - view/admin/index.php
   - view/admin/view.php
   - view/admin/download.php
   - controller/derivacionController.php
   - Todos los archivos que usen estas URLs

5. DESPUÉS DEL CAMBIO:
   - Verifica que el nuevo servidor esté funcionando
   - Prueba las funcionalidades de descarga y visualización
   - Verifica que las consultas de estudios funcionen correctamente
*/

?>
