<?php
require_once 'config/config.php';
require_once 'includes/orthanc-api.php';
require_once 'includes/functions.php';

// Verificar sesión y permisos
session_start();

// Permitir acceso tanto a pacientes como a médicos
if (!isset($_SESSION['paciente_actual']) && !isset($_SESSION['user_id'])) {
    die("Acceso no autorizado");
}

$studyId = $_GET['id'] ?? null;

if (!$studyId) {
    die("ID de estudio no proporcionado");
}

// Obtener detalles del estudio
$url = ORTHANC_LOCAL_HOST . '/studies/' . $studyId;
$ch = curl_init($url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

// Agregar autenticación
if (defined('ORTHANC_USERNAME') && defined('ORTHANC_PASSWORD')) {
    curl_setopt($ch, CURLOPT_USERPWD, ORTHANC_USERNAME . ':' . ORTHANC_PASSWORD);
}

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

if ($httpCode !== 200) {
    die("Error al obtener detalles del estudio (Código: $httpCode): " . curl_error($ch));
}

$studyDetails = json_decode($response, true);
if (empty($studyDetails) || !isset($studyDetails['Series']) || empty($studyDetails['Series'])) {
    die("No se encontraron series en el estudio");
}

// Buscar serie del informe
$reportSeriesId = null;
foreach ($studyDetails['Series'] as $seriesId) {
    $seriesUrl = ORTHANC_LOCAL_HOST . '/series/' . $seriesId;
    curl_setopt($ch, CURLOPT_URL, $seriesUrl);
    $seriesResponse = curl_exec($ch);
    
    if (curl_errno($ch)) continue;
    
    $seriesDetails = json_decode($seriesResponse, true);
    
    if (isset($seriesDetails['MainDicomTags']['SeriesDescription']) && 
        (stripos($seriesDetails['MainDicomTags']['SeriesDescription'], 'INFORME') !== false ||
         stripos($seriesDetails['MainDicomTags']['SeriesDescription'], 'OT') !== false)) {
        $reportSeriesId = $seriesId;
        break;
    }
    
    if (isset($seriesDetails['MainDicomTags']['Modality']) && 
        $seriesDetails['MainDicomTags']['Modality'] === 'OT') {
        $reportSeriesId = $seriesId;
        break;
    }
}

if (!$reportSeriesId) {
    die("No se encontró ninguna serie de tipo informe en este estudio");
}

// Obtener ID de instancia
$seriesUrl = ORTHANC_LOCAL_HOST . '/series/' . $reportSeriesId;
curl_setopt($ch, CURLOPT_URL, $seriesUrl);
$seriesResponse = curl_exec($ch);

if (curl_errno($ch)) {
    die("Error al obtener detalles de la serie: " . curl_error($ch));
}

$seriesDetails = json_decode($seriesResponse, true);

if (empty($seriesDetails) || !isset($seriesDetails['Instances']) || empty($seriesDetails['Instances'])) {
    die("No se encontraron instancias en la serie del informe");
}

$instanceId = $seriesDetails['Instances'][0];

// Obtener información del paciente
$patientName = $studyDetails['PatientMainDicomTags']['PatientName'] ?? 'Paciente';
$patientName = extractFirstName($patientName);
$studyDate = $studyDetails['MainDicomTags']['StudyDate'] ?? '';
$studyDesc = $studyDetails['MainDicomTags']['StudyDescription'] ?? 'Informe';

$pdfUrl = "pdf-proxy.php?instance_id=" . $instanceId;

curl_close($ch);
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Visor de Informes - IMAC Tucumán</title>
    <link rel="stylesheet" href="css/styles.css">
    <!-- Incluir los archivos necesarios de PDF.js -->
    <script src="js/pdfjs/build/pdf.mjs" type="module"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: Arial, sans-serif;
            background-color: #f5f5f5;
            height: 100vh;
            overflow: hidden;
        }

        .app-container {
            display: flex;
            flex-direction: column;
            height: 100vh;
        }

        .header {
            background-color: #fff;
            padding: 0.5rem 1rem;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
            z-index: 10;
        }

        .header-left {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            max-width: 70%;
        }

        .header-logo {
            height: 40px;
            width: auto;
        }

        .header-title {
            font-size: 1rem;
            color: #333;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        .btn {
            min-height: 40px;
            min-width: 40px;
            padding: 0.5rem;
            background-color: #007bff;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 0.9rem;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .btn:hover {
            background-color: #0056b3;
        }

        .btn-secondary {
            background-color: #6c757d;
        }

        .btn-secondary:hover {
            background-color: #545b62;
        }

        .pdf-container {
            flex: 1;
            position: relative;
            overflow: auto;
            background-color: #525659;
            padding: 20px;
        }

        #pdfViewer {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 20px;
        }

        .page {
            background-color: white;
            box-shadow: 0 0 10px rgba(0,0,0,0.3);
            margin: 0 auto;
        }

        .loading-indicator {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background-color: rgba(255,255,255,0.9);
            padding: 1rem 2rem;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.2);
            display: none;
        }

        /* Barra de herramientas */
        .toolbar {
            background-color: #f8f9fa;
            border-bottom: 1px solid #dee2e6;
            padding: 0.5rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 0.5rem;
        }

        .toolbar-group {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .toolbar-btn {
            background-color: transparent;
            border: 1px solid #dee2e6;
            border-radius: 4px;
            padding: 0.25rem 0.5rem;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.9rem;
            color: #495057;
        }

        .toolbar-btn:hover {
            background-color: #e9ecef;
        }

        .toolbar-btn.active {
            background-color: #e9ecef;
            border-color: #adb5bd;
        }

        .zoom-controls {
            display: flex;
            align-items: center;
            gap: 0.25rem;
        }

        .zoom-btn {
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
            background-color: transparent;
            border: 1px solid #dee2e6;
            border-radius: 4px;
            cursor: pointer;
            font-size: 1rem;
            color: #495057;
        }

        .zoom-btn:hover {
            background-color: #e9ecef;
        }

        .zoom-level {
            min-width: 60px;
            text-align: center;
            font-size: 0.9rem;
            color: #495057;
        }

        .page-info {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.9rem;
            color: #495057;
        }

        .page-input {
            width: 50px;
            text-align: center;
            padding: 0.25rem;
            border: 1px solid #dee2e6;
            border-radius: 4px;
        }

        /* Estilos para móviles */
        @media (max-width: 768px) {
            .header {
                padding: 0.4rem;
            }
            
            .header-left {
                max-width: 60%;
            }
            
            .header-logo {
                height: 30px;
            }
            
            .header-title {
                font-size: 0.8rem;
            }

            .toolbar {
                flex-direction: column;
                align-items: stretch;
            }

            .toolbar-group {
                justify-content: center;
            }
        }

        /* Pantallas muy pequeñas */
        @media (max-width: 380px) {
            .header-title {
                font-size: 0.7rem;
            }
        }
    </style>
</head>
<body>
    <div class="app-container">
        <header class="header">
            <div class="header-left">
                <img src="images/imac_logo.png" alt="IMAC Tucumán" class="header-logo">
                <div class="header-title">
                    Informe de <?php echo sanitizeInput($patientName); ?> - <?php echo formatStudyDate($studyDate); ?>
                </div>
            </div>
            <div class="header-right">
                <?php
                // Determinar de dónde viene el usuario para el botón volver
                $referer = $_SERVER['HTTP_REFERER'] ?? '';
                if (strpos($referer, 'studies.php') !== false) {
                    // Si viene de studies.php, cerrar la ventana
                    echo '<button class="btn btn-secondary" onclick="window.close()">Cerrar</button>';
                } else {
                    // Si viene de index.php o cualquier otro lugar, regresar
                    echo '<button class="btn btn-secondary" onclick="window.location.href=\'index.php?action=showResults\'">Volver</button>';
                }
                ?>
            </div>
        </header>

        <div class="toolbar">
            <div class="toolbar-group">
                <button class="toolbar-btn" id="prevPage" title="Página anterior">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                        <path fill-rule="evenodd" d="M11.354 1.646a.5.5 0 0 1 0 .708L5.707 8l5.647 5.646a.5.5 0 0 1-.708.708l-6-6a.5.5 0 0 1 0-.708l6-6a.5.5 0 0 1 .708 0z"/>
                    </svg>
                </button>
                <div class="page-info">
                    <span id="pageNum">1</span> / <span id="pageCount">1</span>
                </div>
                <button class="toolbar-btn" id="nextPage" title="Página siguiente">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                        <path fill-rule="evenodd" d="M4.646 1.646a.5.5 0 0 1 .708 0l6 6a.5.5 0 0 1 0 .708l-6 6a.5.5 0 0 1-.708-.708L10.293 8 4.646 2.354a.5.5 0 0 1 0-.708z"/>
                    </svg>
                </button>
            </div>
            
            <div class="toolbar-group">
                <div class="zoom-controls">
                    <button class="zoom-btn" id="zoomOut" title="Reducir">-</button>
                    <span class="zoom-level" id="zoomLevel">100%</span>
                    <button class="zoom-btn" id="zoomIn" title="Ampliar">+</button>
                </div>
                <button class="toolbar-btn" id="zoomFit" title="Ajustar a la página">Ajustar</button>
            </div>
            
            <div class="toolbar-group">
                <button class="toolbar-btn" id="downloadBtn" title="Descargar PDF">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                        <path d="M8 0a8 8 0 0 0-8 8 8 8 0 0 0 8 8 8 8 0 0 0 8-8 8 8 0 0 0-8-8zM4.5 7.5a.5.5 0 0 0 0 1h5.793l-2.147 2.146a.5.5 0 0 0 .708.708l3-3a.5.5 0 0 0 0-.708l-3-3a.5.5 0 1 0-.708.708L10.293 7.5H4.5z"/>
                    </svg>
                </button>
            </div>
        </div>

        <div class="pdf-container">
            <div class="loading-indicator" id="loadingIndicator">Cargando informe...</div>
            <div id="pdfViewer"></div>
        </div>
    </div>

    <script type="module">
        import * as pdfjsLib from './js/pdfjs/build/pdf.mjs';
        
        // Configurar el worker de PDF.js
        pdfjsLib.GlobalWorkerOptions.workerSrc = './js/pdfjs/build/pdf.worker.mjs';
        
        let pdfDoc = null;
        let currentPage = 1;
        let totalPages = 0;
        let currentScale = 1;
        let isPageRendering = false;
        let pageNumPending = null;
        
        const pdfUrl = '<?php echo $pdfUrl; ?>';
        const viewer = document.getElementById('pdfViewer');
        const loadingIndicator = document.getElementById('loadingIndicator');
        
        // Elementos de la barra de herramientas
        const prevButton = document.getElementById('prevPage');
        const nextButton = document.getElementById('nextPage');
        const pageNum = document.getElementById('pageNum');
        const pageCount = document.getElementById('pageCount');
        const zoomInButton = document.getElementById('zoomIn');
        const zoomOutButton = document.getElementById('zoomOut');
        const zoomLevel = document.getElementById('zoomLevel');
        const zoomFitButton = document.getElementById('zoomFit');
        const downloadButton = document.getElementById('downloadBtn');
        
        // Cargar el PDF
        loadingIndicator.style.display = 'block';
        
        console.log('Intentando cargar PDF desde:', pdfUrl);
        
        pdfjsLib.getDocument(pdfUrl).promise.then(function(pdf) {
            console.log('PDF cargado correctamente');
            pdfDoc = pdf;
            totalPages = pdf.numPages;
            
            // Actualizar información de páginas
            pageCount.textContent = totalPages;
            
            // Renderizar la primera página
            renderPage(currentPage);
        }).catch(function(error) {
            console.error('Error al cargar el PDF:', error);
            loadingIndicator.textContent = 'Error al cargar el PDF: ' + error.message;
        });
        
        function renderPage(pageNumber) {
            isPageRendering = true;
            loadingIndicator.style.display = 'block';
            
            pdfDoc.getPage(pageNumber).then(function(page) {
                const viewport = page.getViewport({ scale: currentScale });
                const canvas = document.createElement('canvas');
                const context = canvas.getContext('2d');
                
                canvas.height = viewport.height;
                canvas.width = viewport.width;
                
                const renderContext = {
                    canvasContext: context,
                    viewport: viewport
                };
                
                const renderTask = page.render(renderContext);
                
                renderTask.promise.then(function() {
                    isPageRendering = false;
                    
                    // Limpiar el visor
                    viewer.innerHTML = '';
                    
                    // Crear contenedor de página
                    const pageDiv = document.createElement('div');
                    pageDiv.className = 'page';
                    pageDiv.appendChild(canvas);
                    
                    // Agregar al visor
                    viewer.appendChild(pageDiv);
                    
                    loadingIndicator.style.display = 'none';
                    
                    // Actualizar información de página
                    pageNum.textContent = pageNumber;
                    
                    // Si hay una página pendiente, renderizarla
                    if (pageNumPending !== null) {
                        renderPage(pageNumPending);
                        pageNumPending = null;
                    }
                });
            });
        }
        
        function queueRenderPage(pageNumber) {
            if (isPageRendering) {
                pageNumPending = pageNumber;
            } else {
                renderPage(pageNumber);
            }
        }
        
        function changePage(delta) {
            const newPage = currentPage + delta;
            if (newPage >= 1 && newPage <= totalPages) {
                currentPage = newPage;
                queueRenderPage(currentPage);
            }
        }
        
        function changeZoom(scale) {
            currentScale = scale;
            zoomLevel.textContent = Math.round(scale * 100) + '%';
            queueRenderPage(currentPage);
        }
        
        function zoomFit() {
            if (!pdfDoc) return;
            
            pdfDoc.getPage(currentPage).then(function(page) {
                const viewport = page.getViewport({ scale: 1 });
                const containerWidth = viewer.clientWidth - 40; // Margen
                const containerHeight = viewer.clientHeight - 40; // Margen
                
                const scaleX = containerWidth / viewport.width;
                const scaleY = containerHeight / viewport.height;
                
                // Usar la escala más pequeña para ajustar completamente
                currentScale = Math.min(scaleX, scaleY);
                
                zoomLevel.textContent = Math.round(currentScale * 100) + '%';
                queueRenderPage(currentPage);
            });
        }
        
        function downloadPDF() {
            window.location.href = pdfUrl;
        }
        
        // Event listeners para los botones
        prevButton.addEventListener('click', function() {
            changePage(-1);
        });
        
        nextButton.addEventListener('click', function() {
            changePage(1);
        });
        
        zoomInButton.addEventListener('click', function() {
            changeZoom(currentScale + 0.25);
        });
        
        zoomOutButton.addEventListener('click', function() {
            if (currentScale > 0.25) {
                changeZoom(currentScale - 0.25);
            }
        });
        
        zoomFitButton.addEventListener('click', function() {
            zoomFit();
        });
        
        downloadButton.addEventListener('click', function() {
            downloadPDF();
        });
        
        // Ajustar al cargar la página
        window.addEventListener('load', function() {
            setTimeout(zoomFit, 500); // Esperar a que el PDF se cargue
        });
        
        // Ajustar al cambiar el tamaño de la ventana
        window.addEventListener('resize', function() {
            zoomFit();
        });
    </script>
</body>
</html>