<?php
require_once dirname(__DIR__) . '/config/config.php';

function findStudiesByPatientId($patientId) {
    $url = ORTHANC_LOCAL_HOST . '/tools/find';
    
    $data = [
        'Level' => 'Study',
        'Query' => ['PatientID' => $patientId],
        'Expand' => true
    ];
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    
    // Establecer timeout para evitar que una sola consulta bloquee todo el script
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5); // 5 segundos para conectar
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);       // 10 segundos para la respuesta total
    
    // Agregar autenticación
    if (defined('ORTHANC_USERNAME') && defined('ORTHANC_PASSWORD')) {
        curl_setopt($ch, CURLOPT_USERPWD, ORTHANC_USERNAME . ':' . ORTHANC_PASSWORD);
    }
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        return ['error' => 'Curl error: ' . curl_error($ch)];
    }
    
    curl_close($ch);
    
    return json_decode($response, true);
}

function getStudyInstances($studyId) {
    $url = ORTHANC_LOCAL_HOST . '/studies/' . $studyId . '/instances';
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    // Establecer timeout para evitar que una sola consulta bloquee todo el script
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5); // 5 segundos para conectar
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);       // 10 segundos para la respuesta total
    
    // Agregar autenticación
    if (defined('ORTHANC_USERNAME') && defined('ORTHANC_PASSWORD')) {
        curl_setopt($ch, CURLOPT_USERPWD, ORTHANC_USERNAME . ':' . ORTHANC_PASSWORD);
    }
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        return ['error' => 'Curl error: ' . curl_error($ch)];
    }
    
    curl_close($ch);
    
    return json_decode($response, true);
}

function getInstanceDetails($instanceId) {
    $url = ORTHANC_LOCAL_HOST . '/instances/' . $instanceId;
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    // Establecer timeout para evitar que una sola consulta bloquee todo el script
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5); // 5 segundos para conectar
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);       // 10 segundos para la respuesta total
    
    // Agregar autenticación
    if (defined('ORTHANC_USERNAME') && defined('ORTHANC_PASSWORD')) {
        curl_setopt($ch, CURLOPT_USERPWD, ORTHANC_USERNAME . ':' . ORTHANC_PASSWORD);
    }
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        return ['error' => 'Curl error: ' . curl_error($ch)];
    }
    
    curl_close($ch);
    
    return json_decode($response, true);
}

function getStudySeries($studyId) {
    // Si studyId es un array, extraer el ID si está disponible
    if (is_array($studyId)) {
        if (isset($studyId['ID'])) {
            $studyId = $studyId['ID'];
        } else {
            return ['error' => 'El objeto de estudio proporcionado no contiene un ID válido'];
        }
    }
    
    // Verificar que studyId sea un string válido
    if (!is_string($studyId)) {
        return ['error' => 'ID de estudio inválido: ' . gettype($studyId)];
    }
    
    $url = ORTHANC_LOCAL_HOST . '/studies/' . $studyId . '/series';
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    // Agregar autenticación
    if (defined('ORTHANC_USERNAME') && defined('ORTHANC_PASSWORD')) {
        curl_setopt($ch, CURLOPT_USERPWD, ORTHANC_USERNAME . ':' . ORTHANC_PASSWORD);
    }
    
    // Establecer timeout para evitar que una sola consulta bloquee todo el script
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5); // 5 segundos para conectar
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);       // 10 segundos para la respuesta total
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        return ['error' => 'Curl error: ' . curl_error($ch)];
    }
    
    curl_close($ch);
    
    return json_decode($response, true);
}

function getSeriesInstances($seriesId) {
    // Si seriesId es un array, extraer el ID si está disponible
    if (is_array($seriesId)) {
        if (isset($seriesId['ID'])) {
            $seriesId = $seriesId['ID'];
        } else {
            return ['error' => 'El objeto de serie proporcionado no contiene un ID válido'];
        }
    }
    
    // Verificar que seriesId sea un string válido
    if (!is_string($seriesId)) {
        return ['error' => 'ID de serie inválido: ' . gettype($seriesId)];
    }
    
    $url = ORTHANC_LOCAL_HOST . '/series/' . $seriesId . '/instances';
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    // Establecer timeout para evitar que una sola consulta bloquee todo el script
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5); // 5 segundos para conectar
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);       // 10 segundos para la respuesta total
    
    // Agregar autenticación
    if (defined('ORTHANC_USERNAME') && defined('ORTHANC_PASSWORD')) {
        curl_setopt($ch, CURLOPT_USERPWD, ORTHANC_USERNAME . ':' . ORTHANC_PASSWORD);
    }
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        return ['error' => 'Curl error: ' . curl_error($ch)];
    }
    
    curl_close($ch);
    
    return json_decode($response, true);
}

function getSeriesDetails($seriesId) {
    // Si seriesId es un array, probablemente sea un objeto de serie completo
    // Verificar si es un objeto de serie con los campos esperados
    if (is_array($seriesId) && isset($seriesId['ID']) && isset($seriesId['MainDicomTags'])) {
        // Ya tenemos los detalles, devolverlos directamente
        return $seriesId;
    }
    
    // Si no es un string, puede ser un error o un formato inesperado
    if (!is_string($seriesId)) {
        return ['error' => 'El ID de serie proporcionado no es válido: ' . gettype($seriesId)];
    }
    
    $url = ORTHANC_LOCAL_HOST . '/series/' . $seriesId;
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    // Agregar autenticación
    if (defined('ORTHANC_USERNAME') && defined('ORTHANC_PASSWORD')) {
        curl_setopt($ch, CURLOPT_USERPWD, ORTHANC_USERNAME . ':' . ORTHANC_PASSWORD);
    }
    
    // Establecer timeout para evitar que una sola consulta bloquee todo el script
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5); // 5 segundos para conectar
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);       // 10 segundos para la respuesta total
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        return ['error' => 'Curl error: ' . curl_error($ch)];
    }
    
    curl_close($ch);
    
    return json_decode($response, true);
}

// Caché para almacenar modalidades ya obtenidas
$modalityCache = [];

function getStudyMainModality($studyId) {
    global $modalityCache;
    
    // Si ya tenemos la modalidad en caché, devolverla directamente
    if (isset($modalityCache[$studyId])) {
        return $modalityCache[$studyId];
    }
    
    // Valor predeterminado en caso de falla
    $defaultModality = 'OT';
    
    // Obtener todas las series del estudio
    $series = getStudySeries($studyId);
    
    if (empty($series) || isset($series['error'])) {
        // En caso de error, devolver el valor predeterminado y no guardar en caché
        return $defaultModality;
    }
    
    // Check if $series is already a detailed object rather than an array of IDs
    if (isset($series[0]) && is_string($series[0])) {
        // It's an array of IDs, get details of the first series
        $firstSeriesId = $series[0];
        $seriesDetails = getSeriesDetails($firstSeriesId);
    } else if (isset($series[0]['ID'])) {
        // It's an array of detailed objects, use the first one directly
        $seriesDetails = $series[0];
    } else {
        // Unknown format
        return $defaultModality;
    }
    
    if (empty($seriesDetails) || isset($seriesDetails['error'])) {
        return $defaultModality;
    }
    
    // Extraer la modalidad de los MainDicomTags
    $modality = $seriesDetails['MainDicomTags']['Modality'] ?? $defaultModality;
    
    // Guardar en caché para futuras consultas
    $modalityCache[$studyId] = $modality;
    
    return $modality;
}

function getStudyStatistics($studyId) {
    // Si studyId es un array, extraer el ID si está disponible
    if (is_array($studyId)) {
        if (isset($studyId['ID'])) {
            $studyId = $studyId['ID'];
        } else {
            return ['error' => 'El objeto de estudio proporcionado no contiene un ID válido'];
        }
    }
    
    // Verificar que studyId sea un string válido
    if (!is_string($studyId)) {
        return ['error' => 'ID de estudio inválido: ' . gettype($studyId)];
    }
    
    $url = ORTHANC_LOCAL_HOST . '/studies/' . $studyId . '/statistics';
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    // Establecer timeout para evitar que una sola consulta bloquee todo el script
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5); // 5 segundos para conectar
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);       // 10 segundos para la respuesta total
    
    // Agregar autenticación
    if (defined('ORTHANC_USERNAME') && defined('ORTHANC_PASSWORD')) {
        curl_setopt($ch, CURLOPT_USERPWD, ORTHANC_USERNAME . ':' . ORTHANC_PASSWORD);
    }
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        return ['error' => 'Curl error: ' . curl_error($ch)];
    }
    
    curl_close($ch);
    
    return json_decode($response, true);
}

function debugSeriesModality($studyId) {
    // Obtener todas las series del estudio
    $series = getStudySeries($studyId);
    
    if (empty($series) || isset($series['error'])) {
        echo "<div style='background-color: #ffdddd; padding: 10px; margin: 10px; border: 1px solid #ff0000;'>";
        echo "<h3>Error al obtener series del estudio</h3>";
        echo "ID de estudio: $studyId<br>";
        echo "Error: " . (isset($series['error']) ? $series['error'] : 'No hay series disponibles');
        echo "</div>";
        return;
    }
    
    // Obtener detalles de la primera serie
    $firstSeriesId = $series[0];
    $seriesDetails = getSeriesDetails($firstSeriesId);
    
    echo "<div style='background-color: #eeffee; padding: 10px; margin: 10px; border: 1px solid #00aa00;'>";
    echo "<h3>Depuración de modalidad</h3>";
    echo "ID de estudio: $studyId<br>";
    echo "ID de primera serie: $firstSeriesId<br><br>";
    
    if (empty($seriesDetails) || isset($seriesDetails['error'])) {
        echo "<span style='color: red;'>Error al obtener detalles de la serie: " . 
             (isset($seriesDetails['error']) ? $seriesDetails['error'] : 'No hay detalles disponibles') . "</span>";
    } else {
        echo "<h4>Detalles de la serie:</h4>";
        
        // Mostrar tags DICOM principales
        if (isset($seriesDetails['MainDicomTags'])) {
            echo "<h5>MainDicomTags:</h5>";
            echo "<pre style='background-color: #f5f5f5; padding: 10px; overflow: auto;'>";
            print_r($seriesDetails['MainDicomTags']);
            echo "</pre>";
            
            // Destacar específicamente la modalidad
            if (isset($seriesDetails['MainDicomTags']['Modality'])) {
                echo "<div style='background-color: #ffffcc; padding: 10px; border: 1px solid #dddd00;'>";
                echo "<strong>Modalidad encontrada:</strong> " . $seriesDetails['MainDicomTags']['Modality'];
                echo "</div>";
            } else {
                echo "<div style='background-color: #ffdddd; padding: 10px; border: 1px solid #ff0000;'>";
                echo "<strong>Modalidad no encontrada en MainDicomTags</strong>";
                echo "</div>";
            }
        } else {
            echo "<div style='background-color: #ffdddd; padding: 10px; border: 1px solid #ff0000;'>";
            echo "<strong>No se encontró la sección MainDicomTags en la respuesta</strong>";
            echo "</div>";
        }
        
        // Mostrar estructura completa para depuración avanzada
        echo "<h5>Estructura completa de la serie:</h5>";
        echo "<div style='max-height: 400px; overflow: auto;'>";
        echo "<pre style='background-color: #f5f5f5; padding: 10px;'>";
        print_r($seriesDetails);
        echo "</pre>";
        echo "</div>";
    }
    
    echo "</div>";
}

function getAllStudies($startDate, $endDate) {
    $ch = curl_init();
    
    // Convert dates to Orthanc format (YYYYMMDD)
    $startDateOrthanc = str_replace('-', '', $startDate);
    $endDateOrthanc = str_replace('-', '', $endDate);
    
    // Build the query URL with date filter
    $url = ORTHANC_HOST . "studies?limit=1000&since=" . $startDateOrthanc . "&until=" . $endDateOrthanc;
    
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_USERPWD, ORTHANC_USERNAME . ":" . ORTHANC_PASSWORD);
    
    // Establecer timeout para evitar que una sola consulta bloquee todo el script
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5); // 5 segundos para conectar
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);       // 20 segundos para la respuesta total (más tiempo para lista grande)
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if ($httpCode !== 200) {
        return ['error' => 'Error al obtener los estudios: ' . $httpCode];
    }
    
    $studies = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        return ['error' => 'Error al decodificar la respuesta JSON'];
    }
    
    // Get detailed information for each study
    $detailedStudies = [];
    foreach ($studies as $study) {
        // Obtener detalles del estudio
        $studyDetails = getStudyDetails($study);
        
        // Si no hay error y el estudio tiene los datos necesarios, lo agregamos
        if (!isset($studyDetails['error']) && 
            isset($studyDetails['MainDicomTags']) && 
            isset($studyDetails['PatientMainDicomTags'])) {
            $detailedStudies[] = $studyDetails;
        }
    }
    
    curl_close($ch);
    return $detailedStudies;
}

function getStudyDetails($study) {
    $ch = curl_init();
    
    // Si $study es un ID, obtener los detalles
    if (is_string($study)) {
        $url = ORTHANC_HOST . "studies/" . $study;
    } else {
        // Si ya es un objeto detallado, devolverlo directamente
        return $study;
    }
    
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_USERPWD, ORTHANC_USERNAME . ":" . ORTHANC_PASSWORD);
    
    // Establecer timeout para evitar que una sola consulta bloquee todo el script
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5); // 5 segundos para conectar
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);       // 10 segundos para la respuesta total
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if ($httpCode !== 200) {
        return ['error' => 'Error al obtener detalles del estudio: ' . $httpCode];
    }
    
    $studyDetails = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        return ['error' => 'Error al decodificar la respuesta JSON'];
    }
    
    curl_close($ch);
    return $studyDetails;
}
