<?php
session_start();

require_once 'config/config.php';
require_once 'includes/functions.php';
require_once 'includes/orthanc-api.php';

// Verificar sesión
checkSession();

// Set default date range (today)
$endDate = date('Y-m-d');
$startDate = date('Y-m-d');

// Variables para los nuevos filtros
$patientName = '';
$patientId = '';

// Handle date filter form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['start_date']) && isset($_POST['end_date'])) {
        $startDate = sanitizeInput($_POST['start_date']);
        $endDate = sanitizeInput($_POST['end_date']);
    }
    // Capturar filtros de nombre e ID de paciente
    if (isset($_POST['patient_name'])) {
        $patientName = sanitizeInput($_POST['patient_name']);
    }
    if (isset($_POST['patient_id'])) {
        $patientId = sanitizeInput($_POST['patient_id']);
    }
}

// Get studies from Orthanc
$estudios = [];
$mensajeError = "";

// Convert dates to Orthanc format (YYYYMMDD)
$startDateOrthanc = str_replace('-', '', $startDate);
$endDateOrthanc = str_replace('-', '', $endDate);

// Build the query data
$query = [
    'StudyDate' => $startDateOrthanc . '-' . $endDateOrthanc
];

// Agregar filtros de nombre e ID de paciente si fueron proporcionados
if (!empty($patientName)) {
    $query['PatientName'] = '*' . $patientName . '*';
}
if (!empty($patientId)) {
    $query['PatientID'] = $patientId;
}

$data = [
    'Level' => 'Study',
    'Expand' => true,
    'Query' => $query
];

// Initialize cURL
$ch = curl_init(ORTHANC_HOST . 'tools/find');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
curl_setopt($ch, CURLOPT_USERPWD, ORTHANC_USERNAME . ":" . ORTHANC_PASSWORD);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

if ($httpCode !== 200) {
    $mensajeError = "Error al obtener los estudios: " . $httpCode;
} else {
    $studies = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        $mensajeError = "Error al decodificar la respuesta JSON";
    } else {
        // Array temporal para almacenar todos los estudios
        $todosLosEstudios = [];
        
        // Get detailed information for each study
        foreach ($studies as $study) {
            // Obtener detalles del estudio
            $studyId = is_array($study) ? $study['ID'] : $study;
            $studyDetails = getStudyDetails($studyId);
            
            // Obtener estadísticas del estudio
            $studyStats = getStudyStatistics($studyId);
            
            // Si no hay error y el estudio tiene los datos necesarios
            if (!isset($studyDetails['error']) && 
                isset($studyDetails['MainDicomTags']) && 
                isset($studyDetails['PatientMainDicomTags'])) {
                
                // Agregar estadísticas al estudio
                if ($studyStats !== null) {
                    $studyDetails['Statistics'] = $studyStats;
                    $studyDetails['SeriesCount'] = $studyStats['CountSeries'];
                    $studyDetails['TotalImageCount'] = $studyStats['CountInstances'];
                }
                
                $todosLosEstudios[] = $studyDetails;
            }
        }
        
        // Filtrar estudios: mostrar solo imágenes o informes que contengan "PDF" en su descripción
        $estudiosFiltrados = [];
        foreach ($todosLosEstudios as $estudio) {
            $mainTags = $estudio['MainDicomTags'] ?? [];
            $studyDesc = $mainTags['StudyDescription'] ?? '';
            $studyId = $estudio['ID'] ?? '';
            $modality = getStudyMainModality($studyId);
            
            // Determinar si es un informe
            $isReport = false;
            if ($modality === 'OT') {
                // Solo incluir estudios OT que contengan PDF en su descripción
                if (stripos($studyDesc, 'PDF') !== false) {
                    $isReport = true;
                } else {
                    // Si es OT pero no contiene PDF, saltamos este estudio
                    continue;
                }
            }
            
            // Obtener información sobre series e imágenes solo para estudios de imágenes (no informes)
            if (!$isReport) {
                // Usar las estadísticas si están disponibles
                if (isset($estudio['Statistics'])) {
                    $estudio['SeriesCount'] = $estudio['Statistics']['CountSeries'];
                    $estudio['TotalImageCount'] = $estudio['Statistics']['CountInstances'];
                } else {
                    // Obtener las series del estudio (fallback)
                    $series = getStudySeries($studyId);
                    
                    if (!isset($series['error']) && is_array($series)) {
                        // Contar número de series
                        $estudio['SeriesCount'] = count($series);
                        
                        // Contar imágenes por serie
                        $estudio['SeriesDetails'] = [];
                        $totalImages = 0;
                        
                        foreach ($series as $serieId) {
                            // Verificar si serieId es un ID o un objeto completo de serie
                            $serieIdString = is_array($serieId) && isset($serieId['ID']) ? $serieId['ID'] : $serieId;
                            
                            $instances = getSeriesInstances($serieIdString);
                            if (!isset($instances['error']) && is_array($instances)) {
                                $imageCount = count($instances);
                                $totalImages += $imageCount;
                                
                                // Obtener detalles de la serie
                                $serieDetails = getSeriesDetails($serieId);
                                if (!isset($serieDetails['error'])) {
                                    $serieDescription = $serieDetails['MainDicomTags']['SeriesDescription'] ?? 'Sin descripción';
                                    $serieNumber = $serieDetails['MainDicomTags']['SeriesNumber'] ?? '';
                                    $serieModality = $serieDetails['MainDicomTags']['Modality'] ?? '';
                                    
                                    $estudio['SeriesDetails'][] = [
                                        'ID' => $serieIdString,
                                        'Description' => $serieDescription,
                                        'Number' => $serieNumber,
                                        'Modality' => $serieModality,
                                        'ImageCount' => $imageCount
                                    ];
                                }
                            }
                        }
                        
                        // Guardar el total de imágenes
                        $estudio['TotalImageCount'] = $totalImages;
                    }
                }
            } else {
                // Para informes, establecer valores predeterminados
                $estudio['SeriesCount'] = 1;
                $estudio['TotalImageCount'] = 1;
                $estudio['SeriesDetails'] = [];
            }
            
            // Si NO es un informe, o si es un informe y contiene "PDF", lo incluimos
            if (!$isReport || (stripos($studyDesc, 'PDF') !== false)) {
                $estudiosFiltrados[] = $estudio;
            }
        }
        
        // Ordenar estudios: primero estudios, luego informes, agrupados por ID de paciente
        usort($estudiosFiltrados, function($a, $b) {
            // Primero comparar por ID de paciente
            $patientIdA = $a['PatientMainDicomTags']['PatientID'] ?? '';
            $patientIdB = $b['PatientMainDicomTags']['PatientID'] ?? '';
            
            $patientCompare = strcmp($patientIdA, $patientIdB);
            if ($patientCompare !== 0) {
                return $patientCompare; // Diferentes pacientes
            }
            
            // Mismo paciente, ahora determinar si son estudios o informes
            $studyDescA = $a['MainDicomTags']['StudyDescription'] ?? '';
            $studyDescB = $b['MainDicomTags']['StudyDescription'] ?? '';
            
            $modalityA = getStudyMainModality($a['ID']);
            $modalityB = getStudyMainModality($b['ID']);
            
            $isReportA = (stripos($studyDescA, 'INFORME') !== false || $modalityA === 'OT');
            $isReportB = (stripos($studyDescB, 'INFORME') !== false || $modalityB === 'OT');
            
            // Si uno es informe y el otro no, el que no es informe va primero
            if ($isReportA && !$isReportB) {
                return 1; // A es informe, B no lo es, B va primero
            }
            if (!$isReportA && $isReportB) {
                return -1; // A no es informe, B lo es, A va primero
            }
            
            // Si ambos son del mismo tipo (estudio o informe), ordenar por fecha más reciente primero
            $dateA = $a['MainDicomTags']['StudyDate'] ?? '';
            $dateB = $b['MainDicomTags']['StudyDate'] ?? '';
            
            return strcmp($dateB, $dateA); // Orden descendente por fecha
        });
        
        // Si no hay filtro, mostramos todos
        $estudios = $estudiosFiltrados;
    }
}

curl_close($ch);

// Convert estudios array to JSON for JavaScript
$estudiosJson = json_encode($estudios);
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Estudios Médicos - Portal de Imágenes</title>
    <link rel="stylesheet" href="css/styles.css">
    <style>
        .filter-container {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            padding: 20px;
            background: #f8f9fa;
            margin-bottom: 20px;
            align-items: flex-end;
        }
        
        .date-filter-form {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            align-items: flex-end;
            flex-grow: 1;
        }
        
        .search-container {
            display: flex;
            gap: 10px;
            flex-grow: 1;
            max-width: 300px;
        }
        
        .search-container input {
            flex-grow: 1;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        
        .filter-switch {
            display: flex;
            align-items: center;
            margin-left: 15px;
        }
        
        .filter-switch a {
            display: inline-block;
            padding: 8px 12px;
            background-color: <?php echo $mostrarSoloPDF ? '#007bff' : '#f8f9fa'; ?>;
            color: <?php echo $mostrarSoloPDF ? '#fff' : '#495057'; ?>;
            text-decoration: none;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 0.9rem;
        }
        
        .hide {
            display: none !important;
        }
        
        #no-results {
            padding: 20px;
            text-align: center;
            font-size: 1.1em;
            color: #666;
            display: none;
        }
        
        /* Estilos para filtros de modalidad */
        .modality-filters {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-top: 15px;
            align-items: center;
            width: 100%;
        }
        
        .modality-filters-title {
            font-size: 0.9rem;
            font-weight: bold;
            margin-right: 10px;
        }
        
        .modality-button {
            display: inline-block;
            cursor: pointer;
            user-select: none;
            background-color: #f1f1f1;
            padding: 8px 15px;
            border-radius: 4px;
            transition: all 0.2s ease;
            border: 1px solid #ddd;
            font-size: 0.9rem;
            text-align: center;
        }
        
        .modality-button.active {
            background-color: #28a745;
            color: white;
            border-color: #28a745;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .modality-button:hover {
            background-color: #e2e2e2;
        }
        
        .modality-button.active:hover {
            background-color: #218838;
        }
        
        /* Estilos para cabeceras de tabla ordenables */
        .estudios-table th {
            position: relative;
            cursor: pointer;
            user-select: none;
            padding-right: 20px; /* Espacio para el icono */
        }
        
        .estudios-table th:after {
            content: "⇕";
            position: absolute;
            right: 8px;
            color: #999;
            font-size: 0.8em;
        }
        
        .estudios-table th.sort-asc:after {
            content: "↑";
            color: #007bff;
        }
        
        .estudios-table th.sort-desc:after {
            content: "↓";
            color: #007bff;
        }
        
        /* Estilos para el contador de estudios */
        .estudios-contador {
            background-color: #f0f8ff;
            padding: 10px 15px;
            border-radius: 4px;
            margin-bottom: 15px;
            font-weight: bold;
            text-align: right;
            color: #0077b6;
            border-left: 4px solid #0077b6;
        }
        
        #contador-total, #contador-filtrados {
            display: inline-block;
        }
        
        /* Estilos para el efecto hover en las filas */
        .estudios-table tbody tr {
            transition: background-color 0.2s ease;
            cursor: pointer;
        }
        
        .estudios-table tbody tr:hover {
            background-color: #f0f7ff;
        }
        
        .estudios-table tbody tr.active {
            background-color: #e0f0ff;
        }
        
        /* Estilo para tarjetas de estudios en móvil */
        .estudio-card {
            transition: transform 0.2s ease, box-shadow 0.2s ease;
            cursor: pointer;
        }
        
        .estudio-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        
        @media (max-width: 768px) {
            .filter-container {
                flex-direction: column;
                align-items: stretch;
            }
            
            .date-filter-form {
                flex-direction: column;
            }
            
            .search-container {
                max-width: 100%;
            }
        }
        
        /* Estilos para el badge de PDF */
        .badge {
            display: inline-block;
            padding: 5px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
        }
        
        .badge-info {
            background-color: #17a2b8;
            color: white;
        }
        
        /* Estilos para el ícono de información */
        .info-icon {
            cursor: pointer;
            margin-left: 5px;
            opacity: 0.7;
            transition: opacity 0.2s;
        }
        
        .info-icon:hover {
            opacity: 1;
        }
        
        /* Estilos para el modal de series */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        
        .modal-content {
            background-color: white;
            padding: 20px;
            border-radius: 8px;
            width: 90%;
            max-width: 700px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
            border-bottom: 1px solid #eee;
            padding-bottom: 10px;
        }
        
        .modal-title {
            font-size: 1.2rem;
            font-weight: bold;
            color: #0077b6;
        }
        
        .modal-close {
            font-size: 20px;
            cursor: pointer;
            color: #666;
            background: none;
            border: none;
            padding: 5px;
        }
        
        .series-list {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        
        .series-item {
            margin-bottom: 15px;
            padding: 10px;
            border: 1px solid #eee;
            border-radius: 4px;
            transition: background-color 0.2s;
        }
        
        .series-item:hover {
            background-color: #f9f9f9;
        }
        
        .series-header {
            display: flex;
            justify-content: space-between;
            margin-bottom: 8px;
            font-weight: bold;
        }
        
        .series-number {
            color: #0077b6;
        }
        
        .series-modality {
            background-color: #e9ecef;
            padding: 2px 8px;
            border-radius: 4px;
            font-size: 12px;
        }
        
        .series-details {
            margin-top: 5px;
            color: #666;
        }
        
        .series-images {
            color: #28a745;
            font-weight: bold;
        }
        
        @media (max-width: 768px) {
            .modal-content {
                width: 95%;
                padding: 15px;
            }
            
            .series-header {
                flex-direction: column;
                gap: 5px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <img src="images/imac_logo.png" alt="IMAC Tucumán" class="header-logo">
            <div class="user-info">
                Bienvenido, <?php echo sanitizeInput($_SESSION['full_name']); ?>
                <a href="logout.php" class="logout-btn">Cerrar Sesión</a>
            </div>
        </div>

        <div class="filter-container">
            <form method="post" class="date-filter-form">
                <div class="form-group">
                    <label for="start_date">Fecha Inicio:</label>
                    <input type="date" id="start_date" name="start_date" value="<?php echo $startDate; ?>" required>
                </div>
                <div class="form-group">
                    <label for="end_date">Fecha Fin:</label>
                    <input type="date" id="end_date" name="end_date" value="<?php echo $endDate; ?>" required>
                </div>
                <div class="form-group">
                    <label for="patient_name">Nombre del Paciente:</label>
                    <input type="text" id="patient_name" name="patient_name" value="<?php echo htmlspecialchars($patientName); ?>">
                </div>
                <div class="form-group">
                    <label for="patient_id">ID del Paciente:</label>
                    <input type="text" id="patient_id" name="patient_id" value="<?php echo htmlspecialchars($patientId); ?>">
                </div>
                <button type="submit" name="filter" class="filter-btn">Buscar</button>
            </form>
            
            <div class="search-container">
                <input type="text" id="search-input" placeholder="Filtro rápido..." autocomplete="off">
            </div>
            
            <div class="modality-filters">
                <span class="modality-filters-title">Filtrar por Modalidad:</span>
                <button class="modality-button" data-modality="tomografía">CT</button>
                <button class="modality-button" data-modality="resonancia magnética">MR</button>
                <button class="modality-button" data-modality="radiología">CR</button>
                <button class="modality-button" data-modality="ecografía">US</button>
                <button class="modality-button" data-modality="ot">OT</button>
                <button class="modality-button" data-modality="all">TODAS</button>
                <button class="modality-button" data-modality="none">NINGUNA</button>
            </div>
        </div>

        <?php if (!empty($mensajeError)): ?>
        <div class="error-message">
            <?php echo $mensajeError; ?>
        </div>
        <?php endif; ?>

        <div class="estudios-container">
            <div id="no-results">No se encontraron estudios que coincidan con la búsqueda.</div>
            
            <?php if (empty($estudios)): ?>
                <p>No se encontraron estudios en el período seleccionado.</p>
            <?php else: ?>
                <!-- Contador de registros -->
                <div class="estudios-contador">
                    <span id="contador-total">Total: <?php echo count($estudios); ?> estudios</span>
                    <span id="contador-filtrados" style="display: none;"></span>
                </div>
                
                <!-- Vista móvil -->
                <div class="mobile-cards-container">
                    <?php foreach ($estudios as $estudio): 
                        $mainTags = $estudio['MainDicomTags'] ?? [];
                        $studyDate = $mainTags['StudyDate'] ?? '';
                        $studyDesc = $mainTags['StudyDescription'] ?? '';
                        $studyId = $estudio['ID'] ?? '';
                        $patientName = $estudio['PatientMainDicomTags']['PatientName'] ?? '';
                        $patientId = $estudio['PatientMainDicomTags']['PatientID'] ?? '';
                        
                        // Obtener la modalidad de la primera serie
                        $modality = getStudyMainModality($studyId);
                        
                        // Obtener información de series e imágenes
                        $seriesCount = $estudio['SeriesCount'] ?? 0;
                        $totalImageCount = $estudio['TotalImageCount'] ?? 0;
                        
                        // Determinar si es un informe
                        $isReport = false;
                        if ($modality === 'OT' && (stripos($studyDesc, 'PDF') !== false)) {
                            $isReport = true;
                        }
                        
                        // Obtener tipo de estudio para filtros
                        $tipoEstudio = getTipoEstudio($modality);
                    ?>
                    <div class="estudio-card<?php echo $isReport ? ' pdf-report' : ''; ?>" data-patient-name="<?php echo strtolower(sanitizeInput($patientName)); ?>" 
                         data-patient-id="<?php echo strtolower(sanitizeInput($patientId)); ?>"
                         data-study-desc="<?php echo strtolower(sanitizeInput($studyDesc)); ?>"
                         data-modality="<?php echo strtolower($modality); ?>"
                         data-study-date="<?php echo $studyDate; ?>"
                         data-study-type="<?php echo strtolower($tipoEstudio); ?>">
                        <div class="estudio-card-header">
                            <div class="estudio-card-title"><?php echo getTipoEstudio($modality); ?></div>
                            <div class="estudio-card-date"><?php echo formatStudyDate($studyDate); ?></div>
                        </div>
                        
                        <div class="estudio-card-details">
                            <div class="estudio-card-row">
                                <span class="estudio-card-label">Paciente:</span>
                                <span class="estudio-card-value"><?php echo sanitizeInput($patientName); ?></span>
                            </div>
                            <div class="estudio-card-row">
                                <span class="estudio-card-label">ID:</span>
                                <span class="estudio-card-value"><?php echo sanitizeInput($patientId); ?></span>
                            </div>
                            <div class="estudio-card-row">
                                <span class="estudio-card-label">Tipo:</span>
                                <span class="estudio-card-value"><?php echo getTipoEstudio($modality); ?></span>
                            </div>
                            <?php if (!empty($studyDesc)): ?>
                            <div class="estudio-card-row">
                                <span class="estudio-card-label">Descripción:</span>
                                <span class="estudio-card-value"><?php echo sanitizeInput($studyDesc); ?></span>
                            </div>
                            <?php else: ?>
                            <div class="estudio-card-row">
                                <span class="estudio-card-label">Descripción:</span>
                                <span class="estudio-card-value">N/A</span>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Información de series e imágenes -->
                            <?php if (!$isReport): ?>
                            <div class="estudio-card-row">
                                <span class="estudio-card-label">Series/Imgs:</span>
                                <span class="estudio-card-value">
                                    <?php echo $seriesCount; ?> series / <?php echo $totalImageCount; ?> imágenes
                                    <i class="info-icon" data-study-id="<?php echo $studyId; ?>">ℹ️</i>
                                </span>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="estudio-card-actions">
                            <?php if ($isReport): ?>
                            <a href="view-report.php?id=<?php echo $studyId; ?>" class="action-btn view-btn" target="_blank">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
                                    <circle cx="12" cy="12" r="3"></circle>
                                </svg>
                                <span class="btn-text">VISUALIZAR</span>
                            </a>
                            <a href="download-report.php?id=<?php echo $studyId; ?>&name=<?php echo urlencode($patientName); ?>&date=<?php echo $studyDate; ?>" class="action-btn download-btn">
                                <span class="btn-icon">⬇️</span>
                                <span class="btn-text">DESCARGA</span>
                            </a>
                            <?php else: ?>
                            <a href="<?php echo ORTHANC_PUBLIC_HOST; ?>/osimis-viewer/app/index.html?study=<?php echo $studyId; ?>" class="action-btn view-btn" target="_blank">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
                                    <circle cx="12" cy="12" r="3"></circle>
                                </svg>
                                <span class="btn-text">VISUALIZAR</span>
                            </a>
                            <a href="<?php echo ORTHANC_PUBLIC_HOST; ?>/studies/<?php echo $studyId; ?>/archive?filename=estudio-<?php echo date('d-m-Y', strtotime($studyDate)); ?>_<?php echo str_replace(' ', '', sanitizeInput($patientName)); ?>.zip" class="action-btn download-btn" target="_blank">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                                    <polyline points="7 10 12 15 17 10"></polyline>
                                    <line x1="12" y1="15" x2="12" y2="3"></line>
                                </svg>
                                <span class="btn-text">DESCARGA</span>
                            </a>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>

                <!-- Vista desktop -->
                <div class="table-responsive desktop-view">
                    <table class="estudios-table">
                        <thead>
                            <tr>
                                <th data-sort="date">Fecha</th>
                                <th data-sort="patient">Paciente</th>
                                <th data-sort="id">ID</th>
                                <th data-sort="type">Tipo de Estudio</th>
                                <th data-sort="desc">Descripción</th>
                                <th data-sort="count">Series/Imgs</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($estudios as $estudio): 
                                $mainTags = $estudio['MainDicomTags'] ?? [];
                                $studyDate = $mainTags['StudyDate'] ?? '';
                                $studyDesc = $mainTags['StudyDescription'] ?? '';
                                $studyId = $estudio['ID'] ?? '';
                                $patientName = $estudio['PatientMainDicomTags']['PatientName'] ?? '';
                                $patientId = $estudio['PatientMainDicomTags']['PatientID'] ?? '';
                                
                                // Obtener la modalidad de la primera serie
                                $modality = getStudyMainModality($studyId);
                                
                                // Obtener información de series e imágenes
                                $seriesCount = $estudio['SeriesCount'] ?? 0;
                                $totalImageCount = $estudio['TotalImageCount'] ?? 0;
                                
                                // Determinar si es un informe
                                $isReport = false;
                                if ($modality === 'OT' && (stripos($studyDesc, 'PDF') !== false)) {
                                    $isReport = true;
                                }
                                
                                // Obtener tipo de estudio para filtros
                                $tipoEstudio = getTipoEstudio($modality);
                            ?>
                            <tr data-patient-name="<?php echo strtolower(sanitizeInput($patientName)); ?>" 
                                data-patient-id="<?php echo strtolower(sanitizeInput($patientId)); ?>"
                                data-study-desc="<?php echo strtolower(sanitizeInput($studyDesc)); ?>"
                                data-modality="<?php echo strtolower($modality); ?>"
                                data-study-date="<?php echo $studyDate; ?>"
                                data-study-type="<?php echo strtolower($tipoEstudio); ?>"
                                class="<?php echo $isReport ? 'pdf-report' : ''; ?>">
                                <td><?php echo formatStudyDate($studyDate); ?></td>
                                <td><?php echo sanitizeInput($patientName); ?></td>
                                <td><?php echo sanitizeInput($patientId); ?></td>
                                <td><?php echo getTipoEstudio($modality); ?></td>
                                <td><?php echo !empty($studyDesc) ? sanitizeInput($studyDesc) : 'N/A'; ?></td>
                                <td>
                                    <?php if ($isReport): ?>
                                        <span class="badge badge-info">PDF</span>
                                    <?php else: ?>
                                        <span title="Series: <?php echo $seriesCount; ?>, Imágenes: <?php echo $totalImageCount; ?>">
                                            <?php echo $seriesCount; ?> / <?php echo $totalImageCount; ?>
                                            <i class="info-icon" data-study-id="<?php echo $studyId; ?>">ℹ️</i>
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($isReport): ?>
                                    <a href="view-report.php?id=<?php echo $studyId; ?>" class="action-btn view-btn" target="_blank">
                                        <span class="btn-icon">👁️</span>
                                        <span class="btn-text">VISUALIZAR</span>
                                    </a>
                                    <a href="download-report.php?id=<?php echo $studyId; ?>&name=<?php echo urlencode($patientName); ?>&date=<?php echo $studyDate; ?>" class="action-btn download-btn">
                                        <span class="btn-icon">⬇️</span>
                                        <span class="btn-text">DESCARGA</span>
                                    </a>
                                    <?php else: ?>
                                    <a href="<?php echo ORTHANC_PUBLIC_HOST; ?>/osimis-viewer/app/index.html?study=<?php echo $studyId; ?>" class="action-btn view-btn" target="_blank">
                                        <span class="btn-icon">👁️</span>
                                        <span class="btn-text">VISUALIZAR</span>
                                    </a>
                                    <a href="<?php echo ORTHANC_PUBLIC_HOST; ?>/studies/<?php echo $studyId; ?>/archive?filename=estudio-<?php echo date('d-m-Y', strtotime($studyDate)); ?>_<?php echo str_replace(' ', '', sanitizeInput($patientName)); ?>.zip" class="action-btn download-btn" target="_blank">
                                        <span class="btn-icon">⬇️</span>
                                        <span class="btn-text">DESCARGA</span>
                                    </a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Modal para detalles de series -->
    <div id="seriesModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title">Detalles de Series</div>
                <button class="modal-close">&times;</button>
            </div>
            <div id="seriesInfo">
                <div id="studyInfo"></div>
                <ul class="series-list" id="seriesList"></ul>
            </div>
        </div>
    </div>
    
    <script>
        // Definir la variable ORTHANC_PUBLIC_HOST para JavaScript
        const ORTHANC_PUBLIC_HOST = '<?php echo ORTHANC_PUBLIC_HOST; ?>';
        // Obtener una referencia a los datos de estudios en JavaScript
        const estudiosData = <?php echo json_encode($estudios); ?>;
        
        // JavaScript para filtrar estudios localmente
        document.addEventListener('DOMContentLoaded', function() {
            const searchInput = document.getElementById('search-input');
            const estudiosCards = document.querySelectorAll('.estudio-card');
            const estudiosRows = document.querySelectorAll('.estudios-table tbody tr');
            const noResultsMessage = document.getElementById('no-results');
            const tableHeaders = document.querySelectorAll('.estudios-table th[data-sort]');
            const modalityButtons = document.querySelectorAll('.modality-button');
            const allModalitiesButton = document.querySelector('.modality-button[data-modality="all"]');
            const seriesModal = document.getElementById('seriesModal');
            const modalClose = document.querySelector('.modal-close');
            const seriesList = document.getElementById('seriesList');
            const studyInfo = document.getElementById('studyInfo');
            
            // Cerrar el modal cuando se hace clic en el botón de cerrar
            modalClose.addEventListener('click', function() {
                seriesModal.style.display = 'none';
            });
            
            // Cerrar el modal cuando se hace clic fuera de su contenido
            window.addEventListener('click', function(event) {
                if (event.target === seriesModal) {
                    seriesModal.style.display = 'none';
                }
            });
            
            // Manejador para íconos de información
            document.addEventListener('click', function(event) {
                if (event.target.classList.contains('info-icon')) {
                    const studyId = event.target.dataset.studyId;
                    if (studyId) {
                        showSeriesDetails(studyId);
                    }
                }
            });
            
            // Función para mostrar detalles de series en el modal
            function showSeriesDetails(studyId) {
                console.log('Mostrando detalles para estudio:', studyId); // Debug log
                
                // Buscar el estudio por ID
                const estudio = estudiosData.find(e => e.ID === studyId);
                
                if (estudio) {
                    console.log('Estudio encontrado:', estudio); // Debug log
                    
                    // Limpiar contenido previo
                    seriesList.innerHTML = '';
                    
                    // Mostrar información del estudio
                    const mainTags = estudio.MainDicomTags || {};
                    const patientTags = estudio.PatientMainDicomTags || {};
                    const studyDate = formatStudyDate(mainTags.StudyDate || '');
                    const studyDesc = mainTags.StudyDescription || 'Sin descripción';
                    const patientName = patientTags.PatientName || 'Sin nombre';
                    const patientId = patientTags.PatientID || 'Sin ID';
                    
                    // Obtener estadísticas del estudio
                    const stats = estudio.Statistics || {};
                    const totalSeries = stats.CountSeries || 0;
                    const totalInstances = stats.CountInstances || 0;
                    const diskSize = stats.DiskSizeMB ? (stats.DiskSizeMB / 1024).toFixed(2) : 0;
                    
                    studyInfo.innerHTML = `
                        <div style="margin-bottom: 15px; padding: 10px; background-color: #f0f8ff; border-radius: 4px;">
                            <h3 style="margin-top: 0;">${studyDesc}</h3>
                            <p><strong>Paciente:</strong> ${patientName} (${patientId})</p>
                            <p><strong>Fecha:</strong> ${studyDate}</p>
                            <div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 10px; margin-top: 10px;">
                                <div style="background-color: #e3f2fd; padding: 8px; border-radius: 4px;">
                                    <strong>Total Series:</strong> ${totalSeries}
                                </div>
                                <div style="background-color: #e3f2fd; padding: 8px; border-radius: 4px;">
                                    <strong>Total Imágenes:</strong> ${totalInstances}
                                </div>
                                <div style="background-color: #e3f2fd; padding: 8px; border-radius: 4px;">
                                    <strong>Tamaño en Disco:</strong> ${diskSize} GB
                                </div>
                            </div>
                        </div>
                    `;
                    
                    // Si hay series disponibles, mostrarlas
                    if (estudio.SeriesDetails && estudio.SeriesDetails.length > 0) {
                        // Ordenar series por número
                        const sortedSeries = [...estudio.SeriesDetails].sort((a, b) => {
                            const numA = parseInt(a.Number) || 0;
                            const numB = parseInt(b.Number) || 0;
                            return numA - numB;
                        });
                        
                        // Mostrar detalles de cada serie
                        sortedSeries.forEach(serie => {
                            const serieItem = document.createElement('li');
                            serieItem.className = 'series-item';
                            
                            serieItem.innerHTML = `
                                <div class="series-header">
                                    <div class="series-number">Serie #${serie.Number || 'N/A'}</div>
                                    <div class="series-modality">${serie.Modality || 'N/A'}</div>
                                </div>
                                <div class="series-description">${serie.Description || 'Sin descripción'}</div>
                                <div class="series-details">
                                    <span class="series-images">${serie.ImageCount} imágenes</span>
                                </div>
                            `;
                            
                            seriesList.appendChild(serieItem);
                        });
                    } else {
                        // Si no hay series, mostrar un mensaje
                        seriesList.innerHTML = '<li class="series-item">No hay series disponibles para este estudio.</li>';
                    }
                    
                    // Mostrar el modal
                    seriesModal.style.display = 'flex';
                } else {
                    console.error('No se encontró el estudio con ID:', studyId); // Debug log
                }
            }
            
            // Función para formatear fechas DICOM (YYYYMMDD)
            function formatStudyDate(fechaDicom) {
                if (!fechaDicom || fechaDicom.length !== 8) return fechaDicom;
                
                const año = fechaDicom.substring(0, 4);
                const mes = fechaDicom.substring(4, 6);
                const dia = fechaDicom.substring(6, 8);
                
                return `${dia}/${mes}/${año}`;
            }
            
            // Estado de ordenamiento
            let currentSort = {
                column: 'id', // Ordenar inicialmente por ID de paciente
                direction: 'asc'
            };
            
            // Estado de filtros de modalidad
            let activeModalities = [];
            let showAllModalities = true; // Cuando es true, se muestran todos los resultados sin filtrar por modalidad
            
            // Aplicar ordenación inicial (por ID de paciente y tipo de estudio)
            sortTable('id');
            
            // Función para manejar el doble clic en una fila de la tabla
            function handleRowDoubleClick(row) {
                // Buscar el botón de visualizar en esta fila
                const viewButton = row.querySelector('.action-btn.view-btn');
                if (viewButton) {
                    // Obtener la URL del botón
                    const url = viewButton.getAttribute('href');
                    // Abrir la URL en una nueva pestaña
                    window.open(url, '_blank');
                }
            }
            
            // Asignar event listeners para doble clic a filas de la tabla
            estudiosRows.forEach(row => {
                row.addEventListener('dblclick', function() {
                    handleRowDoubleClick(this);
                });
                
                // Resaltar fila al hacer clic
                row.addEventListener('click', function() {
                    // Eliminar clase 'active' de todas las filas
                    estudiosRows.forEach(r => r.classList.remove('active'));
                    // Añadir clase 'active' a la fila actual
                    this.classList.add('active');
                });
            });
            
            // Asignar event listeners para doble clic a tarjetas (vista móvil)
            estudiosCards.forEach(card => {
                card.addEventListener('dblclick', function() {
                    // Buscar el botón de visualizar en esta tarjeta
                    const viewButton = this.querySelector('.action-btn.view-btn');
                    if (viewButton) {
                        // Obtener la URL del botón
                        const url = viewButton.getAttribute('href');
                        // Abrir la URL en una nueva pestaña
                        window.open(url, '_blank');
                    }
                });
            });
            
            // Toggle para descripciones largas
            document.querySelectorAll('.description-toggle').forEach(toggle => {
                toggle.addEventListener('click', function() {
                    const targetId = this.dataset.target;
                    const description = document.getElementById(targetId);
                    
                    if (description.style.display === 'none' || !description.style.display) {
                        description.style.display = 'block';
                        this.textContent = 'Ver menos';
                    } else {
                        description.style.display = 'none';
                        this.textContent = 'Ver más';
                    }
                });
            });
            
            // Manejar filtros de modalidad
            modalityButtons.forEach(button => {
                button.addEventListener('click', function() {
                    const modality = this.dataset.modality;
                    const isActive = this.classList.contains('active');
                    
                    // Desactivar showAllModalities cuando se presiona cualquier botón
                    showAllModalities = false;
                    
                    // Si es "TODAS"
                    if (modality === 'all') {
                        // Quitar activo de "NINGUNA" si está activo
                        const noneButton = document.querySelector('.modality-button[data-modality="none"]');
                        if (noneButton.classList.contains('active')) {
                            noneButton.classList.remove('active');
                        }
                        
                        // Activar todas las modalidades excepto "NINGUNA"
                        modalityButtons.forEach(btn => {
                            if (btn.dataset.modality !== 'none') {
                                btn.classList.add('active');
                            }
                        });
                        
                        activeModalities = ['tomografía', 'resonancia magnética', 'radiología', 'ecografía', 'ot', 'all'];
                    }
                    // Si es "NINGUNA"
                    else if (modality === 'none') {
                        // Desactivar todos los demás botones
                        modalityButtons.forEach(btn => {
                            if (btn.dataset.modality !== 'none') {
                                btn.classList.remove('active');
                            }
                        });
                        
                        // Activar solo este botón
                        this.classList.add('active');
                        
                        // Limpiar modalidades activas
                        activeModalities = ['none'];
                    }
                    // Cualquier otra modalidad
                    else {
                        // Quitar "NINGUNA" si está activo
                        const noneButton = document.querySelector('.modality-button[data-modality="none"]');
                        if (noneButton.classList.contains('active')) {
                            noneButton.classList.remove('active');
                            activeModalities = activeModalities.filter(m => m !== 'none');
                        }
                        
                        // Toggle estado actual
                        this.classList.toggle('active');
                        
                        // Actualizar array de modalidades activas
                        if (isActive) {
                            activeModalities = activeModalities.filter(m => m !== modality);
                            // Desactivar "TODAS" si se desactiva cualquier modalidad
                            if (allModalitiesButton.classList.contains('active')) {
                                allModalitiesButton.classList.remove('active');
                                activeModalities = activeModalities.filter(m => m !== 'all');
                            }
                        } else {
                            activeModalities.push(modality);
                            // Verificar si todas las modalidades individuales están activas
                            const allIndividualActive = ['tomografía', 'resonancia magnética', 'radiología', 'ecografía', 'ot'].every(m => 
                                activeModalities.includes(m)
                            );
                            if (allIndividualActive && !activeModalities.includes('all')) {
                                allModalitiesButton.classList.add('active');
                                activeModalities.push('all');
                            }
                        }
                    }
                    
                    // Aplicar filtros
                    filterEstudios();
                });
            });
            
            // Función para filtrar estudios
            function filterEstudios() {
                const searchValue = searchInput.value.toLowerCase();
                let hasVisibleElements = false;
                let visibleCount = 0;
                
                // Filtrar tarjetas (vista móvil)
                estudiosCards.forEach(card => {
                    const patientName = card.dataset.patientName || '';
                    const patientId = card.dataset.patientId || '';
                    const studyDesc = card.dataset.studyDesc || '';
                    const modality = (card.dataset.modality || '').toLowerCase();
                    const studyDate = card.dataset.studyDate || '';
                    const studyType = (card.dataset.studyType || '').toLowerCase();
                    
                    // Verificar filtro de texto
                    const matchesSearch = patientName.includes(searchValue) ||
                                     patientId.includes(searchValue) ||
                                     studyDesc.includes(searchValue) ||
                                     modality.includes(searchValue) ||
                                     studyDate.includes(searchValue);
                    
                    // Verificar filtro de modalidad
                    const matchesModality = 
                        showAllModalities || // Mostrar todo si no hay filtro activo
                        activeModalities.includes('none') || // Si "NINGUNA" está activo, mostrar todos
                        activeModalities.includes('all') || 
                        (activeModalities.length > 0 && activeModalities.some(m => {
                            if (m === 'ot') {
                                return studyType.includes('informe de estudio');
                            }
                            return studyType.includes(m);
                        }));
                    
                    const isVisible = matchesSearch && matchesModality;
                    
                    if (isVisible) {
                        hasVisibleElements = true;
                        card.classList.remove('hide');
                        visibleCount++;
                    } else {
                        card.classList.add('hide');
                    }
                });
                
                // Filtrar filas (vista desktop)
                estudiosRows.forEach(row => {
                    const patientName = row.dataset.patientName || '';
                    const patientId = row.dataset.patientId || '';
                    const studyDesc = row.dataset.studyDesc || '';
                    const modality = (row.dataset.modality || '').toLowerCase();
                    const studyDate = row.dataset.studyDate || '';
                    const studyType = (row.dataset.studyType || '').toLowerCase();
                    
                    // Verificar filtro de texto
                    const matchesSearch = patientName.includes(searchValue) ||
                                     patientId.includes(searchValue) ||
                                     studyDesc.includes(searchValue) ||
                                     modality.includes(searchValue) ||
                                     studyDate.includes(searchValue);
                    
                    // Verificar filtro de modalidad
                    const matchesModality = 
                        showAllModalities || // Mostrar todo si no hay filtro activo
                        activeModalities.includes('none') || // Si "NINGUNA" está activo, mostrar todos
                        activeModalities.includes('all') || 
                        (activeModalities.length > 0 && activeModalities.some(m => {
                            if (m === 'ot') {
                                return studyType.includes('informe de estudio');
                            }
                            return studyType.includes(m);
                        }));
                    
                    const isVisible = matchesSearch && matchesModality;
                    
                    if (isVisible) {
                        hasVisibleElements = true;
                        row.classList.remove('hide');
                    } else {
                        row.classList.add('hide');
                    }
                });
                
                // Actualizar contador de estudios filtrados
                const contadorTotal = document.getElementById('contador-total');
                const contadorFiltrados = document.getElementById('contador-filtrados');
                const totalEstudios = estudiosRows.length;
                
                if (visibleCount < totalEstudios) {
                    contadorTotal.style.display = 'none';
                    contadorFiltrados.textContent = `Mostrando ${visibleCount} de ${totalEstudios} estudios`;
                    contadorFiltrados.style.display = 'inline';
                } else {
                    contadorTotal.style.display = 'inline';
                    contadorFiltrados.style.display = 'none';
                }
                
                // Mostrar mensaje si no hay resultados
                if (!hasVisibleElements && (searchValue || activeModalities.length < 6)) {
                    noResultsMessage.style.display = 'block';
                } else {
                    noResultsMessage.style.display = 'none';
                }
            }
            
            // Función para ordenar la tabla
            function sortTable(column) {
                // Si hacemos clic en la misma columna, invertimos la dirección
                if (currentSort.column === column) {
                    currentSort.direction = currentSort.direction === 'asc' ? 'desc' : 'asc';
                } else {
                    currentSort.column = column;
                    currentSort.direction = 'asc';
                }
                
                // Actualizar clases de los encabezados
                tableHeaders.forEach(header => {
                    header.classList.remove('sort-asc', 'sort-desc');
                    if (header.dataset.sort === column) {
                        header.classList.add(currentSort.direction === 'asc' ? 'sort-asc' : 'sort-desc');
                    }
                });
                
                // Ordenar filas de la tabla (vista desktop)
                const tbody = document.querySelector('.estudios-table tbody');
                const rows = Array.from(tbody.querySelectorAll('tr:not(.hide)'));
                
                // Ordenar filas
                const sortedRows = rows.sort((a, b) => {
                    return compareElements(a, b, column);
                });
                
                // Volver a agregar las filas ordenadas
                sortedRows.forEach(row => {
                    tbody.appendChild(row);
                });

                // Ordenar tarjetas (vista móvil)
                const cardsContainer = document.querySelector('.mobile-cards-container');
                const cards = Array.from(cardsContainer.querySelectorAll('.estudio-card:not(.hide)'));
                
                // Ordenar tarjetas usando la misma lógica
                const sortedCards = cards.sort((a, b) => {
                    return compareElements(a, b, column);
                });
                
                // Volver a agregar las tarjetas ordenadas
                sortedCards.forEach(card => {
                    cardsContainer.appendChild(card);
                });
            }

            // Función de comparación común para filas y tarjetas
            function compareElements(a, b, column) {
                // Si estamos ordenando por paciente o ID, incluimos la lógica de estudios antes que informes
                if (column === 'patient' || column === 'id') {
                    const patientIdA = a.dataset.patientId || '';
                    const patientIdB = b.dataset.patientId || '';
                    
                    // Si los IDs de paciente son diferentes, ordenar por ID
                    if (patientIdA !== patientIdB) {
                        if (patientIdA < patientIdB) {
                            return currentSort.direction === 'asc' ? -1 : 1;
                        }
                        if (patientIdA > patientIdB) {
                            return currentSort.direction === 'asc' ? 1 : -1;
                        }
                    }
                    
                    // Si los IDs son iguales, ordenar por tipo (estudio antes que informe)
                    const studyDescA = a.dataset.studyDesc || '';
                    const studyDescB = b.dataset.studyDesc || '';
                    const modalityA = (a.dataset.modality || '').toLowerCase();
                    const modalityB = (b.dataset.modality || '').toLowerCase();
                    
                    const isReportA = studyDescA.includes('informe') || modalityA === 'ot';
                    const isReportB = studyDescB.includes('informe') || modalityB === 'ot';
                    
                    if (isReportA !== isReportB) {
                        return isReportA ? 1 : -1; // Estudios primero, informes después
                    }
                    
                    // Si ambos son el mismo tipo, ordenar por fecha (más reciente primero)
                    const dateA = a.dataset.studyDate || '';
                    const dateB = b.dataset.studyDate || '';
                    
                    if (dateA !== dateB) {
                        return dateA > dateB ? -1 : 1;
                    }
                }
                
                    let valueA, valueB;
                    
                    // Determinar qué valores comparar según la columna
                    switch(column) {
                        case 'date':
                            valueA = a.dataset.studyDate || '';
                            valueB = b.dataset.studyDate || '';
                            break;
                        case 'patient':
                            valueA = a.dataset.patientName || '';
                            valueB = b.dataset.patientName || '';
                            break;
                        case 'id':
                            valueA = a.dataset.patientId || '';
                            valueB = b.dataset.patientId || '';
                            break;
                        case 'type':
                            valueA = a.dataset.modality || '';
                            valueB = b.dataset.modality || '';
                            break;
                        case 'desc':
                            valueA = a.dataset.studyDesc || '';
                            valueB = b.dataset.studyDesc || '';
                            break;
                        default:
                            valueA = '';
                            valueB = '';
                    }
                    
                    // Comparación
                    if (valueA < valueB) {
                        return currentSort.direction === 'asc' ? -1 : 1;
                    }
                    if (valueA > valueB) {
                        return currentSort.direction === 'asc' ? 1 : -1;
                    }
                    return 0;
            }
            
            // Agregar event listeners para ordenamiento
            tableHeaders.forEach(header => {
                header.addEventListener('click', function() {
                    const column = this.dataset.sort;
                    sortTable(column);
                });
            });
            
            // Escuchar eventos de entrada en el campo de búsqueda
            searchInput.addEventListener('input', filterEstudios);
        });
    </script>
</body>
</html>
