<?php
session_start();
require_once 'config/config.php';
require_once 'includes/functions.php';

// Verificar si el usuario está logueado y es administrador
$isAdmin = false;
if (isset($_SESSION['user_id'])) {
    $conn = new mysqli(PORTAL_DB_HOST, PORTAL_DB_USERNAME, PORTAL_DB_PASSWORD, PORTAL_DB_NAME);
    if (!$conn->connect_error) {
        $stmt = $conn->prepare("SELECT role FROM users WHERE id = ? AND is_active = 1");
        $stmt->bind_param("i", $_SESSION['user_id']);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows === 1) {
            $user = $result->fetch_assoc();
            $isAdmin = ($user['role'] === 'admin');
        }
        $stmt->close();
    }
    $conn->close();
}

// Eliminar la redirección automática que causa el bucle infinito
// if ($isAdmin && !isset($_GET['edit']) && !isset($_POST['register']) && !isset($_POST['update_user'])) {
//     header("Location: register.php?admin=1");
//     exit;
// }

$error = '';
$success = '';
$users = []; // Array para almacenar la lista de usuarios

// Función para obtener todos los usuarios
function getAllUsers($conn) {
    $users = [];
    $stmt = $conn->prepare("SELECT id, username, full_name, email, role, last_login, is_active, created_at FROM users ORDER BY created_at DESC");
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $users[] = $row;
    }
    $stmt->close();
    return $users;
}

// Procesamiento de la eliminación de usuario
if ($isAdmin && isset($_POST['delete_user']) && isset($_POST['user_id'])) {
    $userId = (int)$_POST['user_id'];
    // No permitir eliminar al propio usuario administrador conectado
    if ($userId == $_SESSION['user_id']) {
        $error = "No puedes eliminar tu propio usuario administrador.";
    } else {
        $conn = new mysqli(PORTAL_DB_HOST, PORTAL_DB_USERNAME, PORTAL_DB_PASSWORD, PORTAL_DB_NAME);
        if (!$conn->connect_error) {
            $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
            $stmt->bind_param("i", $userId);
            if ($stmt->execute()) {
                $success = "Usuario eliminado correctamente.";
            } else {
                $error = "Error al eliminar el usuario: " . $conn->error;
            }
            $stmt->close();
            $conn->close();
        } else {
            $error = "Error de conexión a la base de datos: " . $conn->connect_error;
        }
    }
}

// Procesamiento de la activación/desactivación de usuario
if ($isAdmin && isset($_POST['toggle_active']) && isset($_POST['user_id'])) {
    $userId = (int)$_POST['user_id'];
    $isActive = (int)$_POST['is_active'];
    $newState = $isActive ? 0 : 1;
    
    // No permitir desactivar al propio usuario administrador conectado
    if ($userId == $_SESSION['user_id'] && $newState == 0) {
        $error = "No puedes desactivar tu propio usuario administrador.";
    } else {
        $conn = new mysqli(PORTAL_DB_HOST, PORTAL_DB_USERNAME, PORTAL_DB_PASSWORD, PORTAL_DB_NAME);
        if (!$conn->connect_error) {
            $stmt = $conn->prepare("UPDATE users SET is_active = ? WHERE id = ?");
            $stmt->bind_param("ii", $newState, $userId);
            if ($stmt->execute()) {
                $success = "Estado del usuario actualizado correctamente.";
            } else {
                $error = "Error al actualizar el estado del usuario: " . $conn->error;
            }
            $stmt->close();
            $conn->close();
        } else {
            $error = "Error de conexión a la base de datos: " . $conn->connect_error;
        }
    }
}

// Procesamiento del registro o actualización de usuario
if ($_SERVER['REQUEST_METHOD'] === 'POST' && (isset($_POST['register']) || isset($_POST['update_user']))) {
    $username = sanitizeInput($_POST['username']);
    $fullName = sanitizeInput($_POST['full_name']);
    $email = sanitizeInput($_POST['email']);
    $role = sanitizeInput($_POST['role']);
    $isActiveInput = isset($_POST['is_active']) ? 1 : 0;
    
    // Campos adicionales para actualización
    $userId = isset($_POST['user_id']) ? (int)$_POST['user_id'] : 0;
    $updatePassword = isset($_POST['update_password']) && $_POST['update_password'] == 1;

    // Validaciones comunes
    if (empty($username) || empty($fullName)) {
        $error = "Todos los campos marcados con * son obligatorios.";
    } elseif (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = "El email no es válido.";
    } else {
        // Conectar a la base de datos
        $conn = new mysqli(PORTAL_DB_HOST, PORTAL_DB_USERNAME, PORTAL_DB_PASSWORD, PORTAL_DB_NAME);
        
        if ($conn->connect_error) {
            $error = "Error de conexión a la base de datos: " . $conn->connect_error;
        } else {
            // Verificación de permisos para ciertas operaciones
            if ((isset($_POST['update_user']) || $role === 'admin') && !$isAdmin) {
                $error = "No tienes permisos para realizar esta operación.";
            } else {
                // Procesamiento de registro de nuevo usuario
                if (isset($_POST['register'])) {
                    $password = $_POST['password'];
                    $confirmPassword = $_POST['confirm_password'];
                    
                    // Validaciones específicas para registro
                    if (empty($password) || empty($confirmPassword)) {
                        $error = "La contraseña es obligatoria para nuevos usuarios.";
                    } elseif ($password !== $confirmPassword) {
                        $error = "Las contraseñas no coinciden.";
                    } elseif (strlen($password) < 6) {
                        $error = "La contraseña debe tener al menos 6 caracteres.";
        } else {
            // Verificar si el usuario ya existe
            $stmt = $conn->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
            $stmt->bind_param("ss", $username, $email);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $error = "El usuario o email ya existe.";
            } else {
                // Hash de la contraseña
                $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                
                // Insertar nuevo usuario
                            $stmt = $conn->prepare("INSERT INTO users (username, password, full_name, email, role, is_active) VALUES (?, ?, ?, ?, ?, ?)");
                            $stmt->bind_param("sssssi", $username, $hashedPassword, $fullName, $email, $role, $isActiveInput);
                
                if ($stmt->execute()) {
                    $success = "Usuario registrado exitosamente.";
                } else {
                    $error = "Error al registrar el usuario: " . $conn->error;
                }
                        }
                        $stmt->close();
                    }
                }
                // Procesamiento de actualización de usuario existente
                elseif (isset($_POST['update_user']) && $isAdmin) {
                    // Verificar si existe otro usuario con el mismo nombre o email
                    $stmt = $conn->prepare("SELECT id FROM users WHERE (username = ? OR email = ?) AND id != ?");
                    $stmt->bind_param("ssi", $username, $email, $userId);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    
                    if ($result->num_rows > 0) {
                        $error = "El usuario o email ya existe para otro usuario.";
                    } else {
                        // Si se va a actualizar la contraseña
                        if ($updatePassword) {
                            $password = $_POST['password'];
                            $confirmPassword = $_POST['confirm_password'];
                            
                            if (empty($password) || empty($confirmPassword)) {
                                $error = "La contraseña es obligatoria para la actualización.";
                            } elseif ($password !== $confirmPassword) {
                                $error = "Las contraseñas no coinciden.";
                            } elseif (strlen($password) < 6) {
                                $error = "La contraseña debe tener al menos 6 caracteres.";
                            } else {
                                $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                                $stmt = $conn->prepare("UPDATE users SET username = ?, password = ?, full_name = ?, email = ?, role = ?, is_active = ? WHERE id = ?");
                                $stmt->bind_param("sssssii", $username, $hashedPassword, $fullName, $email, $role, $isActiveInput, $userId);
                            }
                        } else {
                            // Actualizar sin cambiar la contraseña
                            $stmt = $conn->prepare("UPDATE users SET username = ?, full_name = ?, email = ?, role = ?, is_active = ? WHERE id = ?");
                            $stmt->bind_param("ssssii", $username, $fullName, $email, $role, $isActiveInput, $userId);
                        }
                        
                        // No permitir cambiar el rol o estado del propio usuario administrador
                        if ($userId == $_SESSION['user_id'] && ($role !== 'admin' || $isActiveInput === 0)) {
                            $error = "No puedes cambiar tu propio rol o desactivar tu cuenta de administrador.";
                        } else if (!isset($error)) {
                            if ($stmt->execute()) {
                                $success = "Usuario actualizado exitosamente.";
                            } else {
                                $error = "Error al actualizar el usuario: " . $conn->error;
                            }
                        }
                    }
                    $stmt->close();
                }
            }
            
            // Si es administrador, obtener la lista de usuarios
            if ($isAdmin) {
                $users = getAllUsers($conn);
            }
            
            $conn->close();
        }
    }
} elseif ($isAdmin) {
    // Si es administrador, obtener la lista de usuarios al cargar la página
    $conn = new mysqli(PORTAL_DB_HOST, PORTAL_DB_USERNAME, PORTAL_DB_PASSWORD, PORTAL_DB_NAME);
    if (!$conn->connect_error) {
        $users = getAllUsers($conn);
        $conn->close();
    }
}

// Obtener datos de un usuario para edición
$editUser = null;
if ($isAdmin && isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    $userId = (int)$_GET['edit'];
    $conn = new mysqli(PORTAL_DB_HOST, PORTAL_DB_USERNAME, PORTAL_DB_PASSWORD, PORTAL_DB_NAME);
    if (!$conn->connect_error) {
        $stmt = $conn->prepare("SELECT id, username, full_name, email, role, is_active FROM users WHERE id = ?");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows === 1) {
            $editUser = $result->fetch_assoc();
        }
        $stmt->close();
        $conn->close();
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $isAdmin ? 'Administración de Usuarios' : 'Registro de Usuario'; ?> - Portal de Imágenes</title>
    <link rel="stylesheet" href="css/styles.css">
    <style>
        .admin-container {
            max-width: 1200px;
            margin: 35px auto;
            padding: 30px 35px;
            background-color: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }
        
        .users-table {
            width: 100%;
            border-collapse: collapse;
            margin: 25px 0;
            font-size: 14px;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
        }
        
        .users-table th,
        .users-table td {
            padding: 14px 16px;
            border: 1px solid #eee;
            text-align: left;
            vertical-align: middle;
        }
        
        .users-table th {
            background-color: #f5f5f5;
            font-weight: 600;
            color: #333;
            position: sticky;
            top: 0;
        }
        
        .users-table tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        
        .users-table tr:hover {
            background-color: #f0f7ff;
            transition: background-color 0.2s ease;
        }
        
        .action-buttons {
            display: flex;
            gap: 8px;
            justify-content: flex-start;
            flex-wrap: wrap;
        }
        
        .button-small {
            padding: 6px 12px;
            font-size: 13px;
            border-radius: 4px;
            cursor: pointer;
            border: none;
            color: white;
            transition: background-color 0.2s ease;
            white-space: nowrap;
        }
        
        .edit-btn {
            background-color: #0077b6;
        }
        
        .edit-btn:hover {
            background-color: #005f92;
        }
        
        .delete-btn {
            background-color: #dc3545;
        }
        
        .delete-btn:hover {
            background-color: #bb2d3b;
        }
        
        .toggle-btn {
            background-color: #ffc107;
            color: #333;
        }
        
        .toggle-btn:hover {
            background-color: #e5ac06;
        }
        
        .tab-container {
            display: flex;
            border-bottom: 1px solid #ddd;
            margin-bottom: 30px;
            flex-wrap: wrap;
            padding-bottom: 2px;
        }
        
        .tab-button {
            padding: 12px 24px;
            background-color: #f5f5f5;
            border: 1px solid #ddd;
            border-bottom: none;
            cursor: pointer;
            border-radius: 6px 6px 0 0;
            margin-right: 8px;
            font-weight: 500;
            transition: all 0.2s ease;
        }
        
        .tab-button.active {
            background-color: #fff;
            border-bottom: 2px solid #0077b6;
            font-weight: 600;
            color: #0077b6;
        }
        
        .tab-button:hover {
            background-color: #fff;
        }
        
        .tab-content {
            background-color: #fff;
            padding: 30px 0;
            border-radius: 0 0 8px 8px;
        }
        
        .checkbox-container {
            display: flex;
            align-items: center;
            margin-bottom: 18px;
        }
        
        .checkbox-container input[type="checkbox"] {
            margin-right: 10px;
            transform: scale(1.2);
        }
        
        .status-active {
            color: #198754;
            font-weight: 600;
        }
        
        .status-inactive {
            color: #dc3545;
            font-weight: 600;
        }
        
        .search-box {
            width: 100%;
            padding: 12px 16px;
            margin-bottom: 25px;
            border: 1px solid #ddd;
            border-radius: 6px;
            box-sizing: border-box;
            font-size: 14px;
            transition: border-color 0.2s ease;
        }
        
        .search-box:focus {
            border-color: #0077b6;
            outline: none;
            box-shadow: 0 0 0 3px rgba(0, 119, 182, 0.1);
        }
        
        .nav-links {
            display: flex;
            gap: 25px;
            margin-bottom: 30px;
            align-items: center;
            flex-wrap: wrap;
        }
        
        .nav-link {
            text-decoration: none;
            color: #0077b6;
            font-weight: 600;
            transition: color 0.2s ease;
            display: flex;
            align-items: center;
        }
        
        .nav-link::before {
            content: "←";
            margin-right: 5px;
            font-size: 16px;
        }
        
        .nav-link:hover {
            color: #005f92;
        }
        
        .register-container {
            max-width: 550px;
            margin: 60px auto;
            padding: 35px 40px;
            background-color: #fff;
            border-radius: 8px;
            box-shadow: 0 3px 20px rgba(0, 0, 0, 0.1);
        }
        
        .register-header {
            text-align: center;
            margin-bottom: 35px;
        }
        
        .register-logo {
            max-width: 180px;
            margin-bottom: 25px;
        }
        
        .register-header h1 {
            font-size: 26px;
            color: #333;
            margin: 0;
        }
        
        .register-form {
            max-width: 600px;
            margin: 0 auto;
        }
        
        .form-group {
            margin-bottom: 24px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 10px;
            font-weight: 500;
            color: #444;
        }
        
        .form-group input,
        .form-group select {
            width: 100%;
            padding: 14px 16px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 15px;
            transition: all 0.2s ease;
            background-color: #f9f9f9;
        }
        
        .form-group input:focus,
        .form-group select:focus {
            border-color: #0077b6;
            outline: none;
            box-shadow: 0 0 0 3px rgba(0, 119, 182, 0.1);
            background-color: #fff;
        }
        
        .form-actions {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 35px;
            gap: 20px;
        }
        
        .form-actions .register-btn,
        .form-actions .back-btn,
        .admin-btn {
            padding: 14px 24px;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            text-align: center;
            transition: all 0.2s ease;
            font-size: 15px;
            display: inline-block;
            border: none;
            min-width: 130px;
        }
        
        .form-actions .register-btn {
            background-color: #0077b6;
            color: white;
            flex-grow: 1;
        }
        
        .form-actions .register-btn:hover {
            background-color: #005f92;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        
        .form-actions .back-btn {
            background-color: #f5f5f5;
            color: #333;
            border: 1px solid #ddd;
        }
        
        .form-actions .back-btn:hover {
            background-color: #e9e9e9;
            transform: translateY(-2px);
        }
        
        .admin-section {
            margin-top: 35px;
            text-align: center;
            border-top: 1px solid #ddd;
            padding-top: 25px;
        }
        
        .admin-section p {
            margin-bottom: 18px;
            font-weight: 500;
            color: #444;
        }
        
        .admin-btn {
            background-color: #28a745;
            color: white;
            padding: 14px 28px;
            font-weight: 600;
            margin: 0 auto;
            display: inline-block;
        }
        
        .admin-btn:hover {
            background-color: #218838;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        
        .login-btn {
            display: block;
            text-align: center;
            margin-top: 25px;
            margin-bottom: 5px;
            color: #0077b6;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.2s ease;
            padding: 5px;
        }
        
        .login-btn:hover {
            color: #005f92;
            text-decoration: underline;
        }
        
        /* Mejoras responsivas para el formulario */
        @media (max-width: 992px) {
            .admin-container {
                padding: 25px;
                margin: 25px auto;
            }
            
            .users-table {
                font-size: 13px;
            }
            
            .users-table th,
            .users-table td {
                padding: 10px 12px;
            }
            
            .button-small {
                padding: 5px 10px;
                font-size: 12px;
            }
        }
        
        @media (max-width: 768px) {
            .tab-button {
                padding: 10px 15px;
                font-size: 14px;
                margin-right: 5px;
            }
            
            .users-table {
                display: block;
                overflow-x: auto;
                white-space: nowrap;
            }
        }
        
        @media (max-width: 576px) {
            .register-container {
                padding: 25px;
                margin: 20px auto;
                max-width: 95%;
            }
            
            .form-group {
                margin-bottom: 20px;
            }
            
            .form-group label {
                margin-bottom: 8px;
            }
            
            .form-group input,
            .form-group select {
                padding: 12px 14px;
            }
            
            .form-actions {
                flex-direction: column;
            }
            
            .form-actions .register-btn,
            .form-actions .back-btn {
                width: 100%;
                margin-bottom: 10px;
            }
            
            .admin-btn {
                width: 100%;
            }
        }
        
        .error-message,
        .success-message {
            padding: 16px;
            margin-bottom: 25px;
            border-radius: 6px;
            font-weight: 500;
        }
        
        .error-message {
            background-color: #f8d7da;
            color: #842029;
            border: 1px solid #f5c2c7;
        }
        
        .success-message {
            background-color: #d1e7dd;
            color: #0f5132;
            border: 1px solid #badbcc;
        }
    </style>
</head>
<body>
    <div class="container <?php echo $isAdmin ? 'admin-container' : ''; ?>">
        <?php if ($isAdmin): ?>
        <div class="nav-links">
            <a href="studies.php" class="nav-link">← Volver a Estudios</a>
            <a href="logout.php" class="nav-link">Cerrar Sesión</a>
        </div>
        <h1>Administración de Usuarios</h1>
        
        <?php if (!empty($error)): ?>
        <div class="error-message">
            <?php echo $error; ?>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($success)): ?>
        <div class="success-message">
            <?php echo $success; ?>
        </div>
        <?php endif; ?>
        
        <div class="tab-container">
            <button class="tab-button <?php echo !isset($editUser) ? 'active' : ''; ?>" onclick="openTab('userList')">Lista de Usuarios</button>
            <button class="tab-button <?php echo isset($editUser) ? 'active' : ''; ?>" onclick="openTab('userForm')"><?php echo isset($editUser) ? 'Editar Usuario' : 'Nuevo Usuario'; ?></button>
        </div>
        
        <div id="userList" class="tab-content" <?php echo isset($editUser) ? 'style="display:none;"' : ''; ?>>
            <input type="text" id="searchUsers" class="search-box" placeholder="Buscar usuarios por nombre, email o rol...">
            
            <table class="users-table">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Usuario</th>
                        <th>Nombre</th>
                        <th>Email</th>
                        <th>Rol</th>
                        <th>Estado</th>
                        <th>Último Acceso</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($users as $user): ?>
                    <tr>
                        <td><?php echo $user['id']; ?></td>
                        <td><?php echo sanitizeInput($user['username']); ?></td>
                        <td><?php echo sanitizeInput($user['full_name']); ?></td>
                        <td><?php echo sanitizeInput($user['email']); ?></td>
                        <td><?php echo $user['role'] === 'admin' ? 'Administrador' : 'Médico'; ?></td>
                        <td class="<?php echo $user['is_active'] ? 'status-active' : 'status-inactive'; ?>">
                            <?php echo $user['is_active'] ? 'Activo' : 'Inactivo'; ?>
                        </td>
                        <td><?php echo $user['last_login'] ? date('d/m/Y H:i', strtotime($user['last_login'])) : 'Nunca'; ?></td>
                        <td>
                            <div class="action-buttons">
                                <a href="register.php?edit=<?php echo $user['id']; ?>" class="button-small edit-btn">Editar</a>
                                
                                <form method="post" style="display:inline;" onsubmit="return confirm('¿Estás seguro de que deseas cambiar el estado de este usuario?');">
                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                    <input type="hidden" name="is_active" value="<?php echo $user['is_active']; ?>">
                                    <button type="submit" name="toggle_active" class="button-small toggle-btn">
                                        <?php echo $user['is_active'] ? 'Desactivar' : 'Activar'; ?>
                                    </button>
                                </form>
                                
                                <?php if ($user['id'] != $_SESSION['user_id']): ?>
                                <form method="post" style="display:inline;" onsubmit="return confirm('¿Estás seguro de que deseas eliminar este usuario? Esta acción no se puede deshacer.');">
                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                    <button type="submit" name="delete_user" class="button-small delete-btn">Eliminar</button>
                                </form>
                                <?php endif; ?>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <div id="userForm" class="tab-content" <?php echo !isset($editUser) ? 'style="display:none;"' : ''; ?>>
            <h2><?php echo isset($editUser) ? 'Editar Usuario' : 'Nuevo Usuario'; ?></h2>
            
            <form method="post" class="register-form">
                <?php if (isset($editUser)): ?>
                <input type="hidden" name="user_id" value="<?php echo $editUser['id']; ?>">
                <?php endif; ?>
                
                <div class="form-group">
                    <label for="username">Usuario *</label>
                    <input type="text" id="username" name="username" value="<?php echo isset($editUser) ? $editUser['username'] : ''; ?>" required>
                </div>
                
                <?php if (isset($editUser)): ?>
                <div class="checkbox-container">
                    <input type="checkbox" id="update_password" name="update_password" value="1">
                    <label for="update_password">Actualizar contraseña</label>
                </div>
                
                <div id="password_fields" style="display:none;">
                <?php endif; ?>
                
                <div class="form-group">
                    <label for="password">Contraseña <?php echo isset($editUser) ? '' : '*'; ?></label>
                    <input type="password" id="password" name="password" <?php echo isset($editUser) ? '' : 'required'; ?>>
                </div>
                
                <div class="form-group">
                    <label for="confirm_password">Confirmar Contraseña <?php echo isset($editUser) ? '' : '*'; ?></label>
                    <input type="password" id="confirm_password" name="confirm_password" <?php echo isset($editUser) ? '' : 'required'; ?>>
                </div>
                
                <?php if (isset($editUser)): ?>
                </div>
                <?php endif; ?>
                
                <div class="form-group">
                    <label for="full_name">Nombre Completo *</label>
                    <input type="text" id="full_name" name="full_name" value="<?php echo isset($editUser) ? $editUser['full_name'] : ''; ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="email">Email</label>
                    <input type="email" id="email" name="email" value="<?php echo isset($editUser) ? $editUser['email'] : ''; ?>">
                </div>
                
                <div class="form-group">
                    <label for="role">Rol *</label>
                    <select id="role" name="role" required>
                        <option value="doctor" <?php echo (isset($editUser) && $editUser['role'] === 'doctor') ? 'selected' : ''; ?>>Médico</option>
                        <option value="admin" <?php echo (isset($editUser) && $editUser['role'] === 'admin') ? 'selected' : ''; ?>>Administrador</option>
                    </select>
                </div>
                
                <?php if (isset($editUser)): ?>
                <div class="checkbox-container">
                    <input type="checkbox" id="is_active" name="is_active" value="1" <?php echo ($editUser['is_active'] == 1) ? 'checked' : ''; ?>>
                    <label for="is_active">Usuario activo</label>
                </div>
                
                <div class="form-actions">
                    <button type="submit" name="update_user" class="register-btn">Actualizar Usuario</button>
                    <a href="register.php" class="back-btn">Cancelar</a>
                </div>
                <?php else: ?>
                <div class="form-actions">
                    <button type="submit" name="register" class="register-btn">Registrar Usuario</button>
                    <a href="studies.php" class="back-btn">Volver a Estudios</a>
                </div>
                <?php endif; ?>
            </form>
        </div>
        
        <?php else: ?>
        <div class="register-container">
            <div class="register-header">
                <img src="images/imac_logo.png" alt="IMAC Tucumán" class="register-logo">
                <h1>Registro de Usuario</h1>
            </div>
            
            <?php if (!empty($error)): ?>
            <div class="error-message">
                <?php echo $error; ?>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($success)): ?>
            <div class="success-message">
                <?php echo $success; ?>
            </div>
            <?php endif; ?>
            
            <form method="post" class="register-form">
                <div class="form-group">
                    <label for="username">Usuario *</label>
                    <input type="text" id="username" name="username" required>
                </div>
                
                <div class="form-group">
                    <label for="password">Contraseña *</label>
                    <input type="password" id="password" name="password" required>
                </div>
                
                <div class="form-group">
                    <label for="confirm_password">Confirmar Contraseña *</label>
                    <input type="password" id="confirm_password" name="confirm_password" required>
                </div>
                
                <div class="form-group">
                    <label for="full_name">Nombre Completo *</label>
                    <input type="text" id="full_name" name="full_name" required>
                </div>
                
                <div class="form-group">
                    <label for="email">Email</label>
                    <input type="email" id="email" name="email">
                </div>
                
                <div class="form-group">
                    <label for="role">Rol *</label>
                    <select id="role" name="role" required>
                        <option value="doctor">Médico</option>
                        <?php if ($isAdmin): ?>
                        <option value="admin">Administrador</option>
                        <?php endif; ?>
                    </select>
                </div>
                
                <div class="form-actions">
                    <button type="submit" name="register" class="register-btn">Registrar Usuario</button>
                    <a href="login.php" class="back-btn">Volver al Login</a>
                </div>
                
                <a href="login.php" class="login-btn">¿Ya tienes cuenta? Inicia sesión aquí</a>
                
                <!-- Área para administradores -->
                <div class="admin-section">
                    <p>¿Eres administrador del sistema?</p>
                    <a href="admin_login.php" class="admin-btn">Acceder a Administración</a>
                </div>
            </form>
        </div>
        <?php endif; ?>
    </div>

    <script>
    function openTab(tabName) {
        const tabs = document.querySelectorAll('.tab-content');
        tabs.forEach(tab => tab.style.display = 'none');
        document.getElementById(tabName).style.display = 'block';
        
        const buttons = document.querySelectorAll('.tab-button');
        buttons.forEach(button => button.classList.remove('active'));
        event.currentTarget.classList.add('active');
    }
    
    <?php if (isset($editUser)): ?>
    document.getElementById('update_password').addEventListener('change', function() {
        const passwordFields = document.getElementById('password_fields');
        passwordFields.style.display = this.checked ? 'block' : 'none';
        
        // Si se desmarca la casilla, limpiar los campos de contraseña
        if (!this.checked) {
            document.getElementById('password').value = '';
            document.getElementById('confirm_password').value = '';
        }
    });
    <?php endif; ?>
    
    // Función para filtrar la tabla de usuarios
    document.getElementById('searchUsers').addEventListener('keyup', function() {
        const searchText = this.value.toLowerCase();
        const rows = document.querySelectorAll('.users-table tbody tr');
        
        rows.forEach(row => {
            const username = row.cells[1].textContent.toLowerCase();
            const fullName = row.cells[2].textContent.toLowerCase();
            const email = row.cells[3].textContent.toLowerCase();
            const role = row.cells[4].textContent.toLowerCase();
            
            if (username.includes(searchText) || 
                fullName.includes(searchText) || 
                email.includes(searchText) || 
                role.includes(searchText)) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        });
    });
    </script>
</body>
</html> 