/**
 * Integración de Orthanc con el Portal de Estudios
 * Maneja la comunicación con la API de Orthanc para obtener estudios de pacientes
 */
class OrthancIntegration {
    constructor() {
        this.apiBaseUrl = 'api/';
    }
    
    /**
     * Obtiene los estudios de un paciente desde Orthanc
     * @param {string} patientId - ID del paciente
     * @returns {Promise<Array>} - Lista de estudios
     */
    async getPatientStudies(patientId) {
        try {
            const response = await fetch(`${this.apiBaseUrl}get_patient_studies.php?patient_id=${encodeURIComponent(patientId)}`);
            const result = await response.json();
            
            if (!result.success) {
                throw new Error(result.error);
            }
            
            return result;
        } catch (error) {
            console.error('Error obteniendo estudios del paciente:', error);
            throw error;
        }
    }
    
    /**
     * Formatea una fecha DICOM (YYYYMMDD) a formato legible
     * @param {string} dateStr - Fecha en formato DICOM
     * @returns {string} - Fecha formateada
     */
    formatDate(dateStr) {
        if (!dateStr || dateStr.length !== 8) return dateStr;
        
        const year = dateStr.substring(0, 4);
        const month = dateStr.substring(4, 6);
        const day = dateStr.substring(6, 8);
        
        return `${day}/${month}/${year}`;
    }
    
    /**
     * Formatea una hora DICOM (HHMMSS) a formato legible
     * @param {string} timeStr - Hora en formato DICOM
     * @returns {string} - Hora formateada
     */
    formatTime(timeStr) {
        if (!timeStr || timeStr.length < 4) return timeStr;
        
        const hours = timeStr.substring(0, 2);
        const minutes = timeStr.substring(2, 4);
        
        return `${hours}:${minutes}`;
    }
    
    /**
     * Genera el HTML para mostrar los estudios en el modal
     * @param {Array} studies - Lista de estudios
     * @returns {string} - HTML generado
     */
    generateStudiesHTML(studies) {
        if (!studies || studies.length === 0) {
            return `
                <tr>
                    <td colspan="4" class="text-center py-4">
                        <i class="fas fa-info-circle text-muted me-2"></i>
                        No se encontraron estudios para este paciente
                    </td>
                </tr>
            `;
        }
        
        return studies.map(study => {
            const formattedDate = this.formatDate(study.study_date);
            const formattedTime = this.formatTime(study.study_time);
            
            return `
                <tr>
                    <td>${formattedDate}</td>
                    <td>${study.modality}</td>
                    <td>${study.study_description}</td>
                    <td>
                        <button class="btn btn-sm btn-primary" onclick="window.open('${study.viewer_url}', '_blank')" title="Ver estudio">
                            <i class="fas fa-eye"></i> Ver
                        </button>
                        <button class="btn btn-sm btn-secondary ms-1" onclick="orthancIntegration.downloadStudy('${study.orthanc_id}')" title="Descargar">
                            <i class="fas fa-download"></i>
                        </button>
                    </td>
                </tr>
            `;
        }).join('');
    }
    
    /**
     * Actualiza el modal de estudios con los datos de Orthanc
     * @param {string} patientId - ID del paciente
     */
    async updateStudiesModal(patientId) {
        const modalBody = document.querySelector('#estudioModal .modal-body');
        const tableBody = document.querySelector('#estudioModal tbody');
        
        if (!tableBody) {
            console.error('No se encontró la tabla de estudios en el modal');
            return;
        }
        
        // Mostrar loading con progreso
        let loadingDots = 0;
        const loadingInterval = setInterval(() => {
            loadingDots = (loadingDots + 1) % 4;
            const dots = '.'.repeat(loadingDots);
            const loadingRow = tableBody.querySelector('.loading-row');
            if (loadingRow) {
                loadingRow.innerHTML = `
                    <td colspan="4" class="text-center py-4">
                        <div class="spinner-border text-primary" role="status">
                            <span class="visually-hidden">Cargando...</span>
                        </div>
                        <div class="mt-2">Consultando CLOUD PACS${dots}</div>
                        <small class="text-muted d-block mt-1">Esto puede tomar unos segundos...</small>
                    </td>
                `;
            }
        }, 500);
        
        tableBody.innerHTML = `
            <tr class="loading-row">
                <td colspan="4" class="text-center py-4">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Cargando...</span>
                    </div>
                    <div class="mt-2">Consultando CLOUD PACS...</div>
                    <small class="text-muted d-block mt-1">Esto puede tomar unos segundos...</small>
                </td>
            </tr>
        `;
        
        try {
            const response = await this.getPatientStudies(patientId);
            const studies = response.data || [];
            const patientName = response.patient_name || 'Paciente';
            
            // Limpiar el intervalo de loading
            clearInterval(loadingInterval);
            
            const studiesHTML = this.generateStudiesHTML(studies);
            tableBody.innerHTML = studiesHTML;
            
            // Actualizar título del modal
            const modalTitle = document.querySelector('#estudioModal .modal-title');
            if (modalTitle) {
                modalTitle.textContent = `Estudios de ${patientName} (${studies.length} encontrados)`;
            }
            
        } catch (error) {
            console.error('Error cargando estudios:', error);
            
            // Limpiar el intervalo de loading
            clearInterval(loadingInterval);
            
            tableBody.innerHTML = `
                <tr>
                    <td colspan="4" class="text-center py-4 text-danger">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        Error al cargar los estudios: ${error.message}
                        <div class="mt-2">
                            <small class="text-muted">Verifique la conexión con el servidor PACS</small>
                        </div>
                    </td>
                </tr>
            `;
        }
    }
    
    /**
     * Descarga un estudio (placeholder para implementación futura)
     * @param {string} studyId - ID del estudio en Orthanc
     */
    downloadStudy(studyId) {
        console.log('Descargando estudio:', studyId);
        // Implementar descarga de estudio
        alert('Función de descarga en desarrollo');
    }
}

// Instancia global para uso en el HTML
const orthancIntegration = new OrthancIntegration();