// Módulo de integración PACS DICOM
const DICOMModule = {
    // Configuración del PACS
    config: {
        pacsURL: 'https://tuc-i310100.tail186bcc.ts.net',
        viewerURL: 'https://tuc-i310100.tail186bcc.ts.net/?studyId=',
        apiEndpoint: '/api/studies'
    },

    // Estado de la conexión
    connectionState: {
        isConnected: false,
        lastError: null
    },

    // Inicializar conexión con PACS
    initConnection: async function() {
        // Deshabilitar conexión automática para evitar errores CORS en desarrollo
        if (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1') {
            console.log('Conexión PACS deshabilitada en entorno de desarrollo');
            this.connectionState.isConnected = false;
            this.connectionState.lastError = 'Conexión deshabilitada en desarrollo';
            return false;
        }
        
        try {
            const response = await fetch(this.config.pacsURL + '/api/echo');
            if (response.ok) {
                this.connectionState.isConnected = true;
                this.connectionState.lastError = null;
                return true;
            } else {
                throw new Error('Error en la conexión con PACS');
            }
        } catch (error) {
            this.connectionState.isConnected = false;
            this.connectionState.lastError = error.message;
            console.log('PACS no disponible:', error.message);
            return false;
        }
    },

    // Enviar informe al PACS
    sendReport: async function(studyId, reportContent) {
        try {
            if (!this.connectionState.isConnected) {
                await this.initConnection();
            }

            const response = await fetch(`${this.config.pacsURL}${this.config.apiEndpoint}/${studyId}/report`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    content: reportContent,
                    timestamp: new Date().toISOString(),
                    format: 'html'
                })
            });

            if (!response.ok) {
                throw new Error('Error al enviar informe al PACS');
            }

            return await response.json();
        } catch (error) {
            console.error('Error al enviar informe:', error);
            throw error;
        }
    },

    // Obtener visor DICOM para un estudio
    getViewer: function(studyId) {
        return `${this.config.viewerURL}${studyId}`;
    },

    // Obtener lista de estudios
    getStudies: async function(filters = {}) {
        try {
            if (!this.connectionState.isConnected) {
                await this.initConnection();
            }

            const queryParams = new URLSearchParams(filters).toString();
            const response = await fetch(`${this.config.pacsURL}${this.config.apiEndpoint}?${queryParams}`);

            if (!response.ok) {
                throw new Error('Error al obtener lista de estudios');
            }

            return await response.json();
        } catch (error) {
            console.error('Error al obtener estudios:', error);
            throw error;
        }
    },

    // Obtener detalles de un estudio específico
    getStudyDetails: async function(studyId) {
        try {
            if (!this.connectionState.isConnected) {
                await this.initConnection();
            }

            const response = await fetch(`${this.config.pacsURL}${this.config.apiEndpoint}/${studyId}`);

            if (!response.ok) {
                throw new Error('Error al obtener detalles del estudio');
            }

            return await response.json();
        } catch (error) {
            console.error('Error al obtener detalles del estudio:', error);
            throw error;
        }
    },

    // Verificar estado de conexión
    checkConnection: async function() {
        const status = await this.initConnection();
        return {
            connected: status,
            lastError: this.connectionState.lastError
        };
    }
};