/**
 * Integración de Orthanc con el Dashboard Profesional
 * Maneja la carga y visualización de estudios desde Orthanc
 */
class DashboardOrthanc {
    constructor() {
        this.apiBaseUrl = 'api/';
        this.studies = [];
        this.filteredStudies = [];
        this.currentFilters = {
            patient: '',
            date: '',
            patientId: '',
            modality: 'all'
        };
    }
    
    /**
     * Inicializa el dashboard
     */
    async init() {
        try {
            await this.loadStudies();
            this.setupEventListeners();
            this.renderStudies();
            this.updateStats();
            this.populatePatientFilter();
        } catch (error) {
            console.error('Error inicializando dashboard:', error);
            this.showError('Error consultando estudios desde CLOUD PACS');
        }
    }
    
    /**
     * Carga todos los estudios desde Orthanc
     */
    async loadStudies() {
        try {
            const response = await fetch(this.apiBaseUrl + 'get_all_studies.php');
            const result = await response.json();
            
            if (!result.success) {
                throw new Error(result.error);
            }
            
            this.studies = result.data;
            this.filteredStudies = [...this.studies];
            
            console.log('Estudios cargados:', this.studies.length);
        } catch (error) {
            console.error('Error cargando estudios:', error);
            throw error;
        }
    }
    
    /**
     * Configura los event listeners
     */
    setupEventListeners() {
        // Filtros
        const filterPatient = document.getElementById('filterPatient');
        if (filterPatient) {
            filterPatient.addEventListener('change', (e) => {
                this.currentFilters.patient = e.target.value;
                this.applyFilters();
            });
        }
        
        const filterDate = document.getElementById('filterDate');
        if (filterDate) {
            filterDate.addEventListener('change', (e) => {
                this.currentFilters.date = e.target.value;
                this.applyFilters();
            });
        }
        
        const patientIdInput = document.getElementById('patientId');
        if (patientIdInput) {
            patientIdInput.addEventListener('input', (e) => {
                this.currentFilters.patientId = e.target.value;
                this.applyFilters();
            });
        }
        
        // Botones de modalidad
        document.querySelectorAll('.modality-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                document.querySelectorAll('.modality-btn').forEach(b => b.classList.remove('active'));
                e.target.classList.add('active');
                this.currentFilters.modality = e.target.dataset.modality;
                this.applyFilters();
            });
        });
        
        // Botón de búsqueda - reemplazar la función existente
        const searchBtn = document.querySelector('button[onclick="searchStudies()"]');
        if (searchBtn) {
            searchBtn.removeAttribute('onclick');
            searchBtn.addEventListener('click', (e) => {
                e.preventDefault();
                this.applyFilters();
            });
        }
    }
    
    /**
     * Aplica los filtros a los estudios
     */
    applyFilters() {
        this.filteredStudies = this.studies.filter(study => {
            // Filtro por paciente
            if (this.currentFilters.patient && !study.patient_name.toLowerCase().includes(this.currentFilters.patient.toLowerCase())) {
                return false;
            }
            
            // Filtro por ID de paciente
            if (this.currentFilters.patientId && !study.patient_id.includes(this.currentFilters.patientId)) {
                return false;
            }
            
            // Filtro por fecha
            if (this.currentFilters.date) {
                const studyDate = new Date(this.formatDate(study.date));
                const today = new Date();
                
                switch (this.currentFilters.date) {
                    case 'today':
                        if (studyDate.toDateString() !== today.toDateString()) return false;
                        break;
                    case 'week':
                        const weekAgo = new Date(today.getTime() - 7 * 24 * 60 * 60 * 1000);
                        if (studyDate < weekAgo) return false;
                        break;
                    case 'month':
                        const monthAgo = new Date(today.getFullYear(), today.getMonth() - 1, today.getDate());
                        if (studyDate < monthAgo) return false;
                        break;
                }
            }
            
            // Filtro por modalidad
            if (this.currentFilters.modality !== 'all') {
                // Mapeo de modalidades de interfaz a códigos DICOM
                const modalityMap = {
                    'RX': ['CR', 'DX', 'RF'],
                    'TC': ['CT'],
                    'RM': ['MR'],
                    'US': ['US'],
                    'MG': ['MG'],
                    'DX': ['DX']
                };
                
                const dicomCodes = modalityMap[this.currentFilters.modality] || [this.currentFilters.modality];
                const hasMatchingModality = dicomCodes.some(code => study.modality.includes(code));
                
                if (!hasMatchingModality) {
                    return false;
                }
            }
            
            return true;
        });
        
        this.renderStudies();
        this.updateStats();
    }
    
    /**
     * Renderiza los estudios en la tabla
     */
    renderStudies() {
        const tbody = document.querySelector('.studies-table tbody');
        if (!tbody) return;
        
        tbody.innerHTML = '';
        
        this.filteredStudies.forEach(study => {
            const row = this.createStudyRow(study);
            tbody.appendChild(row);
        });
    }
    
    /**
     * Crea una fila de la tabla para un estudio
     */
    createStudyRow(study) {
        const row = document.createElement('tr');
        
        const formattedDate = this.formatDate(study.date);
        const formattedTime = this.formatTime(study.time);
        const modalityBadge = this.getModalityBadge(study.modality);
        
        row.innerHTML = `
            <td>${formattedDate}</td>
            <td class="d-none d-md-table-cell">${formattedTime}</td>
            <td>
                <div class="fw-semibold">${study.patient_name}</div>
                <small class="text-muted d-block d-lg-none">ID: ${study.patient_id}</small>
            </td>
            <td class="d-none d-lg-table-cell">${study.patient_id}</td>
            <td class="d-none d-sm-table-cell">${modalityBadge}</td>
            <td class="d-none d-xl-table-cell">${study.study_description}</td>
            <td class="d-none d-lg-table-cell">
                <span class="badge status-badge status-completed">${study.status}</span>
            </td>
            <td>
                <div class="d-flex gap-1">
                    <button class="btn btn-sm btn-action btn-view" title="Ver" onclick="window.open('${study.viewer_url}', '_blank')">
                        <i class="fas fa-eye"></i>
                        <span class="d-none d-sm-inline ms-1">Ver</span>
                    </button>
                    <button class="btn btn-sm btn-action btn-download" title="Descargar" onclick="dashboardOrthanc.downloadStudy('${study.orthanc_study_id}')">
                        <i class="fas fa-download"></i>
                        <span class="d-none d-md-inline ms-1">Descargar</span>
                    </button>
                    <button class="btn btn-sm btn-action btn-report" title="Generar Informe" onclick="generateReport('${study.patient_id}', '${study.patient_name}', '${study.modality}', '${study.study_description}')">
                        <i class="fas fa-file-medical"></i>
                        <span class="d-none d-lg-inline ms-1">Informe</span>
                    </button>
                </div>
            </td>
        `;
        
        return row;
    }
    
    /**
     * Actualiza las estadísticas del dashboard
     */
    updateStats() {
        const today = new Date().toISOString().split('T')[0].replace(/-/g, '');
        const todayStudies = this.studies.filter(study => study.date === today).length;
        
        const thisMonth = new Date();
        const monthStart = new Date(thisMonth.getFullYear(), thisMonth.getMonth(), 1).toISOString().split('T')[0].replace(/-/g, '');
        const monthStudies = this.studies.filter(study => study.date >= monthStart).length;
        
        // Actualizar estadísticas en el DOM
        const statNumbers = document.querySelectorAll('.stat-number');
        const badgeElement = document.querySelector('.badge.bg-primary');
        
        if (statNumbers[0]) statNumbers[0].textContent = todayStudies;
        if (statNumbers[1]) statNumbers[1].textContent = monthStudies;
        if (badgeElement) badgeElement.textContent = `${this.filteredStudies.length} estudios`;
    }
    
    /**
     * Popula el filtro de pacientes con los nombres únicos
     */
    populatePatientFilter() {
        const filterPatient = document.getElementById('filterPatient');
        if (!filterPatient) return;
        
        // Obtener nombres únicos de pacientes
        const uniquePatients = [...new Set(this.studies.map(study => study.patient_name))]
            .filter(name => name && name.trim() !== '')
            .sort();
        
        // Limpiar opciones existentes (excepto la primera)
        while (filterPatient.children.length > 1) {
            filterPatient.removeChild(filterPatient.lastChild);
        }
        
        // Agregar opciones de pacientes
        uniquePatients.forEach(patientName => {
            const option = document.createElement('option');
            option.value = patientName;
            option.textContent = patientName;
            filterPatient.appendChild(option);
        });
    }
    
    /**
     * Formatea una fecha DICOM (YYYYMMDD) a formato legible
     */
    formatDate(dateStr) {
        if (!dateStr || dateStr.length !== 8) return dateStr;
        
        const year = dateStr.substring(0, 4);
        const month = dateStr.substring(4, 6);
        const day = dateStr.substring(6, 8);
        
        return `${day}/${month}/${year}`;
    }
    
    /**
     * Formatea una hora DICOM (HHMMSS) a formato legible
     */
    formatTime(timeStr) {
        if (!timeStr || timeStr.length < 4) return timeStr;
        
        const hours = timeStr.substring(0, 2);
        const minutes = timeStr.substring(2, 4);
        
        return `${hours}:${minutes}`;
    }
    
    /**
     * Obtiene el badge HTML para una modalidad
     */
    getModalityBadge(modality) {
        const modalityClass = modality.toLowerCase().replace(/[^a-z]/g, '');
        return `<span class="badge modality-badge modality-${modalityClass}">${modality}</span>`;
    }
    
    /**
     * Descarga un estudio (placeholder)
     */
    downloadStudy(studyId) {
        // Implementar descarga de estudio
        console.log('Descargando estudio:', studyId);
        alert('Función de descarga en desarrollo');
    }
    
    /**
     * Muestra un error en el dashboard
     */
    showError(message) {
        const container = document.querySelector('.main-content .container-fluid');
        if (container) {
            container.innerHTML = `
                <div class="alert alert-danger" role="alert">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    ${message}
                    <br><br>
                    <small>Verifique que el servidor Orthanc esté ejecutándose y la configuración sea correcta.</small>
                </div>
            `;
        }
    }
}

// Instancia global
let dashboardOrthanc;

// Inicializar cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', function() {
    // Solo inicializar si estamos en el dashboard
    if (window.location.pathname.includes('dashboard-unified.html')) {
        dashboardOrthanc = new DashboardOrthanc();
        dashboardOrthanc.init();
    }
});

// Función global para búsqueda (mantener compatibilidad)
function searchStudies() {
    if (dashboardOrthanc) {
        dashboardOrthanc.applyFilters();
    }
}