// Módulo de grabación de audio
const AudioModule = {
    // Estado de la grabación
    recordingState: {
        mediaRecorder: null,
        audioChunks: [],
        isRecording: false,
        stream: null
    },

    // Inicializar el grabador
    initRecorder: function() {
        // El stream será proporcionado por el coordinador cuando sea necesario
        console.log('Módulo de audio inicializado - usando stream compartido');
        this.updateUI();
    },

    // Configurar MediaRecorder con el stream compartido
    setupMediaRecorder: function(stream) {
        try {
            this.recordingState.stream = stream;
            this.recordingState.mediaRecorder = new MediaRecorder(stream);

            this.recordingState.mediaRecorder.ondataavailable = (event) => {
                this.recordingState.audioChunks.push(event.data);
            };

            this.recordingState.mediaRecorder.onstop = () => {
                const audioBlob = new Blob(this.recordingState.audioChunks, { type: 'audio/wav' });
                const audioUrl = URL.createObjectURL(audioBlob);
                this.displayRecording(audioUrl);
                this.recordingState.audioChunks = [];
            };

            return true;
        } catch (error) {
            console.error('Error al configurar MediaRecorder:', error);
            return false;
        }
    },

    // Iniciar grabación
    startRecording: function() {
        // Activar grabación a través del coordinador
        MicrophoneCoordinator.enableRecording((success) => {
            if (success) {
                const stream = MicrophoneCoordinator.getRecordingStream();
                if (stream && this.setupMediaRecorder(stream)) {
                    this.recordingState.mediaRecorder.start();
                    this.recordingState.isRecording = true;
                    this.updateUI();
                    this.startTimer();
                    console.log('Grabación iniciada con stream compartido');
                } else {
                    console.error('No se pudo configurar la grabación');
                    MicrophoneCoordinator.disableRecording();
                }
            }
        });
    },

    // Detener grabación
    stopRecording: function() {
        if (this.recordingState.mediaRecorder && this.recordingState.isRecording) {
            this.recordingState.mediaRecorder.stop();
            this.recordingState.isRecording = false;
            // Desactivar grabación en el coordinador
            MicrophoneCoordinator.disableRecording();
            this.updateUI();
            this.stopTimer();
            console.log('Grabación detenida');
        }
    },

    // Mostrar grabación
    displayRecording: function(audioUrl) {
        const audioPlayer = document.getElementById('audioPlayer');
        const audioContainer = document.getElementById('audioContainer');
        
        audioPlayer.src = audioUrl;
        audioContainer.style.display = 'block';
    },

    // Actualizar interfaz
    updateUI: function() {
        const recordButton = document.getElementById('recordButton');
        const stopButton = document.getElementById('stopButton');
        
        if (this.recordingState.isRecording) {
            recordButton.disabled = true;
            stopButton.disabled = false;
            recordButton.textContent = 'Grabando...';
        } else {
            recordButton.disabled = false;
            stopButton.disabled = true;
            recordButton.textContent = 'Iniciar Grabación';
        }
    },

    // Timer para la grabación
    timer: {
        interval: null,
        seconds: 0,
        display: document.getElementById('timer')
    },

    // Iniciar timer
    startTimer: function() {
        this.timer.seconds = 0;
        this.timer.interval = setInterval(() => {
            this.timer.seconds++;
            const minutes = Math.floor(this.timer.seconds / 60);
            const seconds = this.timer.seconds % 60;
            this.timer.display.textContent = 
                `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
        }, 1000);
    },

    // Detener timer
    stopTimer: function() {
        if (this.timer.interval) {
            clearInterval(this.timer.interval);
            this.timer.interval = null;
        }
    },

    // Limpiar recursos
    cleanup: function() {
        if (this.recordingState.stream) {
            this.recordingState.stream.getTracks().forEach(track => track.stop());
        }
        this.stopTimer();
    }
};