<?php
require_once 'config/orthanc_config.php';

/**
 * Cliente para interactuar con la API REST de Orthanc
 * Maneja todas las operaciones de consulta y recuperación de datos
 */
class OrthancClient {
    private $baseUrl;
    private $credentials;
    private $timeout;
    
    public function __construct() {
        $this->baseUrl = OrthancConfig::getServerUrl();
        $this->credentials = OrthancConfig::getCredentials();
        $this->timeout = OrthancConfig::getConfig()['api']['timeout'];
    }
    
    /**
     * Realiza una petición HTTP a la API de Orthanc
     */
    private function makeRequest($endpoint, $method = 'GET', $data = null) {
        $url = $this->baseUrl . $endpoint;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
        curl_setopt($ch, CURLOPT_USERPWD, $this->credentials['username'] . ':' . $this->credentials['password']);
        curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, OrthancConfig::getConfig()['api']['connect_timeout']);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, OrthancConfig::getConfig()['api']['verify_ssl']);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
        
        if ($method === 'POST' && $data) {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            throw new Exception('Error de conexión: ' . $error);
        }
        
        if ($httpCode !== 200) {
            throw new Exception('Error HTTP: ' . $httpCode);
        }
        
        return json_decode($response, true);
    }
    
    /**
     * Busca estudios por ID de paciente usando /tools/find (optimizado)
     */
    public function findStudiesByPatientId($patientId) {
        $query = [
            'Level' => 'Study',
            'Query' => [
                'PatientID' => $patientId
            ],
            'Expand' => true  // Obtener detalles expandidos en una sola consulta
        ];
        
        $results = $this->makeRequest('/tools/find', 'POST', $query);
        
        $studies = [];
        $patientCache = [];  // Cache para evitar consultas repetidas de paciente
        
        foreach ($results as $study) {
            try {
                // Si expand=true funciona, $study ya contiene los detalles
                if (is_array($study) && isset($study['MainDicomTags'])) {
                    $studyData = $study;
                } else {
                    // Fallback: consulta individual si expand no funciona
                    $studyData = $this->makeRequest('/studies/' . $study);
                }
                
                $patientId = $studyData['ParentPatient'];
                
                // Usar cache para datos del paciente
                if (!isset($patientCache[$patientId])) {
                    $patientCache[$patientId] = $this->makeRequest('/patients/' . $patientId);
                }
                $patient = $patientCache[$patientId];
                
                $studyId = is_string($study) ? $study : $studyData['ID'];
                
                $studies[] = [
                    'study_id' => $studyId,
                    'orthanc_id' => $studyId,
                    'patient_id' => $patient['MainDicomTags']['PatientID'] ?? '',
                    'patient_name' => $patient['MainDicomTags']['PatientName'] ?? '',
                    'patient_birth_date' => $patient['MainDicomTags']['PatientBirthDate'] ?? '',
                    'study_date' => $studyData['MainDicomTags']['StudyDate'] ?? '',
                    'study_time' => $studyData['MainDicomTags']['StudyTime'] ?? '',
                    'study_description' => $studyData['MainDicomTags']['StudyDescription'] ?? '',
                    'modality' => $this->getStudyModalitiesOptimized($studyData),
                    'accession_number' => $studyData['MainDicomTags']['AccessionNumber'] ?? '',
                    'referring_physician' => $studyData['MainDicomTags']['ReferringPhysicianName'] ?? '',
                    'series_count' => count($studyData['Series'] ?? []),
                    'viewer_url' => OrthancConfig::getViewerUrl($studyId)
                ];
            } catch (Exception $e) {
                error_log('Error procesando estudio: ' . $e->getMessage());
                continue;
            }
        }
        
        return $studies;
    }
    
    /**
     * Obtiene detalles de un estudio específico
     */
    public function getStudyDetails($studyId) {
        try {
            $study = $this->makeRequest('/studies/' . $studyId);
            $patient = $this->makeRequest('/patients/' . $study['ParentPatient']);
            
            return [
                'study_id' => $studyId,
                'orthanc_id' => $studyId,
                'patient_id' => $patient['MainDicomTags']['PatientID'] ?? '',
                'patient_name' => $patient['MainDicomTags']['PatientName'] ?? '',
                'patient_birth_date' => $patient['MainDicomTags']['PatientBirthDate'] ?? '',
                'study_date' => $study['MainDicomTags']['StudyDate'] ?? '',
                'study_time' => $study['MainDicomTags']['StudyTime'] ?? '',
                'study_description' => $study['MainDicomTags']['StudyDescription'] ?? '',
                'modality' => $this->getStudyModalities($studyId),
                'accession_number' => $study['MainDicomTags']['AccessionNumber'] ?? '',
                'referring_physician' => $study['MainDicomTags']['ReferringPhysicianName'] ?? '',
                'series_count' => count($study['Series'] ?? []),
                'viewer_url' => OrthancConfig::getViewerUrl($studyId)
            ];
        } catch (Exception $e) {
            error_log('Error obteniendo detalles del estudio: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Obtiene las modalidades de un estudio (optimizado)
     */
    private function getStudyModalitiesOptimized($studyData) {
        try {
            $modalities = [];
            
            // Si tenemos las series en los datos del estudio
            if (isset($studyData['Series']) && is_array($studyData['Series'])) {
                // Intentar obtener modalidades de las primeras series (máximo 3 para evitar lentitud)
                $seriesToCheck = array_slice($studyData['Series'], 0, 3);
                
                foreach ($seriesToCheck as $seriesId) {
                    try {
                        $series = $this->makeRequest('/series/' . $seriesId);
                        $modality = $series['MainDicomTags']['Modality'] ?? '';
                        if ($modality && !in_array($modality, $modalities)) {
                            $modalities[] = $modality;
                        }
                    } catch (Exception $e) {
                        // Continuar con la siguiente serie si hay error
                        continue;
                    }
                }
            }
            
            return !empty($modalities) ? implode(', ', $modalities) : 'N/A';
        } catch (Exception $e) {
            error_log('Error obteniendo modalidades: ' . $e->getMessage());
            return 'N/A';
        }
    }
    
    /**
     * Obtiene las modalidades de un estudio (método original)
     */
    private function getStudyModalities($studyId) {
        try {
            $study = $this->makeRequest('/studies/' . $studyId);
            $modalities = [];
            
            foreach ($study['Series'] as $seriesId) {
                $series = $this->makeRequest('/series/' . $seriesId);
                $modality = $series['MainDicomTags']['Modality'] ?? '';
                if ($modality && !in_array($modality, $modalities)) {
                    $modalities[] = $modality;
                }
            }
            
            return implode(', ', $modalities);
        } catch (Exception $e) {
            return '';
        }
    }
    
    /**
     * Obtiene todos los pacientes del servidor Orthanc
     */
    public function getAllPatients() {
        try {
            $patientIds = $this->makeRequest('/patients');
            $patients = [];
            
            foreach ($patientIds as $patientId) {
                $patient = $this->makeRequest('/patients/' . $patientId);
                $studies = $this->getPatientStudies($patientId);
                
                $patients[] = [
                    'orthanc_id' => $patientId,
                    'patient_id' => $patient['MainDicomTags']['PatientID'] ?? '',
                    'patient_name' => $patient['MainDicomTags']['PatientName'] ?? '',
                    'patient_birth_date' => $patient['MainDicomTags']['PatientBirthDate'] ?? '',
                    'patient_sex' => $patient['MainDicomTags']['PatientSex'] ?? '',
                    'studies' => $studies,
                    'studies_count' => count($studies)
                ];
            }
            
            return $patients;
        } catch (Exception $e) {
            error_log('Error obteniendo pacientes: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Obtiene los estudios de un paciente específico
     */
    public function getPatientStudies($orthancPatientId) {
        try {
            $patient = $this->makeRequest('/patients/' . $orthancPatientId);
            $studies = [];
            
            foreach ($patient['Studies'] as $studyId) {
                $studyDetails = $this->getStudyDetails($studyId);
                if ($studyDetails) {
                    $studies[] = $studyDetails;
                }
            }
            
            return $studies;
        } catch (Exception $e) {
            error_log('Error obteniendo estudios del paciente: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Verifica el estado del servidor Orthanc
     */
    public function getServerStatus() {
        try {
            $system = $this->makeRequest('/system');
            return [
                'status' => 'connected',
                'version' => $system['Version'] ?? 'Unknown',
                'name' => $system['Name'] ?? 'Orthanc',
                'database_version' => $system['DatabaseVersion'] ?? 'Unknown'
            ];
        } catch (Exception $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }
}
?>